#'Calculates the share value from total Equity Value (that is present value of given amount of future FCFE) divided by number of outstanding shares.
#'@description
#'The value of equity can also be found by discounting FCFE at the required rate of return on equity (r). Free cash flow to equity is the cash flow available to holders of common equity after all operating expenses, interest, and principal payments have been paid and necessary investments in working and fixed capital have been made. FCFE is the cash flow from operations minus capital expenditures minus payments to (and plus receipts from) debt holders. An estimate of the value of equity is then found by subtracting the value of debt from the estimated value of the firm. Because FCFE is the cash flow remaining for equity holders after all other claims have been satisfied, discounting FCFE by r (the required rate of return on equity) gives the value of equity of the firm. Dividing the total value of equity by the number of outstanding shares gives the value per share (Jerald E. Pinto, 2020).
#'@details
#'According to information provided by Jerald E. Pinto (2020), the method \code{shareValueUsingDiscFCFE} is developed to compute the share value from total equity value (that is present value of given amount of future FCFE)  that is divided by number of outstanding shares. Here, \code{FCFE} is the given amount of future Free Cash Flow to the Firm (FCFF) in millions of dollars. For example, a value of 0.32 means 0.32 millions or 320,000 dollars, \code{t} is a vector of number of years ranging from 1 to any specified number of years for which FCFF is to be discounted, \code{r} is the required rate of return on equity, and \code{shares} is number of shares. Value of shares at 0.5 represent half a million shares (that means 500,000 shares). Values used for FCFF, and Market Value of Debt  are in millions of dollars. Value of shares at 0.5 represent half a million shares (that means 500,000 shares). An output of 1.68 means one share is valued at 1.68 dollars.
#'@param FCFE A vector.
#'@param t A vector.
#'@param r A number.
#'@param shares A number.
#'@return Input values to four arguments  \code{FCFE}, \code{expSharePriceIn1yr}, \code{r} and  \code{shares}.
#'@author MaheshP Kumar, \email{maheshparamjitkumar@@gmail.com}
#'@references
#'Pinto, J. E. (2020). Equity Asset Valuation (4th ed.). Wiley Professional Development (P&T). https://bookshelf.vitalsource.com/books/9781119628194
#'@examples
#'shareValueUsingDiscFCFE(FCFE=c(0.32,0.34,0.36),t=c(1,2,3),r=0.10,shares=0.5)
#'@export
shareValueUsingDiscFCFE<-function (FCFE,t,r,shares)
{
     equityVal <- sum(FCFE/(1 + r)^t)
     shareValu <-equityVal/shares
     (shareValu = round(shareValu, digits=2))
}
