% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metrics.R
\name{metrics}
\alias{metrics}
\alias{counter_metric}
\alias{gauge_metric}
\alias{histogram_metric}
\title{Metrics}
\usage{
counter_metric(
  name,
  help,
  labels = character(),
  ...,
  unit = NULL,
  registry = global_registry()
)

gauge_metric(
  name,
  help,
  labels = character(),
  ...,
  unit = NULL,
  registry = global_registry()
)

histogram_metric(
  name,
  help,
  buckets = c(0.005, 0.01, 0.025, 0.05, 0.1, 0.25, 0.5, 1, 2.5, 5, 10),
  labels = character(),
  ...,
  unit = NULL,
  registry = global_registry()
)
}
\arguments{
\item{name}{The name of the metric.}

\item{help}{A brief, one-sentence explanation of the metric's meaning.}

\item{labels}{A vector of label names for the metric.}

\item{...}{For backward compatibility, otherwise ignored.}

\item{unit}{An optional unit for the metric, e.g. \code{"seconds"}. Must
match the metric name.}

\item{registry}{Where to register the metric for later retrieval.}

\item{buckets}{A sequence of buckets to bin observations into. Defaults to
Prometheus's suggested buckets, which are a good fit for measuring
user-visible latency in seconds (e.g. for web services).}
}
\value{
An object with methods to manipulate the metric. See details.
}
\description{
A metric is a measure which can be aggregated into a time series, and comes
in one of three types: counters, gauges, and histograms.

Metrics must have a unique name.
}
\details{
All metric objects have a \code{reset()} method that reverts the underlying value
(or values) to zero, an \code{unregister()} method that removes them from the
registry they were created in, and a \code{render()} method that writes a
representation of the metric in the text-based \href{https://prometheus.io/docs/instrumenting/exposition_formats/#text-based-format}{OpenMetrics format}.
Normally, \code{\link[=render_metrics]{render_metrics()}} is used instead.

In addition, various metrics have their own methods:
\itemize{
\item \code{inc(by = 1, ...)}: Increments the metric by some positive number,
defaulting to 1. Further parameters are interpreted as labels. Available
for counters and gauges.
\item \code{dec(by = 1, ...)}: Decrements the metric by some number, defaulting to 1.
Further parameters are interpreted as labels. Available for gauges.
\item \code{set(value, ...)}: Sets the metric to some number. Further parameters are
interpreted as labels. Available for gauges.
\item \code{set_to_current_time(...)}: Sets the metric to the current time, in seconds
from the Unix epoch. Further parameters are interpreted as labels.
Available for gauges.
\item \code{observe(value, ...)}: Records an observation of some number. Further
parameters are interpreted as labels. Available for histograms.
\item \code{time(expr, ...)}: Records an observation for the time elapsed evaluating
\code{expr}, in seconds. Further parameters are interpreted as labels.
Available for histograms.
}
}
\examples{
meows <- counter_metric("meows", "Heard around the house.", labels = "cat")
meows$inc(cat = "Shamus") # Count one meow from Shamus.
meows$inc(3, cat = "Unknown") # Count three meows of unknown origin.
meows$render()

thermostat <- gauge_metric("thermostat", "Thermostat display.")
thermostat$set(21.3) # Read from the display...
thermostat$dec(2) # ... and then turn it down 2 degrees.
thermostat$render()

temperature <- histogram_metric(
  "temperature", "Ambient room temperature measurements.",
  buckets = c(10, 15, 20, 22, 25), labels = "room"
)
set.seed(9090)
# Simulate taking ambient temperature samples.
for (measure in rnorm(20, mean = 21.5)) {
  temperature$observe(measure, room = sample(c("kitchen", "bathroom"), 1))
}
temperature$render()

}
\seealso{
The official documentation on \href{https://prometheus.io/docs/concepts/metric_types/}{Metric Types}.
}
