% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sdu.R
\name{sd_overall}
\alias{sd_overall}
\title{Compute MCTQ overall sleep duration (only for MCTQ\eqn{^{Shift}}{ Shift})}
\usage{
sd_overall(sd_w, sd_f, n_w, n_f)
}
\arguments{
\item{sd_w}{A \code{\link[lubridate:duration]{Duration}} object corresponding to the
\strong{sleep duration between two days in a particular shift} from a shift
version of the MCTQ questionnaire. You can use \code{\link[=sdu]{sdu()}} to compute it.}

\item{sd_f}{A \code{\link[lubridate:duration]{Duration}} object corresponding to the
\strong{sleep duration between two free days after a particular shift} from a
shift version of the MCTQ questionnaire. You can use \code{\link[=sdu]{sdu()}}
to compute it.}

\item{n_w}{An \link[checkmate:checkIntegerish]{integerish}
\code{\link[base:numeric]{numeric}} object or an \code{\link[base:integer]{integer}}
object corresponding to the \strong{number of days worked in a particular shift
within a shift cycle} from a shift version of the MCTQ questionnaire.}

\item{n_f}{An \link[checkmate:checkIntegerish]{integerish}
\code{\link[base:numeric]{numeric}} object or an \code{\link[base:integer]{integer}}
object corresponding to the \strong{number of free days after a particular shift
within a shift cycle} from a shift version of the MCTQ questionnaire.}
}
\value{
A \code{\link[lubridate:duration]{Duration}} object corresponding to the
vectorized weighted mean of \code{sd_w} and \code{sd_f} with \code{n_w} and \code{n_f} as
weights.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{sd_overall()} computes the \strong{overall sleep duration in a particular shift}
for the shift version of the Munich ChronoType Questionnaire (MCTQ).

See \code{\link[=sd_week]{sd_week()}} to compute the average weekly sleep
duration for the standard and micro versions of the MCTQ.
}
\details{
\strong{Standard MCTQ} functions were created following the guidelines in
Roenneberg, Wirz-Justice, & Merrow (2003), Roenneberg, Allebrandt, Merrow, &
Vetter (2012), and from The Worldwide Experimental Platform (theWeP, n.d.).

\strong{\eqn{\mu}MCTQ} functions were created following the guidelines in Ghotbi
et al. (2020), in addition to the guidelines used for the standard MCTQ.

\strong{MCTQ\eqn{^{Shift}}{ Shift}} functions were created following the
guidelines in Juda, Vetter, & Roenneberg (2013), in addition to the
guidelines used for the standard MCTQ.

See the References section to learn more.
\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the
\link[lubridate:lubridate-package]{lubridate} and \link[hms:hms-package]{hms}
packages. Please refer to those package documentations to learn more about
them.
}

\subsection{Rounding and fractional time}{

Some operations may produce an output with fractional time (e.g.,
\code{"19538.3828571429s (~5.43 hours)"}, \code{01:15:44.505}). If you want, you
can round it with \code{\link[=round_time]{round_time()}}.

Our recommendation is to avoid rounding, but, if you do, make sure that you
only round your values after all computations are done. That way you avoid
\href{https://en.wikipedia.org/wiki/Round-off_error}{round-off errors}.
}
}
\section{Guidelines}{


Juda, Vetter, & Roenneberg (2013) and The Worldwide Experimental Platform
(n.d.) guidelines for \code{sd_overall()} (\eqn{\emptyset SD}{OSD}) computation
are as follows.
\subsection{Notes}{
\itemize{
\item This computation must be applied to each section of the questionnaire. If
you're using the three-shift design proposed by the MCTQ authors, you need to
compute three overall sleep duration (e.g., \eqn{\emptyset SD^M}{OSD_M};
\eqn{\emptyset SD^E}{OSD_E}; \eqn{\emptyset SD^N}{OSD_N}).
\item The overall sleep duration is the weighted average of the shift-specific
mean sleep durations.
\item If you are visualizing this documentation in plain text, you may have some
trouble understanding the equations. You can see this documentation on the
package \href{https://docs.ropensci.org/mctq/reference/}{website}.
}
}

\subsection{Computation}{

\deqn{\emptyset SD^{M/E/N} = \frac{(SD_{W}^{M/E/N} \times n_{W}^{M/E/N}) +
(SD_{F}^{M/E/N} \times n_{F}^{M/E/N})}{n_W^{M/E/N} + n_{F}^{M/E/N}}}

Where:
\itemize{
\item \eqn{\emptyset SD^{M/E/N}} = Overall sleep duration in a particular shift.
\item \eqn{SD_W^{M/E/N}} = Sleep duration between two days in a particular shift.
\item \eqn{SD_F^{M/E/N}} = Sleep duration between two free days after a
particular shift.
\item \eqn{n_W^{M/E/N}} = Number of days worked in a particular shift within a
shift cycle.
\item \eqn{n_F^{M/E/N}} = Number of free days after a particular shift within a
shift cycle.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days, \eqn{M} =
Morning shift; \eqn{E} = Evening shift; \eqn{N} = Night shift.
}
}

\examples{
## Scalar example

sd_w <- lubridate::dhours(5)
sd_f <- lubridate::dhours(9)
n_w <- 2
n_f <- 2
sd_overall(sd_w, sd_f, n_w, n_f)
#> [1] "25200s (~7 hours)" # Expected

sd_w <- lubridate::dhours(3.45)
sd_f <- lubridate::dhours(10)
n_w <- 3
n_f <- 1
sd_overall(sd_w, sd_f, n_w, n_f)
#> [1] "18315s (~5.09 hours)" # Expected

sd_w <- lubridate::as.duration(NA)
sd_f <- lubridate::dhours(12)
n_w <- 4
n_f <- 4
sd_overall(sd_w, sd_f, n_w, n_f)
#> [1] NA # Expected

## Vector example

sd_w <- c(lubridate::dhours(4), lubridate::dhours(7))
sd_f <- c(lubridate::dhours(12), lubridate::dhours(9))
n_w <- c(3, 4)
n_f <- c(2, 4)
sd_overall(sd_w, sd_f, n_w, n_f)
#> [1] "25920s (~7.2 hours)" "28800s (~8 hours)"  # Expected

## Checking second output from vector example

if (requireNamespace("stats", quietly = TRUE)) {
    i <- 2
    x <- c(sd_w[i], sd_f[i])
    w <- c(n_w[i], n_f[i])
    lubridate::as.duration(stats::weighted.mean(x, w))
}
#> [1] "28800s (~8 hours)" # Expected

## Converting the output to 'hms'

sd_w <- lubridate::dhours(4.75)
sd_f <- lubridate::dhours(10)
n_w <- 5
n_f <- 2
sd_overall(sd_w, sd_f, n_w, n_f)
#> [1] "22500s (~6.25 hours)" # Expected

hms::as_hms(as.numeric(sd_overall(sd_w, sd_f, n_w, n_f)))
#> 06:15:00 # Expected

## Rounding the output at the seconds level

sd_w <- lubridate::dhours(5.9874)
sd_f <- lubridate::dhours(9.3)
n_w <- 3
n_f <- 2
sd_overall(sd_w, sd_f, n_w, n_f)
#> [1] "26324.784s (~7.31 hours)" # Expected

round_time(sd_overall(sd_w, sd_f, n_w, n_f))
#> [1] "26325s (~7.31 hours)" # Expected
}
\references{
Ghotbi, N., Pilz, L. K., Winnebeck, E. C., Vetter, C., Zerbini, G., Lenssen,
D., Frighetto, G., Salamanca, M., Costa, R., Montagnese, S., & Roenneberg, T.
(2020). The \eqn{\mu}MCTQ: an ultra-short version of the Munich ChronoType
Questionnaire. \emph{Journal of Biological Rhythms}, \emph{35}(1), 98-110.
\doi{10.1177/0748730419886986}

Juda, M., Vetter, C., & Roenneberg, T. (2013). The Munich ChronoType
Questionnaire for shift-workers (MCTQ\eqn{^{Shift}}{ Shift}). \emph{Journal of
Biological Rhythms}, \emph{28}(2), 130-140. \doi{10.1177/0748730412475041}

Roenneberg T., Allebrandt K. V., Merrow M., & Vetter C. (2012). Social jetlag
and obesity. \emph{Current Biology}, \emph{22}(10), 939-43.
\doi{10.1016/j.cub.2012.03.038}

Roenneberg, T., Wirz-Justice, A., & Merrow, M. (2003). Life between clocks:
daily temporal patterns of human chronotypes. \emph{Journal of Biological
Rhythms}, \emph{18}(1), 80-90. \doi{10.1177/0748730402239679}

The Worldwide Experimental Platform (n.d.). MCTQ.
\url{https://www.thewep.org/documentations/mctq/}
}
\seealso{
Other MCTQ functions: 
\code{\link{fd}()},
\code{\link{gu}()},
\code{\link{le_week}()},
\code{\link{msf_sc}()},
\code{\link{msl}()},
\code{\link{napd}()},
\code{\link{sd24}()},
\code{\link{sd_week}()},
\code{\link{sdu}()},
\code{\link{sjl_sc}()},
\code{\link{sjl_weighted}()},
\code{\link{sjl}()},
\code{\link{so}()},
\code{\link{tbt}()}
}
\concept{MCTQ functions}
