% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/densityMclustBounded.R
\name{cdfDensityBounded}
\alias{cdfDensityBounded}
\alias{quantileDensityBounded}
\alias{densityMclustBounded.diagnostic}
\title{Cumulative distribution and quantiles of univariate model-based mixture
density estimation for bounded data}
\usage{
cdfDensityBounded(object, data, ngrid = 100, ...)

quantileDensityBounded(object, p, ...)

densityMclustBounded.diagnostic(
  object,
  type = c("cdf", "qq"),
  col = c("black", "black"),
  lwd = c(2, 1),
  lty = c(1, 1),
  legend = TRUE,
  grid = TRUE,
  ...
)
}
\arguments{
\item{object}{An object of class \code{'mclustDensityBounded'} obtained from
a call to \code{\link[=densityMclustBounded]{densityMclustBounded()}} function.}

\item{data}{A numeric vector of evaluation points.}

\item{ngrid}{The number of points in a regular grid to be used as evaluation
points if no \code{data} are provided.}

\item{\dots}{Additional arguments.}

\item{p}{A numeric vector of probabilities corresponding to quantiles.}

\item{type}{The type of graph requested:
\itemize{
\item \code{"cdf"} A plot of the estimated CDF versus the empirical distribution
function.
\item \code{"qq"} A Q-Q plot of sample quantiles versus the quantiles obtained from
the inverse of the estimated cdf.
}}

\item{col}{A pair of values for the color to be used for plotting,
respectively, the estimated CDF and the empirical cdf.}

\item{lwd}{A pair of values for the line width to be used for plotting,
respectively, the estimated CDF and the empirical cdf.}

\item{lty}{A pair of values for the line type to be used for plotting,
respectively, the estimated CDF and the empirical cdf.}

\item{legend}{A logical indicating if a legend must be added to the plot of
fitted CDF vs the empirical CDF.}

\item{grid}{A logical indicating if a \code{\link[=grid]{grid()}} should be added to
the plot.}
}
\value{
\code{cdfDensityBounded()} returns a list of \code{x} and \code{y} values providing,
respectively, the evaluation points and the estimated cdf.

\code{quantileDensityBounded()} returns a vector of quantiles.

No return value, called for side effects.
}
\description{
Compute the cumulative density function (cdf) or quantiles of a
one-dimensional density for bounded data estimated via the
transformation-based approach for Gaussian mixtures in
\code{\link[=densityMclustBounded]{densityMclustBounded()}}.

Diagnostic plots for density estimation of bounded data via
transformation-based approach of Gaussian mixtures. Only available for the
one-dimensional case.

The two diagnostic plots for density estimation in the one-dimensional case
are discussed in Loader (1999, pp- 87-90).
}
\details{
The cdf is evaluated at points given by the optional argument \code{data}.
If not provided, a regular grid of length \code{ngrid} for the evaluation
points is used.

The quantiles are computed using bisection linear search algorithm.
}
\examples{
\donttest{
# univariate case with lower bound
x <- rchisq(200, 3)
dens <- densityMclustBounded(x, lbound = 0)

xgrid <- seq(-2, max(x), length=1000)
cdf <- cdfDensityBounded(dens, xgrid)
str(cdf)
plot(xgrid, pchisq(xgrid, df = 3), type = "l", xlab = "x", ylab = "CDF")
lines(cdf, col = 4, lwd = 2)

q <- quantileDensityBounded(dens, p = c(0.01, 0.1, 0.5, 0.9, 0.99))
cbind(quantile = q, cdf = cdfDensityBounded(dens, q)$y)
plot(cdf, type = "l", col = 4, xlab = "x", ylab = "CDF")
points(q, cdfDensityBounded(dens, q)$y, pch = 19, col = 4)

# univariate case with lower & upper bounds
x <- rbeta(200, 5, 1.5)
dens <- densityMclustBounded(x, lbound = 0, ubound = 1)

xgrid <- seq(-0.1, 1.1, length=1000)
cdf <- cdfDensityBounded(dens, xgrid)
str(cdf)
plot(xgrid, pbeta(xgrid, 5, 1.5), type = "l", xlab = "x", ylab = "CDF")
lines(cdf, col = 4, lwd = 2)

q <- quantileDensityBounded(dens, p = c(0.01, 0.1, 0.5, 0.9, 0.99))
cbind(quantile = q, cdf = cdfDensityBounded(dens, q)$y)
plot(cdf, type = "l", col = 4, xlab = "x", ylab = "CDF")
points(q, cdfDensityBounded(dens, q)$y, pch = 19, col = 4)
}

\donttest{
# univariate case with lower bound
x <- rchisq(200, 3)
dens <- densityMclustBounded(x, lbound = 0)
plot(dens, x, what = "diagnostic")
# or
densityMclustBounded.diagnostic(dens, type = "cdf")
densityMclustBounded.diagnostic(dens, type = "qq")

# univariate case with lower & upper bounds
x <- rbeta(200, 5, 1.5)
dens <- densityMclustBounded(x, lbound = 0, ubound = 1)
plot(dens, x, what = "diagnostic")
# or
densityMclustBounded.diagnostic(dens, type = "cdf")
densityMclustBounded.diagnostic(dens, type = "qq")
}

}
\references{
Loader C. (1999), Local Regression and Likelihood. New York, Springer.
}
\seealso{
\code{\link[=densityMclustBounded]{densityMclustBounded()}}, \code{\link[=plot.densityMclustBounded]{plot.densityMclustBounded()}}.

\code{\link[=densityMclustBounded]{densityMclustBounded()}}, \code{\link[=plot.densityMclustBounded]{plot.densityMclustBounded()}}.
}
\author{
Luca Scrucca
}
