#' Get DRI Intersubjective Consistency (IC)
#'
#' `get_dri_ic` calculates the intersubjective consistency (IC) between
#' deliberation participants using their DRI survey responses
#'
#' @param data the raw DRI survey response data
#'
#' @seealso [deliberr::human_data] for raw survey response data formatting
#' @family DRI survey methods
#'
#' @returns dataframe with `pnums` or participant numbers, `pnum1` and `pnum2`,
#' or the unique number of participant 1 and 2, respectively, `ccor` and `pcor`, or
#' the correlation between considerations statement ratings and policy
#' preference rankings, respecively, and `dj`, or the modal orthogonal distance
#' `d` for a given  pair `j`
#'
#' @export
#' @import dplyr
#' @importFrom stats cor
#' @importFrom tibble column_to_rownames
#' @importFrom tidyr pivot_longer
#' @importFrom rlang .data
#' @examples
#'
#' # get post-deliberation (stage_id == 2) data from Zukunft case
#' data <- human_data[human_data$stage_id == 2 & human_data$case == "Zukunft", ]
#' get_dri_ic(data)
#'
get_dri_ic <- function(data) {

  # create sequential participant numbers if the original data doesn't have it
  # make it negative to differentiate from real pnums
  if (!"pnum" %in% colnames(data)) {
    data$pnum <- -as.integer(rownames(data))
  }

  # create separate dataframes for C and P columns
  df_c <- data %>% select(.data$pnum, matches("^C\\d+$") &
                            where(~ !all(is.na(.x))))
  df_p <- data %>% select(.data$pnum, matches("^P\\d+$") &
                            where(~ !all(is.na(.x))))

  # transpose the data and make pnum the row names for correlation calculation
  df_c_t <- df_c %>% column_to_rownames("pnum") %>% t()
  df_p_t <- df_p %>% column_to_rownames("pnum") %>% t()

  # calculate correlation matrices and reshapes them to a long format
  corr_c <- cor(df_c_t, method = "spearman")
  corr_p <- cor(df_p_t, method = "spearman")

  # convert matrices to dataframes and add a column for pnum
  df_corr_c <- as.data.frame(corr_c) %>%
    rownames_to_column(var = "pnum1") %>%
    mutate(pnum1 = as.numeric(.data$pnum1))

  df_corr_p <- as.data.frame(corr_p) %>%
    rownames_to_column(var = "pnum1") %>%
    mutate(pnum1 = as.numeric(.data$pnum1))

  # reshape from wide to long format
  long_corr_c <- df_corr_c %>%
    pivot_longer(
      cols = -.data$pnum1,
      names_to = "pnum2",
      values_to = "ccor",
      names_transform = as.numeric
    )

  long_corr_p <- df_corr_p %>%
    pivot_longer(
      cols = -.data$pnum1,
      names_to = "pnum2",
      values_to = "pcor",
      names_transform = as.numeric
    )

  # join the two dataframes, filter for unique pairs, and create the final output
  ic <- long_corr_c %>%
    inner_join(long_corr_p, by = c("pnum1", "pnum2")) %>%
    filter(.data$pnum1 < .data$pnum2) %>% # filter out duplicate pairs and self-correlations
    mutate(
      pnums = paste0(.data$pnum1, "-", .data$pnum2),
      dj = abs(.data$ccor - .data$pcor) / sqrt(2) # calculate modal orthogonal distance
    ) %>%
    select(.data$pnums, .data$pnum1, .data$pnum2, .data$ccor, .data$pcor, .data$dj)

  return(ic)

}
