test_that("select block expression generation - include mode", {
  skip_if_not_installed("shiny")
  skip_if_not_installed("blockr.core")

  # Test with selected columns (include mode)
  test_data <- reactive(data.frame(
    a = c(1, 2, 3),
    b = c("x", "y", "z"),
    c = c(10, 20, 30)
  ))

  blk <- new_select_block(columns = c("a", "c"), exclude = FALSE)

  # Test that server function works
  shiny::testServer(
    blk$expr_server,
    args = list(data = test_data),
    {
      session$flushReact()

      # Check that server returned a list with expr and state
      result <- session$returned
      expect_true(is.reactive(result$expr))
      expect_true(is.list(result$state))

      # Check expression
      expr_result <- result$expr()
      expect_true(inherits(expr_result, "call"))
      expr_text <- deparse(expr_result)
      expect_true(any(grepl("dplyr::select", expr_text)))
      expect_true(any(grepl("a", expr_text)))
      expect_true(any(grepl("c", expr_text)))
    }
  )
})

test_that("select block expression generation - exclude mode", {
  skip_if_not_installed("shiny")
  skip_if_not_installed("blockr.core")

  # Test with selected columns (exclude mode)
  test_data <- reactive(data.frame(
    a = c(1, 2, 3),
    b = c("x", "y", "z"),
    c = c(10, 20, 30)
  ))

  blk <- new_select_block(columns = c("b"), exclude = TRUE)

  # Test that server function works
  shiny::testServer(
    blk$expr_server,
    args = list(data = test_data),
    {
      session$flushReact()

      # Check that server returned a list with expr and state
      result <- session$returned
      expect_true(is.reactive(result$expr))
      expect_true(is.list(result$state))

      # Set inputs (they don't get automatically set from constructor in testServer)
      session$setInputs(columns = "b", exclude = TRUE)
      session$flushReact()

      # Check expression - should use minus syntax
      expr_result <- result$expr()
      expect_true(inherits(expr_result, "call"))
      expr_text <- deparse(expr_result)
      expect_true(any(grepl("dplyr::select", expr_text)))
      expect_true(any(grepl("-c\\(", expr_text))) # Minus syntax with -c(
      expect_true(any(grepl("`?b`?", expr_text)))
    }
  )
})

test_that("select block expression generation - empty selection", {
  skip_if_not_installed("shiny")
  skip_if_not_installed("blockr.core")

  test_data <- reactive(data.frame(
    a = c(1, 2, 3),
    b = c("x", "y", "z")
  ))

  # Empty selection in include mode = select all
  blk_include <- new_select_block(columns = character(0), exclude = FALSE)

  shiny::testServer(
    blk_include$expr_server,
    args = list(data = test_data),
    {
      session$flushReact()

      result <- session$returned
      expr_result <- result$expr()
      expr_text <- deparse(expr_result)
      # Should select everything when empty
      expect_true(any(grepl("everything", expr_text)))
    }
  )

  # Empty selection in exclude mode = select all
  blk_exclude <- new_select_block(columns = character(0), exclude = TRUE)

  shiny::testServer(
    blk_exclude$expr_server,
    args = list(data = test_data),
    {
      session$flushReact()

      result <- session$returned
      expr_result <- result$expr()
      expr_text <- deparse(expr_result)
      # Should select all using everything()
      expect_true(any(grepl("everything", expr_text)))
    }
  )
})

test_that("select block UI generation", {
  blk <- new_select_block(columns = c("mpg", "cyl"))

  ui_output <- blk$expr_ui("test_id")

  expect_s3_class(ui_output, "shiny.tag.list")
  ui_text <- as.character(ui_output)

  # Should contain selectizeInput
  expect_true(grepl("selectize", ui_text, ignore.case = TRUE))

  # Should contain checkbox for exclude mode
  expect_true(grepl("checkbox", ui_text, ignore.case = TRUE))
  expect_true(grepl("Exclude", ui_text, ignore.case = TRUE))
})

test_that("select block reactive updates", {
  skip_if_not_installed("shiny")
  skip_if_not_installed("blockr.core")

  test_data <- reactive(data.frame(
    mpg = c(21, 21, 22.8),
    cyl = c(6, 6, 4),
    hp = c(110, 110, 93)
  ))

  blk <- new_select_block(columns = c("mpg"))

  shiny::testServer(
    blk$expr_server,
    args = list(data = test_data),
    {
      session$flushReact()

      result <- session$returned

      # Set initial inputs first (they don't get automatically set from constructor in testServer)
      session$setInputs(columns = "mpg", exclude = FALSE)
      session$flushReact()

      # Check that columns() reactive returns "mpg"
      expect_equal(result$state$columns(), "mpg")
      expect_false(isTRUE(result$state$exclude()))

      # Update columns
      session$setInputs(columns = c("mpg", "cyl"))
      session$flushReact()
      expect_equal(result$state$columns(), c("mpg", "cyl"))

      # Update exclude mode
      session$setInputs(exclude = TRUE)
      session$flushReact()
      expect_true(isTRUE(result$state$exclude()))
    }
  )
})

# Data transformation tests using block_server
test_that("select block selects columns - testServer", {
  block <- new_select_block(columns = c("mpg", "cyl", "hp"))

  testServer(
    blockr.core:::get_s3_method("block_server", block),
    {
      session$flushReact()
      result <- session$returned$result()

      expect_true(is.data.frame(result))
      expect_equal(nrow(result), nrow(mtcars))
      expect_equal(ncol(result), 3)
      expect_true(all(c("mpg", "cyl", "hp") %in% names(result)))
      expect_false("wt" %in% names(result))
    },
    args = list(x = block, data = list(data = function() mtcars))
  )
})

test_that("select block with exclude mode - testServer", {
  block <- new_select_block(columns = c("mpg", "cyl"), exclude = TRUE)

  testServer(
    blockr.core:::get_s3_method("block_server", block),
    {
      session$flushReact()
      result <- session$returned$result()

      expect_true(is.data.frame(result))
      expect_false("mpg" %in% names(result))
      expect_false("cyl" %in% names(result))
      expect_true("hp" %in% names(result))
      expect_true("wt" %in% names(result))
    },
    args = list(x = block, data = list(data = function() mtcars))
  )
})

test_that("select block with distinct=TRUE and specific columns - testServer", {
  # Create data with duplicate rows
  test_data <- data.frame(
    cyl = c(4, 4, 6, 6, 8, 8),
    gear = c(4, 4, 4, 5, 3, 3),
    am = c(1, 1, 1, 1, 0, 0),
    mpg = c(21, 21, 19, 20, 15, 15)
  )

  block <- new_select_block(columns = c("cyl", "gear"), distinct = TRUE)

  testServer(
    blockr.core:::get_s3_method("block_server", block),
    {
      session$flushReact()
      result <- session$returned$result()

      expect_true(is.data.frame(result))
      # Should only have cyl and gear columns
      expect_equal(ncol(result), 2)
      expect_true(all(c("cyl", "gear") %in% names(result)))
      # Should have only unique combinations
      expect_equal(nrow(result), 4)  # (4,4), (6,4), (6,5), (8,3)
    },
    args = list(x = block, data = list(data = function() test_data))
  )
})

test_that("select block with distinct=TRUE and empty selection - testServer", {
  # Create data with fully duplicate rows
  test_data <- data.frame(
    a = c(1, 1, 2, 2, 3),
    b = c("x", "x", "y", "y", "z")
  )

  block <- new_select_block(columns = character(0), distinct = TRUE)

  testServer(
    blockr.core:::get_s3_method("block_server", block),
    {
      session$flushReact()
      result <- session$returned$result()

      expect_true(is.data.frame(result))
      # With empty selection + distinct, should deduplicate all columns
      expect_equal(ncol(result), 2)  # All columns
      expect_equal(nrow(result), 3)  # Only unique rows (1,x), (2,y), (3,z)
    },
    args = list(x = block, data = list(data = function() test_data))
  )
})

test_that("select block with distinct=TRUE and exclude=TRUE - testServer", {
  test_data <- data.frame(
    id = c(1, 1, 2, 2, 3),
    value_a = c(10, 10, 20, 20, 30),
    value_b = c(15, 15, 25, 25, 35),
    flag = c("A", "A", "B", "B", "C")
  )

  # Exclude value_a, then get distinct
  block <- new_select_block(
    columns = c("value_a"),
    exclude = TRUE,
    distinct = TRUE
  )

  testServer(
    blockr.core:::get_s3_method("block_server", block),
    {
      session$flushReact()
      result <- session$returned$result()

      expect_true(is.data.frame(result))
      # Should have id, value_b, flag (not value_a)
      expect_false("value_a" %in% names(result))
      expect_true(all(c("id", "value_b", "flag") %in% names(result)))
      # Should have unique combinations of remaining columns
      expect_equal(nrow(result), 3)  # 3 unique combinations
    },
    args = list(x = block, data = list(data = function() test_data))
  )
})
