#' Extract Benchmark Metrics from Scored Data
#'
#' Extracts benchmarking metrics from a data frame that was scored with
#' \code{score_anomaly()} using ground truth labels.
#'
#' @param scored_data A data frame with anomaly scores, typically the output
#'   of \code{score_anomaly()} with a \code{ground_truth_col} specified.
#'
#' @return A list containing benchmarking metrics, or NULL if no metrics
#'   are available.
#'
#' @export
#'
#' @examples
#' \donttest{
#' data <- data.frame(
#'   patient_id = 1:50,
#'   age = rnorm(50, 50, 15),
#'   is_error = sample(c(0, 1), 50, replace = TRUE, prob = c(0.95, 0.05))
#' )
#' scored_data <- score_anomaly(data, ground_truth_col = "is_error")
#' metrics <- extract_benchmark_metrics(scored_data)
#' print(metrics$auc_roc)
#' }
extract_benchmark_metrics <- function(scored_data) {
  metrics <- attr(scored_data, "benchmark_metrics")
  if (is.null(metrics)) {
    warning("No benchmark metrics found. Did you provide ground_truth_col to score_anomaly()?")
  }
  return(metrics)
}

#' Get Top Anomalous Records
#'
#' Convenience function to extract the top N most anomalous records from
#' scored data.
#'
#' @param scored_data A data frame with anomaly scores.
#' @param n Integer indicating the number of top records to return.
#'   Default is 100.
#'
#' @return A data frame containing the top N most anomalous records,
#'   sorted by anomaly_score (descending).
#'
#' @export
#'
#' @examples
#' \donttest{
#' data <- data.frame(
#'   patient_id = 1:50,
#'   age = rnorm(50, 50, 15),
#'   cost = rnorm(50, 10000, 5000)
#' )
#' scored_data <- score_anomaly(data)
#' top_10 <- get_top_anomalies(scored_data, n = 10)
#' }
get_top_anomalies <- function(scored_data, n = 100) {
  if (!is.data.frame(scored_data)) {
    stop("scored_data must be a data frame")
  }
  
  if (!"anomaly_score" %in% names(scored_data)) {
    stop("scored_data must contain an 'anomaly_score' column")
  }
  
  n <- min(n, nrow(scored_data))
  
  result <- scored_data %>%
    dplyr::arrange(dplyr::desc(.data$anomaly_score)) %>%
    dplyr::slice_head(n = n)
  
  return(result)
}

