% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svem_random_table_multi.R
\name{svem_random_table_multi}
\alias{svem_random_table_multi}
\title{Generate a Random Prediction Table from Multiple SVEMnet Models (no refit)}
\usage{
svem_random_table_multi(
  objects,
  n = 1000,
  mixture_groups = NULL,
  debias = FALSE,
  range_tol = 1e-08,
  numeric_sampler = c("random", "uniform")
)
}
\arguments{
\item{objects}{A list of fitted \code{svem_model} objects returned by
\code{SVEMnet()}. Each object must contain a valid \code{$sampling_schema}
produced by the updated \code{SVEMnet()} implementation. A single model is
also accepted and treated as a length-one list.}

\item{n}{Number of random points to generate (rows in the output tables).
Default is \code{1000}.}

\item{mixture_groups}{Optional list of mixture constraint groups. Each group
is a list with elements \code{vars}, \code{lower}, \code{upper}, \code{total}
(see \emph{Notes on mixtures}). Mixture variables must be numeric-like and
must also appear in the models' \code{predictor_vars} (that is, they must
be used as predictors in all models).}

\item{debias}{Logical; if \code{TRUE}, apply each model's calibration during
prediction when available (for Gaussian fits). This is passed to
\code{predict.svem_model()}. Default is \code{FALSE}.}

\item{range_tol}{Numeric tolerance for comparing numeric ranges across models
(used when checking that all \code{$sampling_schema$num_ranges} agree).
Default is \code{1e-8}.}

\item{numeric_sampler}{Sampler for non-mixture numeric predictors:
\code{"random"} (default), or \code{"uniform"}.
\itemize{
\item \code{"random"}: random Latin hypercube when the \pkg{lhs} package
is available; otherwise independent uniforms via \code{runif()}.
\item \code{"uniform"}: independent uniform draws within numeric ranges
(fastest; no \pkg{lhs} dependency).
}}
}
\value{
A list with three data frames:
\itemize{
\item \code{data}: the sampled predictor settings, one row per random point.
\item \code{pred}: one column per response, aligned to \code{data} rows.
\item \code{all}: \code{cbind(data, pred)} for convenience.
}
Each prediction column is named by the model's response (left-hand side)
with a "_pred" suffix (for example, "y1_pred"). If that name would collide
with a predictor name or with another prediction column, the function stops
with an error and asks the user to rename the response or predictor.
}
\description{
Samples the original predictor factor space cached in fitted \code{svem_model}
objects and computes predictions from each model at the same random points.
This is intended for multiple responses built over the same factor space and
a deterministic factor expansion (for example via a shared
\code{\link{bigexp_terms}}), so that a shared sampling schema is available.
}
\details{
No refitting is performed. Predictions are obtained by
averaging per-bootstrap member predictions on the requested scale.

All models must share an identical predictor schema. Specifically, their
\code{$sampling_schema} entries must agree on:
\itemize{
\item The same \code{predictor_vars} in the same order.
\item The same \code{var_classes} for each predictor.
\item Identical factor \code{levels} and level order for all categorical
predictors.
\item Numeric \code{num_ranges} that match within \code{range_tol} for all
continuous predictors.
\item When present, the same \code{blocking} set (up to order).
}
The function stops with an informative error message if any of these checks fail.

Models may be Gaussian or binomial. For binomial fits, predictions are
returned on the probability scale (that is, on the response scale) by default,
consistent with the default behaviour of \code{predict.svem_model()}.
}
\section{Typical workflow}{

\enumerate{
\item Build a deterministic expansion (for example with \code{\link{bigexp_terms}})
and fit several \code{SVEMnet()} models for different responses on
the same factor space, using the same expansion / sampling settings.
\item Ensure that the fitted models were created with a version of
\code{SVEMnet()} that populates \code{$sampling_schema}.
\item Collect the fitted models in a list and pass them to
\code{svem_random_table_multi()}.
\item Use \code{$data} (predictors), \code{$pred} (response columns), or
\code{$all} (\code{cbind(data, pred)}) for downstream plotting,
summarization, or cross-response comparison.
}
}

\section{Blocking variables}{

If the models were fit using a \code{bigexp_spec} that included blocking
variables (for example \code{blocking = c("Operator", "Plate_ID")}) and
\code{SVEMnet()} stored these in \code{$sampling_schema$blocking}, then
\code{svem_random_table_multi()} will:
\itemize{
\item treat those variables as blocking factors; and
\item hold them fixed at a single value across the sampled table.
}

Specifically:
\itemize{
\item For blocking numeric variables, the function uses the midpoint of the
recorded numeric range, \code{(min + max) / 2}, for all rows.
\item For blocking categorical variables, the function uses a single
reference level equal to the most frequent observed level (mode) in the
training data, with ties broken deterministically; if the mode is
unavailable, it falls back to the first stored level.
}

Blocking variables are not allowed to appear in \code{mixture_groups}. If
any mixture group tries to use a blocking variable, the function stops with
an error.

When no blocking information is present in \code{$sampling_schema} (for
example for models fit without a \code{bigexp_spec} or without blocking),
the behavior is unchanged from earlier versions: all predictors are sampled
according to the rules described under "Sampling strategy".
}

\section{Sampling strategy}{

Non-mixture numeric variables are sampled using the chosen \code{numeric_sampler}
within the numeric ranges recorded in \code{$sampling_schema$num_ranges}:
\itemize{
\item \code{"random"}: random Latin hypercube when \pkg{lhs} is available,
else independent uniforms on each range.
\item \code{"uniform"}: independent uniform draws within numeric ranges
(fastest; no \pkg{lhs} dependency).
}

Mixture variables (if any) are sampled jointly within each specified group using
a truncated Dirichlet so that elementwise bounds and the total sum are satisfied.
Categorical variables are sampled from cached factor levels. Blocking variables
(if present) are held fixed (single level or single numeric value) and are not
randomized.

The same random predictor table is fed to each model so response columns are
directly comparable.
}

\section{Notes on mixtures}{

Each mixture group should list only numeric-like variables. Bounds are interpreted
on the original scale of those variables. If \code{total} equals the sum of lower
bounds, the sampler returns the lower-bound corner for that group. Infeasible
constraints (that is, \code{sum(lower) > total} or \code{sum(upper) < total})
produce an error.

Mixture variables are removed from the pool of "non-mixture" numeric variables
before numeric sampling, so they are controlled entirely by the mixture
constraints and not also sampled independently. Mixture variables are not
allowed to be blocking variables.
}

\examples{
\donttest{
set.seed(1)
n <- 60
X1 <- runif(n); X2 <- runif(n)
A <- runif(n); B <- runif(n); C <- pmax(0, 1 - A - B)
F <- factor(sample(c("lo","hi"), n, TRUE))

## Gaussian responses
y1 <- 1 + 2*X1 - X2 + 3*A + 1.5*B + 0.5*C + (F=="hi") + rnorm(n, 0, 0.3)
y2 <- 0.5 + 0.8*X1 + 0.4*X2 + rnorm(n, 0, 0.2)

## Binomial response (probability via logistic link)
eta  <- -0.5 + 1.2*X1 - 0.7*X2 + 0.8*(F=="hi") + 0.6*A
p    <- 1 / (1 + exp(-eta))
yb   <- rbinom(n, size = 1, prob = p)

d  <- data.frame(y1, y2, yb, X1, X2, A, B, C, F)

fit1 <- SVEMnet(y1 ~ X1 + X2 + A + B + C + F, d, nBoot = 40, family = "gaussian")
fit2 <- SVEMnet(y2 ~ X1 + X2 + A + B + C + F, d, nBoot = 40, family = "gaussian")
fitb <- SVEMnet(yb ~ X1 + X2 + A + B + C + F, d, nBoot = 40, family = "binomial")

# Mixture constraint for A, B, C that sum to 1
mix <- list(list(vars  = c("A","B","C"),
                 lower = c(0,0,0),
                 upper = c(1,1,1),
                 total = 1))

# Fast random sampler (shared predictor table; predictions bound as columns)
tab_fast <- svem_random_table_multi(
  objects         = list(y1 = fit1, y2 = fit2, yb = fitb),
  n               = 2000,
  mixture_groups  = mix,
  debias          = FALSE,
  numeric_sampler = "random"
)
head(tab_fast$all)

# Check that the binomial predictions are on [0,1]
range(tab_fast$pred$yb_pred)

# Uniform sampler (fastest)
tab_uni <- svem_random_table_multi(
  objects         = list(y1 = fit1, y2 = fit2, yb = fitb),
  n               = 2000,
  debias          = FALSE,
  numeric_sampler = "uniform"
)
head(tab_uni$all)

## Example with blocking (requires SVEMnet to store sampling_schema$blocking)
set.seed(2)
df_block <- data.frame(
  y1         = rnorm(40),
  y2         = rnorm(40),
  X1         = runif(40),
  X2         = runif(40),
  Operator   = factor(sample(paste0("Op", 1:3), 40, TRUE)),
  AmbientTmp = rnorm(40, mean = 22, sd = 2)
)

spec_block <- bigexp_terms(
  y1 ~ X1 + X2,
  data             = df_block,
  factorial_order  = 2,
  polynomial_order = 2,
  blocking         = c("Operator", "AmbientTmp")
)

fit_b1 <- SVEMnet(spec_block, df_block, response = "y1", nBoot = 30)
fit_b2 <- SVEMnet(spec_block, df_block, response = "y2", nBoot = 30)

tab_block <- svem_random_table_multi(list(fit_b1, fit_b2), n = 500)

## Operator and AmbientTmp are held fixed across rows:
length(unique(tab_block$data$Operator))
range(tab_block$data$AmbientTmp)
}

}
\seealso{
\code{\link{SVEMnet}}, \code{\link{predict.svem_model}},
\code{\link{bigexp_terms}}, \code{\link{bigexp_formula}}
}
