# Helper functions for stable distribution analysis

# =============================================================================
# GLOBAL VARIABLES
# =============================================================================
# Global variables for iterative optimization
L_alpha <- c()
L_beta <- c()
L_delta <- c()
L_omega <- c()
M_w <- c()

# =============================================================================
# UTILITY FUNCTIONS
# =============================================================================

#' Helper function to unpack parameters
#'
#' @param p A list containing alpha, beta, gamma, delta.
#' @return A numeric vector of parameters.
#' @export
unpack_params <- function(p) {
  return(c(p$alpha, p$beta, p$gamma, p$delta))
}

#' Clip values between lower and upper bounds
#'
#' @param x Numeric input.
#' @param lower Minimum allowed value.
#' @param upper Maximum allowed value.
#' @return Clipped numeric vector.
#' @export
clip <- function(x, lower, upper) {
  pmax(lower, pmin(upper, x))
}

#' Ensure positive scale parameter
#'
#' @param val Numeric input.
#' @param min_scale Minimum scale.
#' @param max_scale Maximum scale.
#' @return A finite positive scale value.
#' @export
ensure_positive_scale <- function(val, min_scale = 1e-6, max_scale = 1e6) {
  if (!is.finite(val) || val <= 0) return(min_scale)
  return(max(min_scale, min(val, max_scale)))
}


# =============================================================================
# BANDWIDTH SELECTION
# =============================================================================

#' KDE bandwidth selection using plugin method
#'
#' @param X Numeric vector of data.
#' @param alpha Stability parameter.
#' @return Bandwidth value.
#' @export
kde_bandwidth_plugin <- function(X, alpha) {
  n <- length(X)
  sigma <- sd(X)
  h_silverman <- 1.06 * sigma * n^(-1/5)

  if (alpha > 1) {
    return(h_silverman)  # Sheather-Jones approx
  } else {
    return(h_silverman * ifelse(alpha < 0.8, 1.06, 1.0))  # Slaoui-like rule
  }
}

# =============================================================================
# KERNEL FUNCTIONS
# =============================================================================


#' Gaussian kernel
#'
#' @param z Numeric input.
#' @return Kernel value.
N_gaussian <- function(z) {
  (1 / sqrt(2 * pi)) * exp(-0.5 * z^2)
}


#' Uniform kernel
#'
#' @param z Numeric input.
#' @return Kernel value.
N_uniform <- function(z) {
  0.5 * (abs(z) <= 1)
}


#' Epanechnikov kernel
#'
#' @param z Numeric input.
#' @return Kernel value.
N_epanechnikov <- function(z) {
  0.75 * (1 - z^2) * (abs(z) <= 1)
}


#' Real part of the ECF integral
#'
#' @param r Integration variable.
#' @param u Frequency.
#' @param x Data point.
#' @param bn Bandwidth.
#' @return Real component value.
Re <- function(r, u, x, bn) {
  cos(u * r) * N_gaussian((r - x) / bn)
}


#' Imaginary part of the ECF integral
#'
#' @param r Integration variable.
#' @param u Frequency.
#' @param x Data point.
#' @param bn Bandwidth.
#' @return Imaginary component value.
Im <- function(r, u, x, bn) {
  sin(u * r) * N_gaussian((r - x) / bn)
}


#' Integrate real component over \eqn{\mathbb{R}}
#'
#' @param u Frequency.
#' @param x Data point.
#' @param bn Bandwidth.
#' @return Integrated real value.
#' @export
Int_Re <- function(u, x, bn) {
  integrate(function(r) Re(r, u, x, bn), lower = -Inf, upper = Inf)$value
}


#' Integrate imaginary component over \eqn{\mathbb{R}}
#'
#' @param u Frequency.
#' @param x Data point.
#' @param bn Bandwidth.
#' @return Integrated imaginary value.
#' @export
Int_Im <- function(u, x, bn) {
  integrate(function(r) Im(r, u, x, bn), lower = -Inf, upper = Inf)$value
}


# =============================================================================
# STABLE DISTRIBUTION FUNCTIONS
# =============================================================================

#' Robust stable PDF computation
#' @param x Points at which to evaluate the density.
#' @param alpha Stability parameter (0,2].
#' @param beta Skewness parameter [-1,1].
#' @param scale Scale (>0).
#' @param location Location parameter.
#' @return Numeric vector of density values.
#' @importFrom stabledist dstable
#' @importFrom libstable4u stable_pdf
#' @export
r_stable_pdf <- function(x, alpha, beta, scale, location) {
  if (!is.numeric(x) || any(!is.finite(x))) stop("x must be numeric and finite.")
  if (!is.finite(alpha) || !(0 < alpha && alpha <= 2)) stop("alpha must be in (0, 2] and finite.")
  if (!is.finite(beta)  || abs(beta) > 1) stop("beta must be in [-1, 1] and finite.")
  if (!is.finite(scale) || scale <= 0) stop("scale must be > 0 and finite.")
  if (!is.finite(location)) stop("location must be finite.")
  if (length(alpha) != 1 || length(beta) != 1 || length(scale) != 1 || length(location) != 1) {
    stop("All parameters (alpha, beta, gamma, delta) must be scalars.")
  }

  tryCatch({
    # Try libstable4u first, fallback to stabledist
    if (requireNamespace("libstable4u", quietly = TRUE)) {
      libstable4u::stable_pdf(x, c(alpha, beta, scale, location))
    } else {
      stabledist::dstable(x, alpha = alpha, beta = beta,
                          gamma = scale, delta = location, pm = 1)
    }
  }, error = function(e) {
    warning("PDF computation failed: ", e$message)
    rep(0, length(x))
  })
}

#' Generate random samples from stable distribution
#' @param n Number of samples.
#' @param alpha Stability parameter (0,2].
#' @param beta Skewness parameter [-1,1].
#' @param gamma Scale (>0).
#' @param delta Location.
#' @param pm Parameterization (0 or 1).
#' @return Numeric vector of samples.
#' @importFrom stabledist rstable
#' @export
rstable <- function(n, alpha, beta, gamma = 1.0, delta = 0.0, pm = 1) {
  if (!(0 < alpha && alpha <= 2)) stop("alpha must be in (0, 2]")
  if (!(abs(beta) <= 1)) stop("beta must be in [-1, 1]")
  if (gamma <= 0) stop("scale must be > 0")
  if (!(pm %in% c(0, 1))) stop("pm must be 0 or 1")

  # Generate samples using stabledist
  samples <- stabledist::rstable(n, alpha, beta, gamma = gamma, delta = delta, pm = pm)
  return(samples)
}

#' Initialize stable distribution parameters
#' @param x Numeric vector of data.
#' @return A list with estimated parameters (alpha, beta, gamma, delta).
#' @importFrom libstable4u stable_fit_init
#' @importFrom stats mad median
#' @export
stable_fit_init <- function(x) {
  if (length(x) < 2 || any(!is.finite(x))) {
    stop("Input data must contain at least 2 finite values.")
  }

  tryCatch({
    if (requireNamespace("libstable4u", quietly = TRUE)) {
      result <- libstable4u::stable_fit_init(x)
      if (length(result) < 4) stop("Result does not contain 4 parameters.")
      list(alpha = result[1], beta = result[2], gamma = result[3], delta = result[4])
    } else {
      # Fallback to moment-based estimates
      list(alpha = 1.5, beta = 0.0, gamma = mad(x), delta = median(x))
    }
  }, error = function(e) {
    warning("Fallback to default parameters due to error: ", e$message)
    list(alpha = 1.5, beta = 0.0, gamma = mad(x), delta = median(x))
  })
}

#' Mixture of two stable PDFs
#'
#' @param x Numeric vector.
#' @param p1 List of parameters for first stable distribution.
#' @param p2 List of parameters for second stable distribution.
#' @param w Mixture weight (0 < w < 1).
#' @return Numeric vector of mixture PDF values.
#' @export
mixture_stable_pdf <- function(x, p1, p2, w) {
  y1 <- do.call(r_stable_pdf, c(list(x), as.list(p1)))
  y2 <- do.call(r_stable_pdf, c(list(x), as.list(p2)))
  return(w * y1 + (1 - w) * y2)
}

# =============================================================================
# LIKELIHOOD FUNCTIONS
# =============================================================================

#' Log-likelihood for mixture of stable distributions
#'
#' @param params Numeric vector of parameters.
#' @param data Numeric vector of observations.
#' @return Negative log-likelihood (for minimization).
#' @importFrom stabledist dstable
#' @export
log_likelihood_mixture <- function(params, data) {
  # Unpack parameters
  w <- params[1]
  a1 <- params[2]; b1 <- params[3]; s1 <- params[4]; l1 <- params[5]
  a2 <- params[6]; b2 <- params[7]; s2 <- params[8]; l2 <- params[9]

  # Parameter constraints
  if (w <= 0 || w >= 1 || a1 <= 0.1 || a1 > 2 || a2 <= 0.1 || a2 > 2 ||
      b1 < -1 || b1 > 1 || b2 < -1 || b2 > 1 || s1 <= 0 || s2 <= 0) {
    return(Inf)
  }

  tryCatch({
    p1 <- dstable(data, alpha = a1, beta = b1, gamma = s1, delta = l1, pm = 1)
    p2 <- dstable(data, alpha = a2, beta = b2, gamma = s2, delta = l2, pm = 1)
    mix_pdf <- w * p1 + (1 - w) * p2
    mix_pdf <- pmax(mix_pdf, 1e-300)  # Avoid log(0)
    log_likelihood <- sum(log(mix_pdf))
    return(-log_likelihood)  # For minimization
  }, error = function(e) {
    message("MLE error:", e$message, "\n")
    return(Inf)
  })
}

#' Negative log-likelihood for single stable distribution
#'
#' @param params Numeric vector (alpha, beta, gamma, delta).
#' @param data Numeric vector of observations.
#' @return Negative log-likelihood.
#' @importFrom stabledist dstable
#' @export
negative_log_likelihood <- function(params, data) {
  alpha <- params[1]; beta <- params[2]; gamma <- params[3]; delta <- params[4]

  if (alpha <= 0 || alpha > 2 || beta < -1 || beta > 1 || gamma <= 0) {
    return(Inf)
  }

  tryCatch({
    pdf_vals <- dstable(data, alpha = alpha, beta = beta, gamma = gamma, delta = delta, pm = 1)
    pdf_vals <- pmax(pdf_vals, 1e-300)
    log_likelihood <- sum(log(pdf_vals))
    return(-log_likelihood)
  }, error = function(e) {
    return(Inf)
  })
}

# =============================================================================
# DISTANCE FUNCTIONS
# =============================================================================

#' Wasserstein distance between two mixture distributions
#'
#' @param params1 List of parameters for first mixture.
#' @param params2 List of parameters for second mixture.
#' @param size Number of samples to approximate distance.
#' @return Wasserstein distance.
#' @importFrom stabledist rstable
#' @export
wasserstein_distance_mixture <- function(params1, params2, size = 5000) {
  # Sample from mixture
  sample_mixture <- function(params, n) {
    if (!is.list(params[[1]])) params <- list(params)

    weights <- sapply(params, function(p) ifelse(is.null(p$pi), 1.0, p$pi))
    weights <- weights / sum(weights)

    samples <- numeric(n)
    for (i in 1:n) {
      comp <- sample(length(params), 1, prob = weights)
      p <- params[[comp]]
      samples[i] <- rstable(1, alpha = p$alpha, beta = p$beta,
                            gamma = p$gamma, delta = p$delta, pm = 1)
    }
    return(samples)
  }

  s1 <- sample_mixture(params1, size)
  s2 <- sample_mixture(params2, size)

  # Simple Wasserstein distance approximation
  s1_sorted <- sort(s1)
  s2_sorted <- sort(s2)
  return(mean(abs(s1_sorted - s2_sorted)))
}

# =============================================================================
# helper function
# =============================================================================


#' Helper function for eta0 computation
#'
#' @param u Frequency vector.
#' @param alpha Stability parameter.
#' @param gamma Scale parameter.
#' @param eps Tolerance for \eqn{\alpha \approx 1}.
#' @return Numeric vector of eta0 values.
eta0 <- function(u, alpha, gamma, eps = 0.05) {
  u <- as.vector(u)

  if (abs(alpha - 1) < eps) {
    y <- (2 / pi) * u * log(gamma * abs(u))
    y[u == 0] <- 0.0
  } else {
    y <- tan(pi * alpha / 2) * sign(u) *
      ((abs(gamma * u) - abs(gamma * u)^alpha) / (gamma^alpha))
  }
  return(y)
}


#' General eta function
#'
#' @param t Frequency vector.
#' @param alpha Stability parameter.
#' @param gamma Scale parameter.
#' @return Numeric vector of eta values.
eta_func <- function(t, alpha, gamma) {
  t <- as.numeric(t)

  if (abs(alpha - 1.0) < .Machine$double.eps^0.5) {
    # Case when alpha is close to 1.0
    return((2 / pi) * t * log(abs(gamma * t)))
  } else {
    # General case
    return(tan(pi * alpha / 2) * sign(t) * (abs(gamma)^(1 - alpha)) * (abs(t) - abs(t)^alpha))
  }
}


#' Recursive weight function
#'
#' @param l Index.
#' @return Weight value.
recursive_weight <- function(l){
  y <- (2/ 3 + 0.05) / l
}


#' Fast numerical integration using trapezoidal rule
#'
#' @param func Function to integrate.
#' @param a Lower bound.
#' @param b Upper bound.
#' @param N Number of points.
#' @return Approximated integral value.
fast_integrate <- function(func, a = -6, b = 6, N = 100) {
  r_vals <- seq(a, b, length.out = N)
  f_vals <- func(r_vals)
  sum(0.5 * (head(f_vals, -1) + tail(f_vals, -1)) * diff(r_vals))
}


# =============================================================================
#  ECF Calculation
# =============================================================================
#' Empirical Characteristic Function
#'
#' @param x Numeric vector of data.
#' @param u Numeric vector of frequencies.
#' @param method Method: "simple", "kernel", or "recursive".
#' @return List with magnitude and phase of ECF.
#' @importFrom stats bw.SJ
#' @export
ecf_fn <- function(x, u, method = "simple") {
  if (length(x) == 0 || length(u) == 0) {
    stop("Input vectors 'x' and 'u' must be non-empty.")
  }

  if (!all(is.finite(x)) || !all(is.finite(u))) {
    stop("Input vectors 'x' and 'u' must contain finite values.")
  }

  if (method == "simple") {
    # Simple empirical characteristic function
    tryCatch({
      outer_prod <- outer(u, x)
      g <- rowSums(cos(outer_prod)) / length(x)
      h <- rowSums(sin(outer_prod)) / length(x)
    }, error = function(e) {
      stop(paste("Error computing ECF components:", e$message))
    })

    g <- pmax(exp(-2), g)
    yr <- sqrt(g^2 + h^2)
    yi <- atan2(h, g)

    return(list(magnitude = yr, phase = yi))

  } else if (method == "kernel") {
    # Bandwidth using Sheather-Jones method
    bn <- bw.SJ(x)

    g <- numeric(length(u))
    h <- numeric(length(u))

    for (i in seq_along(x)) {
      xi <- x[i]
      for (j in seq_along(u)) {
        uj <- u[j]
        g[j] <- g[j] + fast_integrate(function(r) cos(uj * r) * N_gaussian((r - xi) / bn))
        h[j] <- h[j] + fast_integrate(function(r) sin(uj * r) * N_gaussian((r - xi) / bn))
      }
    }
    g <- g / (length(x) * bn)
    h <- h / (length(x) * bn)

    yr <- sqrt(g^2 + h^2)
    yi <- atan2(h, g)

    return(list(magnitude = yr, phase = yi))

  } else if (method == "recursive") {
    # Similar implementation for recursive method
    bn <- bw.SJ(x)
    g <- numeric(length(u))
    h <- numeric(length(u))

    step_n <- prod(1 - recursive_weight(1:length(x)))
    for (i in seq_along(x)) {
      xi <- x[i]
      for (j in seq_along(u)) {
        uj <- u[j]
        weight <- (prod(1 - recursive_weight(1:(i + 1)))^-1) * recursive_weight(i + 1)
        g[j] <- g[j] + weight * fast_integrate(function(r) cos(uj * r) * N_gaussian((r - xi) / bn))
        h[j] <- h[j] + weight * fast_integrate(function(r) sin(uj * r) * N_gaussian((r - xi) / bn))
      }
    }
    g <- g * step_n / bn
    h <- h * step_n / bn

    yr <- sqrt(g^2 + h^2)
    yi <- atan2(h, g)

    return(list(magnitude = yr, phase = yi))

  } else {
    stop("Invalid method. Choose 'simple', 'kernel', or 'recursive'.")
  }
}

