% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.R
\name{predict.lmm}
\alias{predict.lmm}
\title{Predicted Mean Value With Uncertainty For Linear Mixed Model}
\usage{
\method{predict}{lmm}(
  object,
  newdata,
  type = "static",
  p = NULL,
  se = NULL,
  robust = FALSE,
  df = NULL,
  level = 0.95,
  keep.data = NULL,
  format = "long",
  export.vcov = FALSE,
  simplify = TRUE,
  ...
)
}
\arguments{
\item{object}{a \code{lmm} object.}

\item{newdata}{[data.frame] a dataset containing covariate values to condition on.
When setting the argument 'dynamic' predictions should also contain cluster, timepoint, and outcome values.}

\item{type}{[character] evaluate the expected outcome conditional on covariates only (\code{"static"}),
the contribution of each variable to this 'static' prediction (\code{"terms"}),
or the expected outcome conditional covariates and outcome values at other timepoints (\code{"dynamic"}).
Based on the observed outcome and the 'dynamic' prediction for the missing outcome,
can also evaluate the change from first repetitition (\code{"change"}), area under the curve (\code{"auc"}), and the area under the curve minus baseline (\code{"auc-b"}).}

\item{p}{[numeric vector] value of the model coefficients at which to evaluate the predictions. Only relevant if differs from the fitted values.}

\item{se}{[logical] should the standard error and confidence intervals for the predictions be output?
It can also be a logical vector of length 2 to indicate the type of uncertainty to be accounted for: estimation and residual variance.
In particular \code{c(TRUE,TRUE)} provides prediction intervals.}

\item{robust}{[logical] Should robust standard errors (aka sandwich estimator) be output instead of the model-based standard errors.
Not feasible for dynamic predictions when using REML.}

\item{df}{[logical] should a Student's t-distribution be used to model the distribution of the predicted mean. Otherwise a normal distribution is used.}

\item{level}{[numeric,0-1] the confidence level of the confidence intervals.}

\item{keep.data}{[logical] should the dataset relative to which the predicted means are evaluated be output along side the predicted values?
Only possible in the long format.}

\item{format}{[character] should the prediction be output
in a matrix format with clusters in row and timepoints in columns (\code{"wide"}),
or in a data.frame/vector with as many rows as observations (\code{"long"})}

\item{export.vcov}{[logical] should the variance-covariance matrix of the prediction error be outcome as an attribute (\code{"vcov"})?}

\item{simplify}{[logical] simplify the data format (vector instead of data.frame) and column names (no mention of the time variable) when possible.}

\item{...}{Not used. For compatibility with the generic method.}

\item{vcov}{[logical] should the variance-covariance matrix of the predictions be output as an attribute.}
}
\value{
When \code{format="long"}, a data.frame containing the following columns:\itemize{
\item \code{estimate}: predicted mean.
\item \code{se}: uncertainty about the predicted mean.
\item \code{df}: degree of freedom
\item \code{lower}: lower bound of the confidence interval of the predicted mean
\item \code{upper}: upper bound of the confidence interval of the predicted mean
}
When \code{format="wide"}, a matrix containing the predict means (one line per cluster, one column per timepoint).
}
\description{
Predicted mean value conditional on covariates or on covariates and other outcome values.
}
\details{
Static prediction are made using the linear predictor \eqn{X\beta} while dynamic prediction uses the conditional normal distribution of the missing outcome given the observed outcomes. So if outcome 1 is observed but not 2, prediction for outcome 2 is obtain by \eqn{X_2\beta + \sigma_{21}\sigma^{-1}_{22}(Y_1-X_1\beta)}. In that case, the uncertainty is computed as the sum of the conditional variance \eqn{\sigma_{22}-\sigma_{21}\sigma^{-1}_{22}\sigma_{12}} plus the uncertainty about the estimated conditional mean (obtained via delta method using numerical derivatives).

The model terms are computing similarly to \code{stats::predict.lm}, by centering the design matrix around the mean value of the covariates used to fit the model.
Then the centered design matrix is multiplied by the mean coefficients and columns assigned to the same variable (e.g. three level factor variable) are summed together.
}
\examples{
## simulate data in the long format
set.seed(10)
dL <- sampleRem(100, n.times = 3, format = "long")

## fit Linear Mixed Model
eUN.lmm <- lmm(Y ~ visit + X1 + X2 + X5,
               repetition = ~visit|id, structure = "UN", data = dL)

## prediction
newd <- data.frame(X1 = 1, X2 = 2, X5 = 3, visit = factor(1:3, levels = 1:3))
predict(eUN.lmm, newdata = newd)
predict(eUN.lmm, newdata = newd, keep.data = TRUE)
predict(eUN.lmm, newdata = newd, keep.data = TRUE, se = c(TRUE,TRUE))

## dynamic prediction
newd.d1 <- cbind(newd, Y = c(NA,NA,NA))
predict(eUN.lmm, newdata = newd.d1, keep.data = TRUE, type = "dynamic")
newd.d2 <- cbind(newd, Y = c(6.61,NA,NA))
predict(eUN.lmm, newdata = newd.d2, keep.data = TRUE, type = "dynamic")
newd.d3 <- cbind(newd, Y = c(1,NA,NA))
predict(eUN.lmm, newdata = newd.d3, keep.data = TRUE, type = "dynamic")
newd.d4 <- cbind(newd, Y = c(1,1,NA))
predict(eUN.lmm, newdata = newd.d4, keep.data = TRUE, type = "dynamic")
}
\keyword{method}
