dwellprobs <- function(M, J, dwelldist, dwellpar) {
  # Initialize dwell time probability matrix
  dwellprobs <- matrix(NA, M, J)

  if (dwelldist == "pois") {
    # Shifted Poisson distribution
    if (is.null(dwellpar$lambda) || is.null(dwellpar$shift)) {
      stop("please supply 'lambda' and 'shift' for the dwell parameters")
    }
    # Validate parameter constraints
    if (any(dwellpar$lambda < 0) || any(dwellpar$shift <= 0) || any(!is.numeric(dwellpar$lambda)) ||
        any(!is.numeric(dwellpar$shift)) || any(floor(dwellpar$shift) != dwellpar$shift)) {
      stop("lambda must be positive and shift must be a strictly positive integer")
    }
    for (j in 1:J) {
      for (i in 1:M) {
        dwellprobs[i, j] <- dpois(i - dwellpar$shift[j], dwellpar$lambda[j])
      }
    }

  } else if (dwelldist == "nbinom") {
    # Shifted negative binomial distribution
    if (is.null(dwellpar$shift) || is.null(dwellpar$size) || is.null(dwellpar$mu)) {
      stop("please supply 'shift', 'size' and 'mu' for the dwell parameters")
    }
    for (j in 1:J) {
      for (i in 1:M) {
        dwellprobs[i, j] <- dnbinom(i - dwellpar$shift[j], size = dwellpar$size[j], mu = dwellpar$mu[j])
      }
    }

  } else if (dwelldist == "betabinom") {
    # Shifted beta-binomial distribution
    if (is.null(dwellpar$size) || is.null(dwellpar$alpha) || is.null(dwellpar$beta) || is.null(dwellpar$shift)) {
      stop("please supply 'size', 'alpha', 'beta' and 'shift' for the dwell parameters")
    }
    for (j in 1:J) {
      for (i in 1:M) {
        dwellprobs[i, j] <- dbetabinom(i - dwellpar$shift[j], size = dwellpar$size[j],
                                       alpha = dwellpar$alpha[j], beta = dwellpar$beta[j])
      }
    }

  } else if (dwelldist == "geom") {
    # Geometric distribution (standard parameterization)
    for (j in 1:J) {
      for (i in 1:M) {
        dwellprobs[i, j] <- dgeom(i - 1, prob = dwellpar$prob[j])
      }
    }
  }

  # Calculate survival function: P(dwell time >= d)
  surv <- matrix(0, nrow = M, ncol = J)
  for (j in seq_len(J)) {
    temp_vec <- dwellprobs[, j]
    for (i in seq_len(M)) {
      # Survival function = sum of PMF from i to M
      surv[i, j] <- sum(temp_vec[i:M])
    }
  }

  return(list(dwellprobs = dwellprobs, surv = surv))
}



