#' Detect a single change point (parametric methods)
#'
#' @param M minute-level Activity vector;
#' @param dist Actigraphy data distribution family,
#'              including `Gaussian`, `Gamma`, `ZAG` (Zero-Augmented Gamma), `Poisson`, `Exponential`
#' @return the location of the single change point
#' @seealso [sleep_detection()]
#' @export

cp_detect <- function(M,dist){
  if(dist=='Gamma'){
    M <- M + 1e-5
    #para <- fitdistr(M, "gamma")$estimate #did not work for some subjects
    nll <- function(par,data){ #negative log-likelihood function
        shape <- par[1]
        scale <- par[2]
        # Calculate Negative Log-Likelihood
        -sum(stats::dgamma(x = data, shape = shape, scale=scale, log = TRUE))
    }
    mle <- stats::optim(par = c(shape = 0.1, scale = 1), fn = nll, data = M, method = "L-BFGS-B",
                lower = c(0.0000001,0.00000001)) #find min of nll; #"Nelder-Mead" is the default method in matlab
    para <- mle$par
    xi <- para[1] #shape
    item1 <- item2 <- L <- item31 <- item32 <- item3  <- item32 <- item5 <- numeric(length=(length(M)-1))
    for (i in 1:(length(M)-1)){
      seg1 <- M[1:i]
      seg2 <- M[(i+1):length(M)]
      n1 <- length(seg1)
      n2 <- length(seg2)
      n <- length(M)
      item1[i] <- 2*n1*xi*log(sum(seg1))
      item2[i] <- 2*n2*xi*log(sum(seg2))
      item32[i] <- -(2*n1*xi*log(n1*xi)+2*n2*xi*log(n2*xi))
      item5[i] <- 50*(2*n1/n-1)^2*log(n)
#      item31[i] <- n^2*exp(2*n*xi)*xi^(2*n)*xi
#      item32[i] <- (n*xi)^(2*n1*xi)*(n2*xi)^(2*n2*xi)
#      item3[i] <- log(item31[i]/item32[i]) # item31 is near zero when n is large and xi is less than 1

      L[i] <- item1[i]+item2[i] + item32[i] + item5[i]
    } #end of for loop
    CP <- which.min(L)   # the index of the minimum value in Object L, estimated K in the paper
  }#end of if gamma

  if(dist =='ZAG'){ ## zero-augmented gamma

    M <- abs(M)
    mle <- stats::optim(par = c(shape = 0.1, scale = 1, p0 = 0.5), fn = nllag, data = M,
               method = "L-BFGS-B",
               lower = c(1e-7, 1e-7, 1e-7),
               upper = c(Inf, Inf, 1 - 1e-7))

    xi <- mle$par[1]

	item1x <- item2x <- item3x <- item1y <- item2y <- item3y <- L <- numeric(length = (length(M) - 1))

	for (i in 1:(length(M) - 1)) {
    seg1 <- M[1:i]
    seg2 <- M[(i + 1):length(M)]

    numx <- sum(seg1 == 0)
    numy <- sum(seg2 == 0)

    px <- numx / i
    py <- numy / (length(M) - i)

    meanx <- sum(seg1[seg1 > 0]) / (i - numx)
    meany <- sum(seg2[seg2 > 0]) / (length(M) - i - numy)
    # Information Components
    item1x[i] <- -2 * numx * log(px)
    item2x[i] <- 2 * (i - numx) * (log(gamma(xi)) - log(1 - px) +
                                     xi * log(meanx) - xi * log(xi) + xi)
   # item3x[i] <- -2 * (xi - 1) * sum(log(seg1[seg1 > 0]))
	##
    item1y[i] <- -2 * numy * log(py)
    item2y[i] <- 2 * (length(M) - i - numy) * (log(gamma(xi)) - log(1 - py) +
                                         xi * log(meany) - xi * log(xi) + xi)
  #  item3y[i] <- -2 * (xi - 1) * sum(log(seg2[seg2 > 0]))

    L[i] <- item1x[i] + item2x[i] + item1y[i] + item2y[i]
  }

  CP <- which.min(L)
}

  if(dist=='Poisson'){
    item1 <- item2 <- L <- numeric(length=(length(M)-1))
    for (i in 1:(length(M)-1)){
      seg1 <- M[1:i]
      seg2 <- M[(i+1):length(M)]
      n1 <- length(seg1)
      n2 <- length(seg2)
      n <- length(M)
      item1[i] <- 2*sum(seg1) - 2*sum(seg1)*log(mean(seg1))
      item2[i] <- 2*sum(seg2) - 2*sum(seg2)*log(mean(seg2))
      L[i] <- item1[i]+item2[i]
    } #end of for loop
    CP <- which.min(L)
  }#end of if poisson



  if(dist=='Gaussian'){ #matlab has two cps while r has one.
    item1 <- item2 <- L <- numeric(length=(length(M)-1))
    for (i in 1:(length(M)-1)){
      seg1 <- M[1:i]
      seg2 <- M[(i+1):length(M)]
      n1 <- length(seg1)
      n2 <- length(seg2)
      n <- length(M)
      item1[i] <- n1*log(mse(seg1))
      item2[i] <- n2*log(mse(seg2))
      L[i] <- item1[i]+item2[i]
    } #end of for loop
    CP <- which.min(L)
  }#end of gauss

  if(dist=='Exponential'){
    item1 <- item2 <- item3 <- item4 <- L <- numeric(length=(length(M)-1))
    for (i in 1:(length(M)-1)){
      seg1 <- M[1:i]
      seg2 <- M[(i+1):length(M)]
      n1 <- length(seg1)
      n2 <- length(seg2)
      n <- length(M)

      item1[i] <- 2*n1*log(sum(seg1));
      item2[i] <- 2*(n-n1+1)*log(sum(seg2));
      item3[i] <- -2*n1*log(n1);
      item4[i] <- -2*(n-n1+1)*log(n-n1+1);
      L[i] <- item1[i] + item2[i] + item3[i] + item4[i]
    } #end of for loop
    CP <- which.max(L)
  }#end of exp

  CP <- unique(CP)
  return(CP)
} #end of function


