#' determineNumSigs
#'
#' This function uses the extracted features and modelled components, and it
#' performs NMF on these ranging from the minimum number of signatures to the
#' max. It repeats this on randomized data and computes various measures to
#' help inform the user on how many signatures to proceed with. This function
#' may take a while to run since it repeats the NMF process many times. It is 
#' suggested to give it multiple cores to allow for parallel processing.
#' 
#' @param scm Sample by component matrix used to find signatures
#' @param rmin The lower bound of signature numbers to check. Default is 2.
#' @param rmax The upper bound of signature numbers to check. Default is 12.
#' @param cores The number of cores to use for parallel analysis. Default is 1.
#' @param nrun Number of runs for NMF. Default is 250.
#' @param saveRes Whether or not to save the plot. Default is FALSE.
#' @param saveDir Directory to save plot in, must be provided if using saveDir
#' @param runName Used to title plots and files when saving results
#' @keywords number of signatures signum
#' @import ggplot2
#' @return Creates a series of plots to help user decide
#' @export
#' @examples
#' \donttest{
#' determineNumSigs(generateSCM(featsExp,compsExp))}
determineNumSigs <- function(scm, rmin = 3, rmax = 12, cores = 1,
                             nrun=250,saveRes=FALSE, saveDir = NULL, runName= "") {

   message("Estimating NMF ranks for dataset.\n")
   estimatedRanks = NMF::nmfEstimateRank(t(scm),rmin:rmax,seed=77777,
                                  nrun,verbose=TRUE,method="KL",
                                  .opt = paste0("p",cores))

  message("Estimating NMF ranks for randomized data.\n")
  scmRand = NMF::randomize(t(scm))
  estimatedRanksRand = NMF::nmfEstimateRank(scmRand, rmin:rmax,seed=77777,nrun,
                                        verbose=TRUE,method="KL",
                                        .opt = paste0("p",cores))

  if (runName != ""){
     runName = paste0(runName,"_")
  }

  if (saveRes)
    if (is.null(saveDir)){
	stop('No saveDir provided for saveRes. Please set saveDir.')
    }
    grDevices::tiff(paste0(saveDir,"/",runName,"DetermineSignaturesPlot.tiff"),
                    width = 6, height = 4, units = 'in', res = 300)

   p = NMF::plot(estimatedRanks,estimatedRanksRand,
            what = c("cophenetic", "dispersion","sparseness", "silhouette", "rss"),
            xname="Observed",yname="Randomised",main="")+
       theme(axis.text=element_text(size=5),axis.title=element_text(size=5),
             strip.text.x = element_text(size = 5),
             strip.text.y = element_text(size = 5),
             legend.text = element_text(size = 5),
             legend.title = element_text(size = 7))

   #Goes through and decreases the size of the points on the graph
   g = ggplotGrob(p)
   g[["grobs"]][[2]]$children[[4]]$gp$fontsize = g[["grobs"]][[2]]$children[[4]]$gp$fontsize/2
   g[["grobs"]][[3]]$children[[4]]$gp$fontsize = g[["grobs"]][[3]]$children[[4]]$gp$fontsize/2
   g[["grobs"]][[4]]$children[[4]]$gp$fontsize = g[["grobs"]][[4]]$children[[4]]$gp$fontsize/2
   g[["grobs"]][[5]]$children[[4]]$gp$fontsize = g[["grobs"]][[5]]$children[[4]]$gp$fontsize/2
   g[["grobs"]][[6]]$children[[4]]$gp$fontsize = g[["grobs"]][[6]]$children[[4]]$gp$fontsize/2

   grid::grid.newpage()
   grid::grid.draw(g)

   if (saveRes)
      grDevices::dev.off()
}
