% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/set_labels.R
\name{set_labels}
\alias{set_labels}
\title{Add value labels to variables}
\usage{
set_labels(x, ..., labels, force.labels = FALSE, force.values = TRUE,
  drop.na = TRUE)
}
\arguments{
\item{x}{A vector or data frame.}

\item{...}{Optional, unquoted names of variables that should be selected for
further processing. Required, if \code{x} is a data frame (and no
vector) and only selected variables from \code{x} should be processed.
You may also use functions like \code{:} or tidyselect's
\code{\link[tidyselect]{select_helpers}}.
See 'Examples'.}

\item{labels}{(Named) character vector of labels that will be added to \code{x} as
\code{"labels"} or \code{"value.labels"} attribute.
\itemize{
  \item if \code{labels} is \strong{not} a \emph{named vector}, its length must equal the value range of \code{x}, i.e. if \code{x} has values from 1 to 3, \code{labels} should have a length of 3;
  \item if length of \code{labels} is intended to differ from length of unique values of \code{x}, a warning is given. You can still add missing labels with the \code{force.labels} or \code{force.values} arguments; see 'Note'.
  \item if \code{labels} \strong{is} a \emph{named vector}, value labels will be set accordingly, even if \code{x} has a different length of unique values. See 'Note' and 'Examples'.
  \item if \code{x} is a data frame, \code{labels} may also be a \code{list} of (named) character vectors;
  \item if \code{labels} is a \code{list}, it must have the same length as number of columns of \code{x};
  \item if \code{labels} is a vector and \code{x} is a data frame, \code{labels} will be applied to each column of \code{x}.
  }
Use \code{labels = ""} to remove labels-attribute from \code{x}.}

\item{force.labels}{Logical; if \code{TRUE}, all \code{labels} are added as value label
attribute, even if \code{x} has less unique values then length of \code{labels}
or if \code{x} has a smaller range then length of \code{labels}. See 'Examples'.
This parameter will be ignored, if \code{labels} is a named vector.}

\item{force.values}{Logical, if \code{TRUE} (default) and \code{labels} has less
elements than unique values of \code{x}, additional values not covered
by \code{labels} will be added as label as well. See 'Examples'.
This parameter will be ignored, if \code{labels} is a named vector.}

\item{drop.na}{Logical, whether existing value labels of tagged NA values
(see \code{\link[haven]{tagged_na}}) should be removed (\code{drop.na = TRUE},
the default) or preserved (\code{drop.na = FALSE}).
See \code{\link{get_na}} for more details on tagged NA values.}
}
\value{
\code{x} with value label attributes; or with removed label-attributes if
          \code{labels = ""}. If \code{x} is a data frame, the complete data
          frame \code{x} will be returned, with removed or added to variables
          specified in \code{...}; if \code{...} is not specified, applies
          to all variables in the data frame.
}
\description{
This function adds labels as attribute (named \code{"labels"})
               to a variable or vector \code{x}, resp. to a set of variables in a
               data frame or a list-object. A use-case is, for instance, the
               \CRANpkg{sjPlot}-package, which supports labelled data and automatically
               assigns labels to axes or legends in plots or to be used in tables.
}
\note{
\itemize{
        \item if \code{labels} is a named vector, \code{force.labels} and \code{force.values} will be ignored, and only values defined in \code{labels} will be labelled;
        \item if \code{x} has less unique values than \code{labels}, redundant labels will be dropped, see \code{force.labels};
        \item if \code{x} has more unique values than \code{labels}, only matching values will be labelled, other values remain unlabelled, see \code{force.values};
        }
        If you only want to change partial value labels, use \code{\link{add_labels}} instead.
        Furthermore, see 'Note' in \code{\link{get_labels}}.
}
\examples{
library(sjmisc)
dummy <- sample(1:4, 40, replace = TRUE)
frq(dummy)

dummy <- set_labels(dummy, labels = c("very low", "low", "mid", "hi"))
frq(dummy)

# assign labels with named vector
dummy <- sample(1:4, 40, replace = TRUE)
dummy <- set_labels(dummy, labels = c("very low" = 1, "very high" = 4))
frq(dummy)

# force using all labels, even if not all labels
# have associated values in vector
x <- c(2, 2, 3, 3, 2)
# only two value labels
x <- set_labels(x, labels = c("1", "2", "3"))
x
frq(x)

# all three value labels
x <- set_labels(x, labels = c("1", "2", "3"), force.labels = TRUE)
x
frq(x)

# create vector
x <- c(1, 2, 3, 2, 4, NA)
# add less labels than values
x <- set_labels(x, labels = c("yes", "maybe", "no"), force.values = FALSE)
x
# add all necessary labels
x <- set_labels(x, labels = c("yes", "maybe", "no"), force.values = TRUE)
x

# set labels and missings
x <- c(1, 1, 1, 2, 2, -2, 3, 3, 3, 3, 3, 9)
x <- set_labels(x, labels = c("Refused", "One", "Two", "Three", "Missing"))
x
set_na(x, na = c(-2, 9))


library(haven)
x <- labelled(c(1:3, tagged_na("a", "c", "z"), 4:1),
              c("Agreement" = 1, "Disagreement" = 4, "First" = tagged_na("c"),
                "Refused" = tagged_na("a"), "Not home" = tagged_na("z")))
# get current NA values
x
get_na(x)
# lose value labels from tagged NA by default, if not specified
set_labels(x, labels = c("New Three" = 3))
# do not drop na
set_labels(x, labels = c("New Three" = 3), drop.na = FALSE)


# set labels via named vector,
# not using all possible values
data(efc)
get_labels(efc$e42dep)

x <- set_labels(
  efc$e42dep,
  labels = c(`independent` = 1,
             `severe dependency` = 2,
             `missing value` = 9)
  )
get_labels(x, include.values = "p")
get_labels(x, include.values = "p", include.non.labelled = TRUE)

# labels can also be set for tagged NA value
# create numeric vector
x <- c(1, 2, 3, 4)
# set 2 and 3 as missing, which will automatically set as
# tagged NA by 'set_na()'
x <- set_na(x, na = c(2, 3))
x
# set label via named vector just for tagged NA(3)
set_labels(x, labels = c(`New Value` = tagged_na("3")))

# setting same value labels to multiple vectors
dummies <- data.frame(
  dummy1 = sample(1:4, 40, replace = TRUE),
  dummy2 = sample(1:4, 40, replace = TRUE),
  dummy3 = sample(1:4, 40, replace = TRUE)
)

# and set same value labels for two of three variables
dummies <- set_labels(
  dummies, dummy1, dummy2,
  labels = c("very low", "low", "mid", "hi")
)
# see result...
get_labels(dummies)

}
\seealso{
See vignette \href{../doc/intro_sjlabelled.html}{Labelled Data and the sjlabelled-Package}
           for more details; \code{\link{set_label}} to manually set variable labels or
           \code{\link{get_label}} to get variable labels; \code{\link{add_labels}} to
           add additional value labels without replacing the existing ones.
}
