% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setupDiagnosticPlot.R
\name{setupDiagnosticPlot}
\alias{setupDiagnosticPlot}
\alias{setupDiagnosticPlot.rlars}
\alias{setupDiagnosticPlot.grplars}
\alias{setupDiagnosticPlot.tslarsP}
\alias{setupDiagnosticPlot.seqModel}
\alias{setupDiagnosticPlot.perrySeqModel}
\alias{setupDiagnosticPlot.tslars}
\alias{setupDiagnosticPlot.sparseLTS}
\alias{setupDiagnosticPlot.perrySparseLTS}
\title{Set up a diagnostic plot for a sequence of regression models}
\usage{
setupDiagnosticPlot(object, ...)

\method{setupDiagnosticPlot}{seqModel}(object, s = NA, covArgs = list(...), ...)

\method{setupDiagnosticPlot}{perrySeqModel}(object, ...)

\method{setupDiagnosticPlot}{tslars}(object, p, ...)

\method{setupDiagnosticPlot}{sparseLTS}(
  object,
  s = NA,
  fit = c("reweighted", "raw", "both"),
  covArgs = list(...),
  ...
)

\method{setupDiagnosticPlot}{perrySparseLTS}(object, ...)
}
\arguments{
\item{object}{the model fit from which to extract information.}

\item{\dots}{additional arguments to be passed to
\code{\link[robustbase]{covMcd}} can be specified directly instead of via
\code{covArgs}.}

\item{s}{for the \code{"seqModel"} method, an integer vector giving the
steps of the submodels from which to extract information (the default is to
use the optimal submodel).  For the \code{"sparseLTS"} method, an integer
vector giving the indices of the models from which to extract information
(the default is to use the optimal model for each of the requested fits).}

\item{covArgs}{a list of arguments to be passed to
\code{\link[robustbase]{covMcd}} for computing robust Mahalanobis distances.}

\item{p}{an integer giving the lag length for which to extract information
(the default is to use the optimal lag length).}

\item{fit}{a character string specifying from which fit to extract
information.  Possible values are \code{"reweighted"} (the default) to
convert the reweighted fit, \code{"raw"} to convert the raw fit, or
\code{"both"} to convert both fits.}
}
\value{
An object of class \code{"setupDiagnosticPlot"} with the following
components:
\describe{
  \item{\code{data}}{a data frame containing the columns listed below.
  \describe{
    \item{\code{step}}{the steps (for the \code{"seqModel"} method) or
    indices (for the \code{"sparseLTS"} method) of the models (only returned
    if more than one model is requested).}
    \item{\code{fit}}{the model fits (only returned if both the reweighted
    and raw fit are requested in the \code{"sparseLTS"} method).}
    \item{\code{index}}{the indices of the observations.}
    \item{\code{fitted}}{the fitted values.}
    \item{\code{residual}}{the standardized residuals.}
    \item{\code{theoretical}}{the corresponding theoretical quantiles from
    the standard normal distribution.}
    \item{\code{qqd}}{the absolute distances from a reference line through
    the first and third sample and theoretical quartiles.}
    \item{\code{rd}}{the robust Mahalanobis distances computed via the MCD
    (see \code{\link[robustbase]{covMcd}}).}
    \item{\code{xyd}}{the pairwise maxima of the absolute values of the
    standardized residuals and the robust Mahalanobis distances, divided by
    the respective other outlier detection cutoff point.}
    \item{\code{weight}}{the weights indicating regression outliers.}
    \item{\code{leverage}}{logicals indicating leverage points (i.e.,
    outliers in the predictor space).}
    \item{\code{Diagnostics}}{a factor with levels \code{"Potential outlier"}
    (potential regression outliers) and \code{"Regular observation"} (data
    points following the model).}
  }}
  \item{\code{qqLine}}{a data frame containing the intercepts and slopes of
  the respective reference lines to be displayed in residual Q-Q plots.}
  \item{\code{q}}{a data frame containing the quantiles of the Mahalanobis
  distribution used as cutoff points for detecting leverage points.}
  \item{\code{facets}}{default faceting formula for the diagnostic plots
  (only returned where applicable).}
}
}
\description{
Extract the fitted values and residuals of a sequence of regression models
(such as robust least angle regression models or sparse least trimmed
squares regression models) and other useful information for diagnostic
plots.
}
\examples{
## generate data
# example is not high-dimensional to keep computation time low
library("mvtnorm")
set.seed(1234)  # for reproducibility
n <- 100  # number of observations
p <- 25   # number of variables
beta <- rep.int(c(1, 0), c(5, p-5))  # coefficients
sigma <- 0.5      # controls signal-to-noise ratio
epsilon <- 0.1    # contamination level
Sigma <- 0.5^t(sapply(1:p, function(i, j) abs(i-j), 1:p))
x <- rmvnorm(n, sigma=Sigma)    # predictor matrix
e <- rnorm(n)                   # error terms
i <- 1:ceiling(epsilon*n)       # observations to be contaminated
e[i] <- e[i] + 5                # vertical outliers
y <- c(x \%*\% beta + sigma * e)  # response
x[i,] <- x[i,] + 5              # bad leverage points


## robust LARS
# fit model
fitRlars <- rlars(x, y, sMax = 10)
# extract information for plotting
setup <- setupDiagnosticPlot(fitRlars)
diagnosticPlot(setup)


## sparse LTS
# fit model
fitSparseLTS <- sparseLTS(x, y, lambda = 0.05, mode = "fraction")
# extract information for plotting
setup1 <- setupDiagnosticPlot(fitSparseLTS)
diagnosticPlot(setup1)
setup2 <- setupDiagnosticPlot(fitSparseLTS, fit = "both")
diagnosticPlot(setup2)
}
\seealso{
\code{\link{diagnosticPlot}}, \code{\link{rlars}},
\code{\link{grplars}}, \code{\link{rgrplars}}, \code{\link{tslarsP}},
\code{\link{rtslarsP}}, \code{\link{tslars}}, \code{\link{rtslars}},
\code{\link{sparseLTS}}
}
\author{
Andreas Alfons
}
\keyword{utilities}
