% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mirai.R
\name{mirai}
\alias{mirai}
\title{mirai (Evaluate Async)}
\usage{
mirai(
  .expr,
  ...,
  .args = list(),
  .timeout = NULL,
  .signal = FALSE,
  .compute = "default"
)
}
\arguments{
\item{.expr}{an expression to evaluate asynchronously (of arbitrary length,
wrapped in \{\} if necessary), \strong{or} a language object passed by
\link{name}.}

\item{...}{(optional) named arguments (name = value pairs) specifying
objects referenced in '.expr'. Used in addition to, and taking precedence
over, any arguments specified via '.args'.}

\item{.args}{(optional) \strong{either} a list of objects to be passed by
\link{name} (found in the current scope), \strong{or else} a list of
name = value pairs, as in '...'.}

\item{.timeout}{[default NULL] for no timeout, or an integer value in
milliseconds. A mirai will resolve to an 'errorValue' 5 (timed out) if
evaluation exceeds this limit.}

\item{.signal}{[default FALSE] logical value, whether to signal the condition
variable within the compute profile (only applicable when daemons have
been set). See \code{\link[nanonext]{cv}} for further information on
condition variables.}

\item{.compute}{[default 'default'] character value for the compute profile
to use when sending the mirai.}
}
\value{
A 'mirai' object.
}
\description{
Evaluate an expression asynchronously in a new background R process or
    persistent daemon (local or remote). This function will return
    immediately with a 'mirai', which will resolve to the evaluated result
    once complete.
}
\details{
This function will return a 'mirai' object immediately.

    The value of a mirai may be accessed at any time at \code{$data}, and
    if yet to resolve, an 'unresolved' logical NA will be returned instead.

    \code{\link{unresolved}} may be used on a mirai, returning TRUE if a
    'mirai' has yet to resolve and FALSE otherwise. This is suitable for use
    in control flow statements such as \code{while} or \code{if}.

    Alternatively, to call (and wait for) the result, use \code{\link{call_mirai}}
    on the returned mirai. This will block until the result is returned
    (although interruptible with e.g. ctrl+c).

    The expression '.expr' will be evaluated in a separate R process in a
    clean environment, which is not the global environment, consisting only
    of the named objects passed as '...' and/or the list supplied to '.args'.

    If an error occurs in evaluation, the error message is returned as a
    character string of class 'miraiError' and 'errorValue'.
    \code{\link{is_mirai_error}} may be used to test for this.

    \code{\link{is_error_value}} tests for all error conditions including
    'mirai' errors, interrupts, and timeouts.

    Specify '.compute' to send the mirai using a specific compute profile (if
    previously created by \code{\link{daemons}}), otherwise leave as 'default'.
}
\examples{
if (interactive()) {
# Only run examples in interactive R sessions

# specifying objects via '...'
n <- 3
m <- mirai(x + y + 2, x = 2, y = n)
m
m$data
Sys.sleep(0.2)
m$data

# passing existing objects by name via '.args'
df1 <- data.frame(a = 1, b = 2)
df2 <- data.frame(a = 3, b = 1)
m <- mirai(as.matrix(rbind(df1, df2)), .args = list(df1, df2), .timeout = 1000)
call_mirai(m)$data

# using unresolved()
m <- mirai(
  {
    res <- rnorm(n)
    res / rev(res)
  },
  n = 1e6
)
while (unresolved(m)) {
  cat("unresolved\n")
  Sys.sleep(0.1)
}
str(m$data)

# evaluating scripts using source(local = TRUE) in '.expr'
n <- 10L
file <- tempfile()
cat("r <- rnorm(n)", file = file)
m <- mirai({source(file, local = TRUE); r}, .args = list(file, n))
call_mirai(m)[["data"]]
unlink(file)

# specifying global variables using list2env(envir = .GlobalEnv) in '.expr'
n <- 10L
file <- tempfile()
cat("r <- rnorm(n)", file = file)
globals <- list(file = file, n = n)
m <- mirai(
  {
    list2env(globals, envir = .GlobalEnv)
    source(file)
    r
  },
  globals = globals
)
call_mirai(m)[["data"]]
unlink(file)

# passing a language object to '.expr' and a named list to '.args'
expr <- quote(a + b + 2)
args <- list(a = 2, b = 3)
m <- mirai(.expr = expr, .args = args)
call_mirai(m)$data

}

}
