#include <stdlib.h>
#include <math.h>
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <R_ext/Arith.h>
#include <R_ext/Applic.h>

/* Multivariate linear binning functions 
   translated from the Fortran code of M. Wand & T.Duong in ks < 1.8.0
   adapted from 1-d massdist.c in stats package */

/* Headers */
void massdist2d(double *x1, double *x2,	int *n, 
		double *a1, double *a2, double *b1, double *b2,
		int *M1, int *M2, double *weight, double *est);

void massdist3d(double *x1, double *x2,	double *x3, int *n, 
		double *a1, double *a2, double *a3, 
		double *b1, double *b2, double *b3,
		int *M1, int *M2, int *M3, double *weight, double *est);

void massdist4d(double *x1, double *x2,	double *x3, double *x4, int *n, 
		double *a1, double *a2, double *a3, double *a4, 
		double *b1, double *b2, double *b3, double *b4, 
		int *M1, int *M2, int *M3, int *M4, double *weight, double *est);


/* Code */
void massdist2d(double *x1, double *x2,	int *n, 
		double *a1, double *a2, double *b1, double *b2,
		int *M1, int *M2, double *weight, double *est)
{
  double fx1, fx2, wi, xdelta1, xdelta2, xpos1, xpos2;   
  int i, ix1, ix2, ixmax1, ixmin1, ixmax2, ixmin2, MM1, MM2;
  
  MM1 = M1[0];
  MM2 = M2[0];
  ixmin1 = 0;
  ixmax1 = MM1 - 2;
  ixmin2 = 0;
  ixmax2 = MM2 - 2;
  xdelta1 = (b1[0] - a1[0]) / (MM1 - 1);
  xdelta2 = (b2[0] - a2[0]) / (MM2 - 1);
 
  // set all est = 0 
  for (i=0; i < MM1*MM2; i++)
    est[i] = 0.0;

  // assign linear binning weights
  for(i=0; i < n[0]; i++) {
    if(R_FINITE(x1[i]) && R_FINITE(x2[i])) {
      xpos1 = (x1[i] - a1[0]) / xdelta1;
      xpos2 = (x2[i] - a2[0]) / xdelta2;
      ix1 = floor(xpos1);
      ix2 = floor(xpos2);
      fx1 = xpos1 - ix1;
      fx2 = xpos2 - ix2;
      wi = weight[i];   
      
      if(ixmin1 <= ix1 && ix1 <= ixmax1 && ixmin2 <= ix2 && ix2 <= ixmax2) {
	est[ix2*MM1 + ix1]         += wi*(1-fx1)*(1-fx2);   
	est[ix2*MM1 + ix1 + 1]     += wi*fx1*(1-fx2);
	est[(ix2+1)*MM1 + ix1]     += wi*(1-fx1)*fx2;   
	est[(ix2+1)*MM1 + ix1 + 1] += wi*fx1*fx2;
      }
    }
  } 
}

void massdist3d(double *x1, double *x2,	double *x3, int *n, 
		double *a1, double *a2, double *a3, 
		double *b1, double *b2, double *b3,
		int *M1, int *M2, int *M3, double *weight, double *est)
{
  double fx1, fx2, fx3, xdelta1, xdelta2, xdelta3, xpos1, xpos2, xpos3, wi;   
  int i, ix1, ix2, ix3, ixmax1, ixmin1, ixmax2, ixmax3, ixmin2, ixmin3, MM1, MM2, MM3;
  
  MM1 = M1[0];
  MM2 = M2[0];
  MM3 = M3[0];
  ixmin1 = 0;
  ixmax1 = MM1 - 2;
  ixmin2 = 0;
  ixmax2 = MM2 - 2;
  ixmin3 = 0;
  ixmax3 = MM3 - 2;
  xdelta1 = (b1[0] - a1[0]) / (MM1 - 1);
  xdelta2 = (b2[0] - a2[0]) / (MM2 - 1);
  xdelta3 = (b3[0] - a3[0]) / (MM3 - 1);
 
  // set all est = 0 
  for (i=0; i < MM1*MM2*MM3; i++)  
    est[i] = 0.0;

  // assign linear binning weights
  for(i=0; i < n[0]; i++) {
    if(R_FINITE(x1[i]) && R_FINITE(x2[i]) && R_FINITE(x3[i])) {
      xpos1 = (x1[i] - a1[0]) / xdelta1;
      xpos2 = (x2[i] - a2[0]) / xdelta2;
      xpos3 = (x3[i] - a3[0]) / xdelta3;
      ix1 = floor(xpos1);
      ix2 = floor(xpos2);
      ix3 = floor(xpos3);
      fx1 = xpos1 - ix1;
      fx2 = xpos2 - ix2;
      fx3 = xpos3 - ix3;
      wi = weight[i];   
      
      if(ixmin1 <= ix1 && ix1 <= ixmax1 && ixmin2 <= ix2 && ix2 <= ixmax2 && ixmin3 <= ix3 && ix3 <= ixmax3) {
	est[ix3*MM1*MM2 + ix2*MM1 + ix1]             += wi*(1-fx1)*(1-fx2)*(1-fx3);   
	est[ix3*MM1*MM2 + ix2*MM1 + ix1 + 1]         += wi*fx1*(1-fx2)*(1-fx3);
	est[ix3*MM1*MM2 + (ix2+1)*MM1 + ix1]         += wi*(1-fx1)*fx2*(1-fx3);   
	est[ix3*MM1*MM2 + (ix2+1)*MM1 + ix1 + 1]     += wi*fx1*fx2*(1-fx3);
	est[(ix3+1)*MM1*MM2 + ix2*MM1 + ix1]         += wi*(1-fx1)*(1-fx2)*fx3;   
	est[(ix3+1)*MM1*MM2 + ix2*MM1 + ix1 + 1]     += wi*fx1*(1-fx2)*fx3;
	est[(ix3+1)*MM1*MM2 + (ix2+1)*MM1 + ix1]     += wi*(1-fx1)*fx2*fx3;   
	est[(ix3+1)*MM1*MM2 + (ix2+1)*MM1 + ix1 + 1] += wi*fx1*fx2*fx3;
      }
    }
  }
}

void massdist4d(double *x1, double *x2,	double *x3, double *x4, int *n, 
		double *a1, double *a2, double *a3, double *a4, 
		double *b1, double *b2, double *b3, double *b4, 
		int *M1, int *M2, int *M3, int *M4, double *weight, double *est)
{
  double fx1, fx2, fx3, fx4, xdelta1, xdelta2, xdelta3, xdelta4, xpos1, xpos2, xpos3, xpos4, wi;   
  int i, ix1, ix2, ix3, ix4, ixmax1, ixmin1, ixmax2, ixmax3, ixmax4, ixmin2, ixmin3, ixmin4, MM1, MM2, MM3, MM4;
  
  MM1 = M1[0];
  MM2 = M2[0];
  MM3 = M3[0];
  MM4 = M4[0];
  ixmin1 = 0;
  ixmax1 = MM1 - 2;
  ixmin2 = 0;
  ixmax2 = MM2 - 2;
  ixmin3 = 0;
  ixmax3 = MM3 - 2;
  ixmin4 = 0;
  ixmax4 = MM4 - 2;
  xdelta1 = (b1[0] - a1[0]) / (MM1 - 1);
  xdelta2 = (b2[0] - a2[0]) / (MM2 - 1);
  xdelta3 = (b3[0] - a3[0]) / (MM3 - 1);
  xdelta4 = (b4[0] - a4[0]) / (MM4 - 1);
 
  // set all est = 0 
  for (i=0; i < MM1*MM2*MM3*MM4; i++)  
    est[i] = 0.0;

  // assign linear binning weights
  for(i=0; i < n[0]; i++) {
    if(R_FINITE(x1[i]) && R_FINITE(x2[i]) && R_FINITE(x3[i]) && R_FINITE(x4[i])) {
      xpos1 = (x1[i] - a1[0]) / xdelta1;
      xpos2 = (x2[i] - a2[0]) / xdelta2;
      xpos3 = (x3[i] - a3[0]) / xdelta3;
      xpos4 = (x4[i] - a4[0]) / xdelta4;
      ix1 = floor(xpos1);
      ix2 = floor(xpos2);
      ix3 = floor(xpos3);
      ix4 = floor(xpos4);
      fx1 = xpos1 - ix1;
      fx2 = xpos2 - ix2;
      fx3 = xpos3 - ix3;
      fx4 = xpos4 - ix4;
      wi = weight[i];   
      
      if(ixmin1 <= ix1 && ix1 <= ixmax1 && ixmin2 <= ix2 && ix2 <= ixmax2 && ixmin3 <= ix3 && ix3 <= ixmax3 && ixmin4 <= ix4 && ix4 <= ixmax4) {
	est[ix4*MM1*MM2*MM3 + ix3*MM1*MM2 + ix2*MM1 + ix1]                 += wi*(1-fx1)*(1-fx2)*(1-fx3)*(1-fx4);   
	est[ix4*MM1*MM2*MM3 + ix3*MM1*MM2 + ix2*MM1 + ix1 + 1]             += wi*fx1*(1-fx2)*(1-fx3)*(1-fx4);
	est[ix4*MM1*MM2*MM3 + ix3*MM1*MM2 + (ix2+1)*MM1 + ix1]             += wi*(1-fx1)*fx2*(1-fx3)*(1-fx4);   
	est[ix4*MM1*MM2*MM3 + ix3*MM1*MM2 + (ix2+1)*MM1 + ix1 + 1]         += wi*fx1*fx2*(1-fx3)*(1-fx4);
	est[ix4*MM1*MM2*MM3 + (ix3+1)*MM1*MM2 + ix2*MM1 + ix1]             += wi*(1-fx1)*(1-fx2)*fx3*(1-fx4);   
	est[ix4*MM1*MM2*MM3 + (ix3+1)*MM1*MM2 + ix2*MM1 + ix1 + 1]         += wi*fx1*(1-fx2)*fx3*(1-fx4);
	est[ix4*MM1*MM2*MM3 + (ix3+1)*MM1*MM2 + (ix2+1)*MM1 + ix1]         += wi*(1-fx1)*fx2*fx3*(1-fx4);   
	est[ix4*MM1*MM2*MM3 + (ix3+1)*MM1*MM2 + (ix2+1)*MM1 + ix1 + 1]     += wi*fx1*fx2*fx3*(1-fx4);
	est[(ix4+1)*MM1*MM2*MM3 + ix3*MM1*MM2 + ix2*MM1 + ix1]             += wi*(1-fx1)*(1-fx2)*(1-fx3)*fx4;   
	est[(ix4+1)*MM1*MM2*MM3 + ix3*MM1*MM2 + ix2*MM1 + ix1 + 1]         += wi*fx1*(1-fx2)*(1-fx3)*fx4;
	est[(ix4+1)*MM1*MM2*MM3 + ix3*MM1*MM2 + (ix2+1)*MM1 + ix1]         += wi*(1-fx1)*fx2*(1-fx3)*fx4;   
	est[(ix4+1)*MM1*MM2*MM3 + ix3*MM1*MM2 + (ix2+1)*MM1 + ix1 + 1]     += wi*fx1*fx2*(1-fx3)*fx4;
	est[(ix4+1)*MM1*MM2*MM3 + (ix3+1)*MM1*MM2 + ix2*MM1 + ix1]         += wi*(1-fx1)*(1-fx2)*fx3*fx4;   
	est[(ix4+1)*MM1*MM2*MM3 + (ix3+1)*MM1*MM2 + ix2*MM1 + ix1 + 1]     += wi*fx1*(1-fx2)*fx3*fx4;
	est[(ix4+1)*MM1*MM2*MM3 + (ix3+1)*MM1*MM2 + (ix2+1)*MM1 + ix1]     += wi*(1-fx1)*fx2*fx3*fx4;   
	est[(ix4+1)*MM1*MM2*MM3 + (ix3+1)*MM1*MM2 + (ix2+1)*MM1 + ix1 + 1] += wi*fx1*fx2*fx3*fx4;
      }
    }
  }
}
