defpar <- par()[setdiff(names(par()),c("cin","cra","csi","cxy","din","page"))]

# lag weights (internal use only)
lagwei <- function(theta,lag,type) {
  res <- c()
  xa <- 0  #####  
  for(i in 1:length(lag)) {
    if(type=="quec.lag") {
      if(lag[i]<theta[1]-1 | lag[i]>theta[2]+1) {
        res[i] <- 0
        } else {
        res[i] <- -4/(theta[2]-theta[1]+2)^2*(lag[i]^2-(theta[1]+theta[2])*lag[i]+(theta[1]-1)*(theta[2]+1))
        }
      } else if(type=="qdec.lag") {
      if(lag[i]<theta[1] | lag[i]>theta[2]+1) {
        res[i] <- 0
        } else {
        res[i] <- (lag[i]^2-2*(theta[2]+1)*lag[i]+(theta[2]+1)^2)/(theta[2]-theta[1]+1)^2
        }
      } else if(type=="gamm.lag") {
      if(lag[i]>=xa) {
        bnum <- (lag[i]-xa+1)^(theta[1]/(1-theta[1]))*theta[2]^(lag[i]-xa)
        xM <- (theta[1]/(theta[1]-1))/log(theta[2])+xa-1
        bden <- (xM-xa+1)^(theta[1]/(1-theta[1]))*theta[2]^(xM-xa)
        res[i] <- bnum/bden
        } else {
        res[i] <- 0   
        }
      }
    }
  names(res) <- lag
  if(type=="gamm.lag") {
    if(sum(res,na.rm=T)==0) res[1] <- 1
    }
  res
  }

# generate lagged instances of a variable (internal use only)
genLag <- function(x,maxlag) {
  if(maxlag>0) {
    out <- x
    for(w in 1:maxlag) {
      if(w<length(x)) {
        wx <- c(rep(NA,w),x[1:(length(x)-w)])
        } else {
        wx <- rep(NA,length(x))
        }
      out <- cbind(out,wx)        
      }
    colnames(out) <- NULL
    out
    } else {
    x
    }
  }

# distributed-lag transformation (internal use only)
Zmat <- function(x,type,theta) {
  if(type=="none") {
    matrix(x,ncol=1)
    } else {
    if(type=="gamm.lag") {
      xa <- 0  #####
      lagp <- gamlim(theta[1],theta[2],xa)
      laglim <- lagp[2]
      } else {
      laglim <- theta[2]
      }
    H <- lagwei(theta,0:laglim,type)
    as.numeric(genLag(x,laglim)%*%matrix(H,ncol=1))
    }
  }

# quec lag transformation
quec.lag <- function(x,a,b,x.group=NULL) {
  if(!identical(a,round(a)) || a<0) stop("Argument 'a' must be a non-negative integer value")
  if(!identical(b,round(b)) || b<0) stop("Argument 'b' must be a non-negative integer value")
  if(a>b) stop("Argument 'a' must be no greater than argument 'b'")
  if(is.null(x.group)) {
    Zmat(x,"quec.lag",c(a,b))
    } else {
    res <- c()  
    gruppi <- levels(factor(x.group))
    for(i in 1:length(gruppi)) {
      auxind <- which(x.group==gruppi[i])
      res[auxind] <- Zmat(x[auxind],"quec.lag",c(a,b))
      }
    res
    }
  }

# qdec lag transformation
qdec.lag <- function(x,a,b,x.group=NULL) {
  if(!identical(a,round(a)) || a<0) stop("Argument 'a' must be a non-negative integer value")
  if(!identical(b,round(b)) || b<0) stop("Argument 'b' must be a non-negative integer value")
  if(a>b) stop("Argument 'a' must be no greater than argument 'b'")
  if(is.null(x.group)) {
    Zmat(x,"qdec.lag",c(a,b))
    } else {
    res <- c()  
    gruppi <- levels(factor(x.group))
    for(i in 1:length(gruppi)) {
      auxind <- which(x.group==gruppi[i])
      res[auxind] <- Zmat(x[auxind],"qdec.lag",c(a,b))
      }
    res
    }
  }

# gamma lag transformation
gamm.lag <- function(x,delta,lambda,x.group=NULL) {
  if(delta<=0 || delta>=1) stop("Argument 'delta' must be a value in the interval (0,1)")
  if(lambda<=0 || lambda>=1) stop("Argument 'lambda' must be a value in the interval (0,1)")
  if(is.null(x.group)) {
    Zmat(x,"gamm.lag",c(delta,lambda))
    } else {
    res <- c()
    gruppi <- levels(factor(x.group))
    for(i in 1:length(gruppi)) {
      auxind <- which(x.group==gruppi[i])
      res[auxind] <- Zmat(x[auxind],"gamm.lag",c(delta,lambda))
      }
    res
    }
  }

# almon lag transformation
almon.lag <- function(x,a,b,p,x.group=NULL) {
  if(!identical(a,round(a)) || a<0) stop("Argument 'a' must be a non-negative integer value")
  if(!identical(b,round(b)) || b<0) stop("Argument 'b' must be a non-negative integer value")
  if(!identical(p,round(p)) || p<0) stop("Argument 'p' must be a non-negative integer value")
  if(a>b) stop("Argument 'a' must be no greater than argument 'b'")
  if(a==0 & b==0) {
    res <- matrix(x,ncol=1)
    } else {
    if(is.null(x.group)) {
      res <- matrix(genLag(x,b)[,(a+1):(b+1)],ncol=b-a+1)
      } else {
      res <- matrix(nrow=length(x),ncol=b-a+1) 
      gruppi <- levels(factor(x.group))
      for(i in 1:length(gruppi)) {
        auxind <- which(x.group==gruppi[i])
        res[auxind,] <- genLag(x[auxind],b)[,(a+1):(b+1)]
        }
      }
    colnames(res) <- a:b
    wei <- matrix(1,ncol=1,nrow=ncol(res))
    if(p>0) {
      for(i in 1:p) {
        wei <- cbind(wei,(0:(ncol(res)-1))^i)
        }
      }
    res%*%wei
    }
  }

# unconstrained lag transformation
uncons.lag <- function(x,a,b,x.group=NULL) {
  if(!identical(a,round(a)) || a<0) stop("Argument 'a' must be a non-negative integer value")
  if(!identical(b,round(b)) || b<0) stop("Argument 'b' must be a non-negative integer value")
  if(a>b) stop("Argument 'a' must be no greater than argument 'b'")
  if(a==0 & b==0) {
    res <- matrix(x,ncol=1)
    } else {
    if(is.null(x.group)) {
      res <- matrix(genLag(x,b)[,(a+1):(b+1)],ncol=b-a+1)
      } else {
      res <- matrix(nrow=length(x),ncol=b-a+1) 
      gruppi <- levels(factor(x.group))
      for(i in 1:length(gruppi)) {
        auxind <- which(x.group==gruppi[i])
        res[auxind,] <- genLag(x[auxind],b)[,(a+1):(b+1)]
        }
      }
    colnames(res) <- a:b
    res
    }
  }

# check if a variable is quantitative (internal use only)
isQuant <- function(x) {
  if(is.numeric(x)) {
    T
    } else {
    if(!is.factor(x) && sum(!is.na(x))==0) {
      T
      } else {
      F
      }
    }
  }

# compute the lag limit (internal use only)
findLagLim <- function(data,group=NULL) {
  if(is.null(group)) {
    nrow(na.omit(data))
    } else {
    auxlaglim <- c()
    gruppi <- levels(factor(data[,group]))
    for(i in 1:length(gruppi)) {
      auxlaglim[i] <- nrow(na.omit(data[which(data[,group]==gruppi[i]),]))
      }
    min(auxlaglim,na.rm=T)
    }
  }

# scan formula (internal use only)
scanForm <- function(x) {
  auxform <- gsub(" ","",formula(x))[-1]
  ynam <- gsub(" ","",auxform[1])
  auX <- gsub(" ","",strsplit(auxform[2],"\\+")[[1]])
  auxopen <- grep("\\(",auX)
  auxclos <- grep("\\)",auX)
  x2merge <- intersect(auxopen,setdiff(1:length(auX),auxclos))
  if(length(x2merge)>0) {
    x2del <- c()
    for(i in 1:length(x2merge)) {
      auX[x2merge[i]] <- paste(auX[x2merge[i]],"+",auX[x2merge[i]+1],sep="")
      x2del <- c(x2del,x2merge[i]+1)
      }
    auX <- auX[setdiff(1:length(auX),x2del)]
    }
  if(sum(!is.na(auX)==0)) auX <- "1"
  auX <- setdiff(auX,c("1","-1"))
  if(length(auX)>0) {
    lnames <- ltype <- rep(NA,length(auX))
    lpar <- list()
    for(i in 1:length(auX)) {        
      if(nchar(auX[i])>0) {
        if(identical("quec.lag(",substr(auX[i],1,9))) {                           
          istr <- gsub("quec\\.lag\\(","",strsplit(auX[i],",")[[1]])
          lnames[i] <- istr[1]
          ltype[i] <- "quec.lag"
          lpar[[i]] <- as.numeric(gsub(")","",istr[2:3]))
          } else if(identical("qdec.lag(",substr(auX[i],1,9))) {                            
          istr <- gsub("qdec\\.lag\\(","",strsplit(auX[i],",")[[1]])
          lnames[i] <- istr[1]
          ltype[i] <- "qdec.lag"
          lpar[[i]] <- as.numeric(gsub(")","",istr[2:3]))             
          } else if(identical("gamm.lag(",substr(auX[i],1,9))) {
          istr <- gsub("gamm\\.lag\\(","",strsplit(auX[i],",")[[1]])
          lnames[i] <- istr[1]
          ltype[i] <- "gamm.lag"
          lpar[[i]] <- as.numeric(gsub(")","",istr[2:3]))             
          } else {
          lnames[i] <- auX[i]
          ltype[i] <- "none"
          lpar[[i]] <- NA
          }
        }    
      }
    names(lpar) <- names(ltype) <- lnames
    } else {
    lpar <- ltype <- c()
    }
  list(y=ynam,X=auX,ltype=ltype,lpar=lpar)
  }

# limits of gamma lag shapes (internal use only)
gamlim <- function(delta,lambda,xa=0) {
  auxmax <- ceiling(qgamma(0.99,1/(1-delta),-log(lambda))+xa)
  xgrid <- 0:auxmax
  b <- lagwei(c(delta,lambda),xgrid,"gamm.lag")
  auxb <- b/(max(b))
  auxind <- which(abs(auxb)>1e-8)
  if(length(auxind)>0) {
    range(xgrid[auxind])
    } else {
    c(0,0)
    }
  #c(floor(qgamma(0.01,1/(1-delta),-log(lambda))+xa),ceiling(qgamma(0.99,1/(1-delta),-log(lambda))+xa))
  }

# gamma table (internal use only)
gammaTabFun <- function() {
  res <- matrix(c(0.01,0.01,0,2,0.01,0.1,0,3,0.01,0.22,0,4,0.01,0.32,0,5,0.01,0.4,0,6,0.01,0.47,0,7,0.01,0.52,0,8,0.01,0.57,0,9,0.01,0.6,0,10,0.01,0.63,0,11,0.01,0.66,0,12,0.01,0.68,0,13,0.01,0.71,0,14,0.01,0.72,0,15,0.01,0.74,0,16,0.01,0.75,0,17,0.01,0.77,0,18,0.01,0.78,0,19,0.01,0.79,0,20,0.01,0.8,0,21,0.01,0.81,0,22,0.01,0.82,0,24,0.01,0.83,0,25,0.01,0.84,0,27,0.01,0.85,0,29,0.01,0.86,0,31,0.01,0.87,0,34,0.01,0.88,0,37,0.01,0.89,0,40,0.01,0.9,0,44,
    0.01,0.91,0,50,0.01,0.92,0,56,0.01,0.93,0,64,0.01,0.94,0,75,0.01,0.95,0,91,0.01,0.96,0,114,0.01,0.97,0,152,0.01,0.98,0,230,0.01,0.99,0,461,0.02,0.81,0,23,0.02,0.9,0,45,0.02,0.93,0,65,0.02,0.94,0,76,0.02,0.97,0,153,0.02,0.98,0,231,0.02,0.99,0,463,0.03,0.83,0,26,0.03,0.86,0,32,0.03,0.89,0,41,0.03,0.92,0,57,0.03,0.95,0,92,0.03,0.96,0,115,0.03,0.97,0,154,0.03,0.98,0,232,0.03,0.99,0,466,0.04,0.96,0,116,0.04,0.97,0,155,0.04,0.98,0,233,0.04,0.99,0,468,0.05,0.84,0,28,
    0.05,0.85,0,30,0.05,0.91,0,51,0.05,0.93,0,66,0.05,0.94,0,77,0.05,0.95,0,93,0.05,0.97,0,156,0.05,0.98,0,234,0.05,0.99,0,471,0.06,0.87,0,35,0.06,0.88,0,38,0.06,0.9,0,46,0.06,0.96,0,117,0.06,0.98,0,236,0.06,0.99,0,473,0.07,0.92,0,58,0.07,0.94,0,78,0.07,0.95,0,94,0.07,0.97,0,157,0.07,0.98,0,237,0.07,0.99,0,476,0.08,0.89,0,42,0.08,0.93,0,67,0.08,0.96,0,118,0.08,0.97,0,158,0.08,0.98,0,238,0.08,0.99,0,478,0.09,0.86,0,33,0.09,0.91,0,52,0.09,0.94,0,79,0.09,0.95,0,95,
    0.09,0.96,0,119,0.09,0.97,0,159,0.09,0.98,0,239,0.09,0.99,0,481,0.1,0.9,0,47,0.1,0.92,0,59,0.1,0.97,0,160,0.1,0.98,0,241,0.1,0.99,0,484,0.11,0.87,0,36,0.11,0.88,0,39,0.11,0.93,0,68,0.11,0.95,0,96,0.11,0.96,0,120,0.11,0.97,0,161,0.11,0.98,0,242,0.11,0.99,0,486,0.12,0.89,0,43,0.12,0.91,0,53,0.12,0.94,0,80,0.12,0.96,0,121,0.12,0.97,0,162,0.12,0.98,0,244,0.12,0.99,0,489,0.13,0.92,0,60,0.13,0.93,0,69,0.13,0.95,0,97,0.13,0.97,0,163,0.13,0.98,0,245,0.13,0.99,0,492,
    0.14,0.9,0,48,0.14,0.94,0,81,0.14,0.96,0,122,0.14,0.97,0,164,0.14,0.98,0,246,0.14,0.99,0,495,0.15,0.95,0,98,0.15,0.96,0,123,0.15,0.97,0,165,0.15,0.98,0,248,0.15,0.99,0,498,0.16,0.91,0,54,0.16,0.92,0,61,0.16,0.93,0,70,0.16,0.94,0,82,0.16,0.95,0,99,0.16,0.96,0,124,0.16,0.97,0,166,0.16,0.98,0,249,0.16,0.99,0,501,0.17,0.9,0,49,0.17,0.97,0,167,0.17,0.98,0,251,0.17,0.99,0,504,0.18,0.92,0,62,0.18,0.93,0,71,0.18,0.94,0,83,0.18,0.95,0,100,0.18,0.96,0,125,0.18,0.97,0,168,
    0.18,0.98,0,252,0.18,0.99,0,507,0.19,0.91,0,55,0.19,0.96,0,126,0.19,0.97,0,169,0.19,0.98,0,254,0.19,0.99,0,510,0.2,0.93,0,72,0.2,0.94,0,84,0.2,0.95,0,101,0.2,0.96,0,127,0.2,0.97,0,170,0.2,0.98,0,256,0.2,0.99,0,513,0.21,0.92,0,63,0.21,0.95,0,102,0.21,0.96,0,128,0.21,0.97,0,171,0.21,0.98,0,257,0.21,0.99,0,517,0.22,0.9,0,50,0.22,0.91,0,56,0.22,0.94,0,85,0.22,0.97,0,172,0.22,0.98,0,259,0.22,0.99,0,520,0.23,0.92,0,64,0.23,0.93,0,73,0.23,0.95,0,103,0.23,0.96,0,129,
    0.23,0.97,0,173,0.23,0.98,0,261,0.23,0.99,0,524,0.24,0.94,0,86,0.24,0.95,0,104,0.24,0.96,0,130,0.24,0.97,0,174,0.24,0.98,0,263,0.24,0.99,0,527,0.25,0.91,0,57,0.25,0.93,0,74,0.25,0.94,0,87,0.25,0.96,0,131,0.25,0.97,0,176,0.25,0.98,0,264,0.25,0.99,0,531,0.26,0.92,0,65,0.26,0.95,0,105,0.26,0.96,0,132,0.26,0.97,0,177,0.26,0.98,0,266,0.26,0.99,0,535,0.27,0.93,0,75,0.27,0.94,0,88,0.27,0.95,0,106,0.27,0.96,0,133,0.27,0.97,0,178,0.27,0.98,0,268,0.27,0.99,0,538,0.28,0.85,0,34,
    0.28,0.86,0,37,0.28,0.87,0,40,0.28,0.92,0,66,0.28,0.93,0,76,0.28,0.94,0,89,0.28,0.95,0,107,0.28,0.96,0,134,0.28,0.97,0,179,0.28,0.98,0,270,0.28,0.99,0,542,0.29,0.96,0,135,0.29,0.97,0,181,0.29,0.98,0,272,0.29,0.99,0,546,0.3,0.88,0,44,0.3,0.92,0,67,0.3,0.93,0,77,0.3,0.94,0,90,0.3,0.95,0,108,0.3,0.96,0,136,0.3,0.97,0,182,0.3,0.98,0,274,0.3,0.99,0,550,0.31,0.94,0,91,0.31,0.95,0,109,0.31,0.96,0,137,0.31,0.97,0,183,0.31,0.98,0,276,0.31,0.99,0,555,0.32,0.82,0,29,
    0.32,0.83,0,31,0.32,0.87,0,41,0.32,0.92,0,68,0.32,0.93,0,78,0.32,0.95,0,110,0.32,0.96,0,138,0.32,0.97,0,185,0.32,0.98,0,278,0.32,0.99,0,559,0.33,0.81,0,27,0.33,0.88,0,45,0.33,0.94,0,92,0.33,0.95,0,111,0.33,0.96,0,139,0.33,0.97,0,186,0.33,0.98,0,281,0.33,0.99,0,563,0.34,0.86,0,38,0.34,0.92,0,69,0.34,0.93,0,79,0.34,0.94,0,93,0.34,0.95,0,112,0.34,0.96,0,140,0.34,0.97,0,188,0.34,0.98,0,283,0.34,0.99,0,568,0.35,0.87,0,42,0.35,0.93,0,80,0.35,0.95,0,113,0.35,0.96,0,141,
    0.35,0.97,0,189,0.35,0.98,0,285,0.35,0.99,0,573,0.36,0.78,0,24,0.36,0.79,0,25,0.36,0.8,0,26,0.36,0.83,0,32,0.36,0.88,0,46,0.36,0.92,0,70,0.36,0.94,0,94,0.36,0.95,0,114,0.36,0.96,0,143,0.36,0.97,0,191,0.36,0.98,0,287,0.36,0.99,0,577,0.37,0.81,0,28,0.37,0.82,0,30,0.37,0.89,0,51,0.37,0.92,0,71,0.37,0.93,0,81,0.37,0.94,0,95,0.37,0.96,0,144,0.37,0.97,0,192,0.37,0.98,0,290,0.37,0.99,0,582,0.38,0.76,0,22,0.38,0.77,0,23,0.38,0.87,0,43,0.38,0.88,0,47,0.38,0.93,0,82,
    0.38,0.94,0,96,0.38,0.95,0,115,0.38,0.96,0,145,0.38,0.97,0,194,0.38,0.98,0,292,0.38,0.99,0,587,0.39,0.74,0,20,0.39,0.75,0,21,0.39,0.84,0,35,0.39,0.89,0,52,0.39,0.92,0,72,0.39,0.94,0,97,0.39,0.95,0,116,0.39,0.96,0,146,0.39,0.97,0,196,0.39,0.98,0,295,0.39,0.99,0,592,0.4,0.83,0,33,0.4,0.93,0,83,0.4,0.94,0,98,0.4,0.95,0,118,0.4,0.96,0,148,0.4,0.97,0,198,0.4,0.98,0,298,0.4,0.99,0,598,0.41,0.7,0,17,0.41,0.72,0,19,0.41,0.88,0,48,0.41,0.9,0,58,0.41,0.92,0,73,
    0.41,0.93,0,84,0.41,0.95,0,119,0.41,0.96,0,149,0.41,0.97,0,199,0.41,0.98,0,300,0.41,0.99,0,603,0.42,0.71,0,18,0.42,0.84,0,36,0.42,0.89,0,53,0.42,0.9,0,59,0.42,0.93,0,85,0.42,0.94,0,99,0.42,0.95,0,120,0.42,0.96,0,150,0.42,0.97,0,201,0.42,0.98,0,303,0.42,0.99,0,609,0.43,0.88,0,49,0.43,0.93,0,86,0.43,0.94,0,100,0.43,0.95,0,121,0.43,0.96,0,152,0.43,0.97,0,203,0.43,0.98,0,306,0.43,0.99,0,615,0.44,0.85,0,39,0.44,0.89,0,54,0.44,0.9,0,60,0.44,0.92,0,75,0.44,0.94,0,101,
    0.44,0.95,0,122,0.44,0.96,0,153,0.44,0.97,0,205,0.44,0.98,0,309,0.44,0.99,0,621,0.45,0.65,0,15,0.45,0.67,0,16,0.45,0.88,0,50,0.45,0.89,0,55,0.45,0.92,0,76,0.45,0.93,0,87,0.45,0.94,0,102,0.45,0.95,0,123,0.45,0.96,0,155,0.45,0.97,0,207,0.45,0.98,0,312,0.45,0.99,0,627,0.46,0.9,0,61,0.46,0.92,0,77,0.46,0.93,0,88,0.46,0.94,0,103,0.46,0.95,0,124,0.46,0.96,0,156,0.46,0.97,0,209,0.46,0.98,0,315,0.46,0.99,0,633,0.47,0.63,0,14,0.47,0.89,0,56,0.47,0.93,0,89,0.47,0.94,0,104,
    0.47,0.95,0,126,0.47,0.96,0,158,0.47,0.97,0,211,0.47,0.98,0,319,0.47,0.99,0,640,0.48,0.6,0,13,0.48,0.9,0,62,0.48,0.92,0,78,0.48,0.93,0,90,0.48,0.94,0,105,0.48,0.95,0,127,0.48,0.96,0,160,0.48,0.97,0,214,0.48,0.98,0,322,0.48,0.99,0,647,0.49,0.89,0,57,0.49,0.9,0,63,0.49,0.92,0,79,0.49,0.93,0,91,0.49,0.94,0,107,0.49,0.95,0,128,0.49,0.96,0,161,0.49,0.97,0,216,0.49,0.98,0,325,0.49,0.99,0,654,0.5,0.57,0,12,0.5,0.9,0,64,0.5,0.92,0,80,0.5,0.93,0,92,0.5,0.94,0,108,
    0.5,0.95,0,130,0.5,0.96,0,163,0.5,0.97,0,218,0.5,0.98,0,329,0.5,0.99,0,661,0.51,0.92,0,81,0.51,0.93,0,93,0.51,0.94,0,109,0.51,0.95,0,131,0.51,0.96,0,165,0.51,0.97,0,221,0.51,0.98,0,333,0.51,0.99,0,669,0.52,0.83,0,37,0.52,0.9,0,65,0.52,0.92,0,82,0.52,0.93,0,94,0.52,0.94,0,110,0.52,0.95,0,133,0.52,0.96,0,167,0.52,0.97,0,224,0.52,0.98,0,337,0.52,0.99,0,676,0.53,0.53,0,11,0.53,0.84,0,40,0.53,0.9,0,66,0.53,0.93,0,95,0.53,0.94,0,112,0.53,0.95,0,135,0.53,0.96,0,169,
    0.53,0.97,0,226,0.53,0.98,0,341,0.53,0.99,0,684,0.54,0.81,0,34,0.54,0.9,0,67,0.54,0.91,0,74,0.54,0.92,0,84,0.54,0.93,0,96,0.54,0.94,0,113,0.54,0.95,0,136,0.54,0.96,0,171,0.54,0.97,0,229,0.54,0.98,0,345,0.54,0.99,0,693,0.55,0.49,0,10,0.55,0.83,0,38,0.55,0.84,0,41,0.55,0.85,0,44,0.55,0.87,0,51,0.55,0.92,0,85,0.55,0.93,0,98,0.55,0.94,0,114,0.55,0.95,0,138,0.55,0.96,0,173,0.55,0.97,0,232,0.55,0.98,0,349,0.55,0.99,0,702,0.56,0.78,0,29,0.56,0.79,0,31,0.56,0.87,0,52,
    0.56,0.9,0,68,0.56,0.92,0,86,0.56,0.93,0,99,0.56,0.94,0,116,0.56,0.95,0,140,0.56,0.96,0,175,0.56,0.97,0,235,0.56,0.98,0,354,0.56,0.99,0,711,0.57,0.84,0,42,0.57,0.85,0,45,0.57,0.9,0,69,0.57,0.92,0,87,0.57,0.93,0,100,0.57,0.94,0,117,0.57,0.95,0,142,0.57,0.96,0,178,0.57,0.97,0,238,0.57,0.98,0,359,0.57,0.99,0,720,0.58,0.76,0,27,0.58,0.79,0,32,0.58,0.84,0,43,0.58,0.85,0,46,0.58,0.87,0,53,0.58,0.88,0,58,0.58,0.9,0,70,0.58,0.92,0,88,0.58,0.93,0,102,0.58,0.94,0,119,
    0.58,0.95,0,143,0.58,0.96,0,180,0.58,0.97,0,241,0.58,0.98,0,363,0.58,0.99,0,730,0.59,0.78,0,30,0.59,0.88,0,59,0.59,0.9,0,71,0.59,0.93,0,103,0.59,0.94,0,121,0.59,0.95,0,145,0.59,0.96,0,183,0.59,0.97,0,245,0.59,0.98,0,369,0.59,0.99,0,740,0.6,0.43,0,9,0.6,0.73,0,24,0.6,0.81,0,36,0.6,0.85,0,47,0.6,0.87,0,55,0.6,0.88,0,60,0.6,0.9,0,72,0.6,0.92,0,91,0.6,0.93,0,104,0.6,0.94,0,122,0.6,0.95,0,148,0.6,0.96,0,185,0.6,0.97,0,248,0.6,0.98,0,374,0.6,0.99,0,751,
    0.61,0.73,0,25,0.61,0.74,0,26,0.61,0.76,0,28,0.61,0.79,0,33,0.61,0.8,0,35,0.61,0.85,0,48,0.61,0.9,0,73,0.61,0.92,0,92,0.61,0.93,0,106,0.61,0.94,0,124,0.61,0.95,0,150,0.61,0.96,0,188,0.61,0.97,0,252,0.61,0.98,0,379,0.61,0.99,0,762,0.62,0.87,0,56,0.62,0.88,0,61,0.62,0.91,0,83,0.62,0.93,0,108,0.62,0.94,0,126,0.62,0.95,0,152,0.62,0.96,0,191,0.62,0.97,0,256,0.62,0.98,0,385,0.62,0.99,0,774,0.63,0.37,0,8,0.63,0.85,0,49,0.63,0.87,0,57,0.63,0.88,0,62,0.63,0.9,0,75,
    0.63,0.92,0,95,0.63,0.93,0,109,0.63,0.94,0,128,0.63,0.95,0,154,0.63,0.96,0,194,0.63,0.97,0,260,0.63,0.98,0,391,0.63,0.99,0,786,0.64,0.68,0,21,0.64,0.69,0,22,0.64,0.7,0,23,0.64,0.81,0,39,0.64,0.85,0,50,0.64,0.86,0,54,0.64,0.88,0,63,0.64,0.9,0,77,0.64,0.92,0,97,0.64,0.93,0,111,0.64,0.94,0,130,0.64,0.95,0,157,0.64,0.96,0,197,0.64,0.97,0,264,0.64,0.98,0,398,0.64,0.99,0,799,0.65,0.66,0,20,0.65,0.88,0,64,0.65,0.9,0,78,0.65,0.92,0,98,0.65,0.93,0,113,0.65,0.94,0,132,
    0.65,0.95,0,160,0.65,0.96,0,200,0.65,0.97,0,268,0.65,0.98,0,405,0.65,0.99,0,813,0.66,0.64,0,19,0.66,0.88,0,65,0.66,0.9,0,79,0.66,0.91,0,89,0.66,0.92,0,100,0.66,0.93,0,115,0.66,0.94,0,135,0.66,0.95,0,162,0.66,0.96,0,204,0.66,0.97,0,273,0.66,0.98,0,412,0.66,0.99,0,827,0.67,0.88,0,67,0.67,0.9,0,81,0.67,0.91,0,90,0.67,0.92,0,102,0.67,0.93,0,117,0.67,0.94,0,137,0.67,0.95,0,165,0.67,0.96,0,208,0.67,0.97,0,278,0.67,0.98,0,419,0.67,0.99,0,842,0.68,0.29,0,7,0.68,0.6,0,17,
    0.68,0.79,0,37,0.68,0.88,0,68,0.68,0.89,0,74,0.68,0.9,0,82,0.68,0.93,0,119,0.68,0.94,0,140,0.68,0.95,0,168,0.68,0.96,0,212,0.68,0.97,0,283,0.68,0.98,0,427,0.68,0.99,0,858,0.69,0.6,0,18,0.69,0.77,0,34,0.69,0.8,0,40,0.69,0.84,0,51,0.69,0.88,0,69,0.69,0.89,0,76,0.69,0.9,0,84,0.69,0.91,0,94,0.69,0.92,0,106,0.69,0.93,0,122,0.69,0.94,0,142,0.69,0.95,0,172,0.69,0.96,0,216,0.69,0.97,0,289,0.69,0.98,0,435,0.69,0.99,0,875,0.7,0.55,0,15,0.7,0.57,0,16,0.7,0.8,0,41,
    0.7,0.84,0,52,0.7,0.88,0,71,0.7,0.9,0,86,0.7,0.91,0,96,0.7,0.92,0,108,0.7,0.93,0,124,0.7,0.94,0,145,0.7,0.95,0,175,0.7,0.96,0,220,0.7,0.97,0,295,0.7,0.98,0,444,0.7,0.99,0,892,0.71,0.74,0,31,0.71,0.8,0,42,0.71,0.81,0,44,0.71,0.84,0,53,0.71,0.87,0,66,0.71,0.88,0,72,0.71,0.9,0,87,0.71,0.92,0,110,0.71,0.93,0,127,0.71,0.94,0,148,0.71,0.95,0,179,0.71,0.96,0,225,0.71,0.97,0,301,0.71,0.98,0,454,0.71,0.99,0,911,0.72,0.21,0,6,0.72,0.72,0,29,0.72,0.78,0,38,
    0.72,0.81,0,45,0.72,0.85,0,58,0.72,0.92,0,113,0.72,0.93,0,129,0.72,0.94,0,152,0.72,0.95,0,183,0.72,0.96,0,230,0.72,0.97,0,308,0.72,0.98,0,464,0.72,0.99,0,931,0.73,0.5,0,14,0.73,0.74,0,32,0.73,0.8,0,43,0.73,0.81,0,46,0.73,0.84,0,55,0.73,0.85,0,59,0.73,0.88,0,75,0.73,0.89,0,83,0.73,0.9,0,91,0.73,0.92,0,115,0.73,0.93,0,132,0.73,0.94,0,155,0.73,0.95,0,187,0.73,0.96,0,235,0.73,0.97,0,315,0.73,0.98,0,474,0.73,0.99,0,953,0.74,0.47,0,13,0.74,0.69,0,27,0.74,0.72,0,30,
    0.74,0.74,0,33,0.74,0.75,0,35,0.74,0.76,0,36,0.74,0.81,0,47,0.74,0.84,0,57,0.74,0.85,0,61,0.74,0.88,0,77,0.74,0.89,0,85,0.74,0.91,0,104,0.74,0.92,0,118,0.74,0.93,0,136,0.74,0.94,0,159,0.74,0.95,0,192,0.74,0.96,0,241,0.74,0.97,0,322,0.74,0.98,0,486,0.74,0.99,0,975,0.75,0.77,0,39,0.75,0.81,0,48,0.75,0.83,0,54,0.75,0.85,0,62,0.75,0.86,0,67,0.75,0.87,0,73,0.75,0.88,0,79,0.75,0.91,0,107,0.75,0.92,0,121,0.75,0.93,0,139,0.75,0.94,0,163,0.75,0.95,0,196,0.75,0.96,0,247,
    0.75,0.97,0,330,0.75,0.98,0,498,0.75,0.99,0,1000,0.76,0.42,0,12,0.76,0.65,0,24,0.76,0.66,0,25,0.76,0.67,0,26,0.76,0.69,0,28,0.76,0.81,0,49,0.76,0.83,0,56,0.76,0.84,0,60,0.76,0.85,0,64,0.76,0.88,0,81,0.76,0.89,0,89,0.76,0.9,0,98,0.76,0.92,0,124,0.76,0.93,0,143,0.76,0.94,0,167,0.76,0.95,0,201,0.76,0.96,0,253,0.76,0.97,0,339,0.76,0.98,0,511,0.76,0.99,0,1026,0.77,0.9,0,101,0.77,0.92,0,128,0.77,0.93,0,146,0.77,0.94,0,172,0.77,0.95,0,207,0.77,0.96,0,260,0.77,0.97,0,348,
    0.77,0.98,0,525,0.77,0.99,0,1055,0.78,0.37,0,11,0.78,0.62,0,23,0.78,0.84,0,63,0.78,0.85,0,68,0.78,0.88,0,86,0.78,0.89,0,94,0.78,0.91,0,116,0.78,0.92,0,131,0.78,0.93,0,151,0.78,0.94,0,177,0.78,0.95,0,213,0.78,0.96,0,268,0.78,0.97,0,358,0.78,0.98,0,540,0.78,0.99,0,1085,0.79,0.57,0,20,0.79,0.58,0,21,0.79,0.59,0,22,0.79,0.75,0,40,0.79,0.76,0,41,0.79,0.8,0,51,0.79,0.84,0,65,0.79,0.85,0,70,0.79,0.88,0,88,0.79,0.89,0,97,0.79,0.91,0,120,0.79,0.92,0,135,0.79,0.93,0,155,
    0.79,0.94,0,182,0.79,0.95,0,220,0.79,0.96,0,276,0.79,0.97,0,369,0.79,0.98,0,557,0.79,0.99,1,1119,0.8,0.31,0,10,0.8,0.54,0,19,0.8,0.71,0,34,0.8,0.73,0,37,0.8,0.79,0,50,0.8,0.8,0,53,0.8,0.85,0,72,0.8,0.86,0,77,0.8,0.87,0,84,0.8,0.88,0,91,0.8,0.89,0,100,0.8,0.9,0,111,0.8,0.92,0,140,0.8,0.93,0,160,0.8,0.94,0,188,0.8,0.95,0,227,0.8,0.96,0,285,0.8,0.97,0,381,0.8,0.98,0,575,0.8,0.99,1,1155,0.81,0.09,0,5,0.81,0.75,0,42,0.81,0.76,0,44,0.81,0.77,0,46,
    0.81,0.81,0,57,0.81,0.82,0,61,0.81,0.84,0,69,0.81,0.85,0,74,0.81,0.86,0,80,0.81,0.87,0,87,0.81,0.89,0,104,0.81,0.9,0,114,0.81,0.92,0,144,0.81,0.93,0,166,0.81,0.94,0,195,0.81,0.95,0,235,0.81,0.96,0,295,0.81,0.97,0,395,0.81,0.98,1,595,0.81,0.99,2,1195,0.82,0.48,0,17,0.82,0.5,0,18,0.82,0.65,0,29,0.82,0.7,0,35,0.82,0.72,0,38,0.82,0.83,0,67,0.82,0.86,0,83,0.82,0.87,0,90,0.82,0.88,0,98,0.82,0.89,0,107,0.82,0.9,0,119,0.82,0.91,0,132,0.82,0.92,0,150,0.82,0.93,0,172,
    0.82,0.94,0,202,0.82,0.95,0,243,0.82,0.96,0,305,0.82,0.97,0,409,0.82,0.98,1,617,0.82,0.99,2,1239,0.83,0.23,0,9,0.83,0.65,0,31,0.83,0.66,0,32,0.83,0.67,0,33,0.83,0.74,0,43,0.83,0.75,0,45,0.83,0.76,0,48,0.83,0.79,0,55,0.83,0.8,0,58,0.83,0.81,0,62,0.83,0.82,0,66,0.83,0.84,0,75,0.83,0.86,0,86,0.83,0.87,0,93,0.83,0.88,0,102,0.83,0.9,0,123,0.83,0.91,0,138,0.83,0.92,0,156,0.83,0.93,0,179,0.83,0.94,0,210,0.83,0.95,0,253,0.83,0.96,1,317,0.83,0.97,1,425,0.83,0.98,2,641,
    0.83,0.99,4,1287,0.84,0.43,0,16,0.84,0.6,0,27,0.84,0.63,0,30,0.84,0.75,0,47,0.84,0.77,0,52,0.84,0.81,0,64,0.84,0.82,0,68,0.84,0.83,0,73,0.84,0.84,0,78,0.84,0.87,0,97,0.84,0.88,0,106,0.84,0.89,0,116,0.84,0.9,0,128,0.84,0.91,0,143,0.84,0.92,0,162,0.84,0.93,0,186,0.84,0.94,0,218,0.84,0.95,1,263,0.84,0.96,1,331,0.84,0.97,1,443,0.84,0.98,2,668,0.84,0.99,5,1342,0.85,0.17,0,8,0.85,0.38,0,15,0.85,0.58,0,26,0.85,0.6,0,28,0.85,0.67,0,36,0.85,0.75,0,49,0.85,0.77,0,54,
    0.85,0.79,0,60,0.85,0.82,0,71,0.85,0.83,0,76,0.85,0.84,0,81,0.85,0.86,0,94,0.85,0.88,0,111,0.85,0.89,0,121,0.85,0.9,0,134,0.85,0.92,0,169,0.85,0.93,1,195,0.85,0.94,1,228,0.85,0.95,1,275,0.85,0.96,2,346,0.85,0.97,2,463,0.85,0.98,4,698,0.85,0.99,8,1402,0.86,0.53,0,24,0.86,0.55,0,25,0.86,0.68,0,39,0.86,0.69,0,40,0.86,0.74,0,50,0.86,0.79,0,63,0.86,0.83,0,80,0.86,0.84,0,85,0.86,0.85,0,91,0.86,0.86,0,98,0.86,0.87,0,107,0.86,0.89,0,127,0.86,0.9,1,141,0.86,0.91,1,157,
    0.86,0.92,1,178,0.86,0.93,1,204,0.86,0.94,1,239,0.86,0.95,2,289,0.86,0.96,2,362,0.86,0.97,3,486,0.86,0.98,5,732,0.86,0.99,11,1471,0.87,0.3,0,13,0.87,0.31,0,14,0.87,0.49,0,22,0.87,0.5,0,23,0.87,0.63,0,34,0.87,0.65,0,37,0.87,0.68,0,41,0.87,0.69,0,42,0.87,0.7,0,44,0.87,0.71,0,46,0.87,0.76,0,57,0.87,0.79,0,67,0.87,0.8,0,70,0.87,0.81,0,74,0.87,0.82,0,79,0.87,0.83,0,84,0.87,0.84,0,90,0.87,0.85,0,96,0.87,0.86,1,104,0.87,0.87,1,112,0.87,0.88,1,122,0.87,0.89,1,134,
    0.87,0.9,1,148,0.87,0.91,1,166,0.87,0.92,1,187,0.87,0.93,2,215,0.87,0.94,2,252,0.87,0.95,3,304,0.87,0.96,3,382,0.87,0.97,5,511,0.87,0.98,7,771,0.87,0.99,16,1549,0.88,0.45,0,21,0.88,0.72,0,51,0.88,0.73,0,53,0.88,0.74,0,55,0.88,0.75,0,58,0.88,0.76,0,61,0.88,0.8,1,74,0.88,0.81,1,79,0.88,0.82,1,83,0.88,0.83,1,89,0.88,0.84,1,95,0.88,0.85,1,102,0.88,0.86,1,110,0.88,0.87,1,119,0.88,0.88,1,129,0.88,0.89,1,142,0.88,0.9,2,157,0.88,0.91,2,175,0.88,0.92,2,198,0.88,0.93,3,227,
    0.88,0.94,3,267,0.88,0.95,4,322,0.88,0.96,5,404,0.88,0.97,7,541,0.88,0.98,11,816,0.88,0.99,22,1639,0.89,0.07,0,7,0.89,0.22,0,12,0.89,0.39,0,19,0.89,0.4,0,20,0.89,0.54,0,29,0.89,0.56,0,31,0.89,0.57,0,32,0.89,0.58,0,33,0.89,0.6,0,35,0.89,0.63,0,38,0.89,0.66,0,43,0.89,0.69,0,48,0.89,0.71,0,52,0.89,0.73,1,56,0.89,0.74,1,59,0.89,0.75,1,61,0.89,0.76,1,64,0.89,0.77,1,68,0.89,0.78,1,71,0.89,0.79,1,75,0.89,0.81,1,84,0.89,0.84,1,101,0.89,0.85,1,108,0.89,0.86,2,117,
    0.89,0.87,2,126,0.89,0.88,2,138,0.89,0.89,2,151,0.89,0.9,3,167,0.89,0.91,3,186,0.89,0.92,3,211,0.89,0.93,4,242,0.89,0.94,5,284,0.89,0.95,6,342,0.89,0.96,7,430,0.89,0.97,10,576,0.89,0.98,15,868,0.89,0.99,31,1745,0.9,0.18,0,11,0.9,0.33,0,17,0.9,0.35,0,18,0.9,0.49,0,27,0.9,0.53,0,30,0.9,0.59,0,36,0.9,0.64,1,43,0.9,0.65,1,44,0.9,0.66,1,46,0.9,0.67,1,47,0.9,0.68,1,49,0.9,0.69,1,51,0.9,0.7,1,53,0.9,0.71,1,55,0.9,0.72,1,58,0.9,0.73,1,60,0.9,0.74,1,63,
    0.9,0.75,1,66,0.9,0.76,1,69,0.9,0.77,1,72,0.9,0.78,1,76,0.9,0.79,1,80,0.9,0.8,2,85,0.9,0.81,2,90,0.9,0.82,2,95,0.9,0.83,2,101,0.9,0.84,2,108,0.9,0.85,2,116,0.9,0.86,2,125,0.9,0.87,3,135,0.9,0.88,3,147,0.9,0.89,3,162,0.9,0.9,4,179,0.9,0.91,4,200,0.9,0.92,5,226,0.9,0.93,6,259,0.9,0.94,7,304,0.9,0.95,8,367,0.9,0.96,11,461,0.9,0.97,14,617,0.9,0.98,22,930,0.9,0.99,44,1869,0.91,0.13,0,10,0.91,0.45,0,26,0.91,0.48,0,28,0.91,0.53,1,32,0.91,0.54,1,33,
    0.91,0.55,1,34,0.91,0.56,1,36,0.91,0.57,1,37,0.91,0.58,1,38,0.91,0.59,1,39,0.91,0.6,1,40,0.91,0.61,1,42,0.91,0.65,1,48,0.91,0.7,1,57,0.91,0.72,1,62,0.91,0.73,2,65,0.91,0.74,2,68,0.91,0.75,2,71,0.91,0.76,2,74,0.91,0.77,2,78,0.91,0.78,2,82,0.91,0.79,2,87,0.91,0.8,2,91,0.91,0.81,3,97,0.91,0.82,3,103,0.91,0.83,3,109,0.91,0.84,3,117,0.91,0.85,3,125,0.91,0.86,4,135,0.91,0.87,4,146,0.91,0.88,5,159,0.91,0.89,5,175,0.91,0.9,6,193,0.91,0.91,6,216,0.91,0.92,7,244,
    0.91,0.93,8,280,0.91,0.94,10,328,0.91,0.95,12,396,0.91,0.96,15,498,0.91,0.97,21,667,0.91,0.98,31,1005,0.91,0.99,63,2020,0.92,0.02,0,6,0.92,0.22,0,15,0.92,0.24,0,16,0.92,0.38,0,23,0.92,0.39,0,24,0.92,0.4,1,25,0.92,0.42,1,26,0.92,0.43,1,27,0.92,0.45,1,28,0.92,0.46,1,29,0.92,0.47,1,30,0.92,0.48,1,31,0.92,0.53,1,35,0.92,0.58,1,41,0.92,0.61,1,45,0.92,0.64,2,50,0.92,0.65,2,52,0.92,0.66,2,54,0.92,0.67,2,56,0.92,0.68,2,58,0.92,0.69,2,60,0.92,0.7,2,63,0.92,0.74,3,74,
    0.92,0.75,3,78,0.92,0.76,3,81,0.92,0.77,3,85,0.92,0.78,3,90,0.92,0.79,3,94,0.92,0.8,4,100,0.92,0.81,4,106,0.92,0.82,4,112,0.92,0.83,4,119,0.92,0.84,5,128,0.92,0.85,5,137,0.92,0.86,6,147,0.92,0.87,6,160,0.92,0.88,7,174,0.92,0.89,7,191,0.92,0.9,8,211,0.92,0.91,9,235,0.92,0.92,11,266,0.92,0.93,12,306,0.92,0.94,14,359,0.92,0.95,18,432,0.92,0.96,22,543,0.92,0.97,30,728,0.92,0.98,45,1097,0.92,0.99,91,2205,0.93,0.06,0,9,0.93,0.17,0,14,0.93,0.26,1,19,0.93,0.28,1,20,0.93,0.3,1,21,
    0.93,0.32,1,22,0.93,0.33,1,23,0.93,0.35,1,24,0.93,0.51,2,37,0.93,0.52,2,38,0.93,0.53,2,39,0.93,0.54,2,40,0.93,0.55,2,42,0.93,0.56,2,43,0.93,0.57,2,44,0.93,0.58,2,46,0.93,0.59,2,47,0.93,0.6,2,48,0.93,0.64,3,55,0.93,0.65,3,57,0.93,0.66,3,59,0.93,0.67,3,62,0.93,0.68,3,64,0.93,0.69,3,67,0.93,0.7,3,69,0.93,0.71,3,72,0.93,0.72,4,75,0.93,0.73,4,78,0.93,0.74,4,82,0.93,0.75,4,86,0.93,0.76,4,90,0.93,0.77,5,94,0.93,0.78,5,99,0.93,0.79,5,104,0.93,0.8,6,110,
    0.93,0.81,6,117,0.93,0.82,6,124,0.93,0.83,7,132,0.93,0.84,7,141,0.93,0.85,8,151,0.93,0.86,8,163,0.93,0.87,9,177,0.93,0.88,10,192,0.93,0.89,11,211,0.93,0.9,12,233,0.93,0.91,14,260,0.93,0.92,16,294,0.93,0.93,18,338,0.93,0.94,21,397,0.93,0.95,26,478,0.93,0.96,33,601,0.93,0.97,44,805,0.93,0.98,66,1214,0.93,0.99,134,2440,0.94,0.14,1,15,0.94,0.16,1,16,0.94,0.18,1,17,0.94,0.2,1,18,0.94,0.37,2,28,0.94,0.38,2,29,0.94,0.39,2,30,0.94,0.4,2,31,0.94,0.42,2,32,0.94,0.43,2,33,0.94,0.44,2,34,
    0.94,0.45,2,35,0.94,0.46,2,36,0.94,0.51,3,41,0.94,0.52,3,43,0.94,0.53,3,44,0.94,0.54,3,45,0.94,0.55,3,47,0.94,0.56,3,48,0.94,0.57,3,50,0.94,0.58,3,51,0.94,0.59,3,53,0.94,0.61,4,56,0.94,0.62,4,58,0.94,0.63,4,60,0.94,0.64,4,62,0.94,0.65,4,65,0.94,0.66,4,67,0.94,0.67,5,69,0.94,0.68,5,72,0.94,0.69,5,75,0.94,0.7,5,78,0.94,0.71,5,81,0.94,0.72,6,85,0.94,0.73,6,88,0.94,0.74,6,92,0.94,0.75,7,96,0.94,0.76,7,101,0.94,0.77,7,106,0.94,0.78,8,112,0.94,0.79,8,118,
    0.94,0.8,9,124,0.94,0.81,9,131,0.94,0.82,10,140,0.94,0.83,10,149,0.94,0.84,11,159,0.94,0.85,12,170,0.94,0.86,13,184,0.94,0.87,14,199,0.94,0.88,15,216,0.94,0.89,17,237,0.94,0.9,19,262,0.94,0.91,21,293,0.94,0.92,24,332,0.94,0.93,27,381,0.94,0.94,32,447,0.94,0.95,39,539,0.94,0.96,49,677,0.94,0.97,66,907,0.94,0.98,100,1367,0.94,0.99,201,2747,0.95,0.05,1,11,0.95,0.06,1,12,0.95,0.08,1,13,0.95,0.09,1,14,0.95,0.21,2,21,0.95,0.22,2,22,0.95,0.24,2,23,0.95,0.26,2,24,0.95,0.27,2,25,0.95,0.28,2,26,
    0.95,0.3,2,27,0.95,0.36,3,32,0.95,0.37,3,33,0.95,0.39,3,34,0.95,0.4,3,35,0.95,0.41,3,36,0.95,0.42,3,37,0.95,0.43,3,38,0.95,0.44,3,39,0.95,0.45,3,40,0.95,0.46,4,42,0.95,0.47,4,43,0.95,0.48,4,44,0.95,0.49,4,45,0.95,0.5,4,46,0.95,0.51,4,48,0.95,0.52,4,49,0.95,0.53,4,51,0.95,0.54,5,52,0.95,0.55,5,54,0.95,0.56,5,55,0.95,0.57,5,57,0.95,0.58,5,59,0.95,0.59,5,61,0.95,0.6,6,63,0.95,0.61,6,65,0.95,0.62,6,67,0.95,0.63,6,69,0.95,0.64,7,72,0.95,0.65,7,74,
    0.95,0.66,7,77,0.95,0.67,7,80,0.95,0.68,8,83,0.95,0.69,8,86,0.95,0.7,8,90,0.95,0.71,9,93,0.95,0.72,9,97,0.95,0.73,9,102,0.95,0.74,10,106,0.95,0.75,10,111,0.95,0.76,11,117,0.95,0.77,11,122,0.95,0.78,12,129,0.95,0.79,13,136,0.95,0.8,14,143,0.95,0.81,14,152,0.95,0.82,15,161,0.95,0.83,16,171,0.95,0.84,17,183,0.95,0.85,19,196,0.95,0.86,20,212,0.95,0.87,22,229,0.95,0.88,24,250,0.95,0.89,26,274,0.95,0.9,29,303,0.95,0.91,33,338,0.95,0.92,37,382,0.95,0.93,43,439,0.95,0.94,50,515,0.95,0.95,60,621,
    0.95,0.96,76,781,0.95,0.97,102,1046,0.95,0.98,154,1577,0.95,0.99,310,3169,0.96,0.01,1,9,0.96,0.02,1,10,0.96,0.08,2,16,0.96,0.1,2,17,0.96,0.11,2,18,0.96,0.13,2,19,0.96,0.14,2,20,0.96,0.19,3,23,0.96,0.2,3,24,0.96,0.21,3,25,0.96,0.22,3,26,0.96,0.24,3,27,0.96,0.25,3,28,0.96,0.26,3,29,0.96,0.27,3,30,0.96,0.29,4,31,0.96,0.3,4,32,0.96,0.31,4,33,0.96,0.32,4,34,0.96,0.33,4,35,0.96,0.34,4,36,0.96,0.35,4,37,0.96,0.36,4,38,0.96,0.37,5,39,0.96,0.38,5,40,0.96,0.39,5,41,
    0.96,0.4,5,42,0.96,0.41,5,43,0.96,0.42,5,44,0.96,0.43,5,46,0.96,0.44,6,47,0.96,0.45,6,48,0.96,0.46,6,50,0.96,0.47,6,51,0.96,0.48,6,52,0.96,0.49,7,54,0.96,0.5,7,55,0.96,0.51,7,57,0.96,0.52,7,59,0.96,0.53,7,60,0.96,0.54,8,62,0.96,0.55,8,64,0.96,0.56,8,66,0.96,0.57,9,68,0.96,0.58,9,70,0.96,0.59,9,73,0.96,0.6,9,75,0.96,0.61,10,78,0.96,0.62,10,80,0.96,0.63,10,83,0.96,0.64,11,86,0.96,0.65,11,89,0.96,0.66,12,92,0.96,0.67,12,96,0.96,0.68,13,99,0.96,0.69,13,103,
    0.96,0.7,14,107,0.96,0.71,14,112,0.96,0.72,15,116,0.96,0.73,16,121,0.96,0.74,16,127,0.96,0.75,17,133,0.96,0.76,18,139,0.96,0.77,19,146,0.96,0.78,20,154,0.96,0.79,21,162,0.96,0.8,22,171,0.96,0.81,24,181,0.96,0.82,25,192,0.96,0.83,27,205,0.96,0.84,29,219,0.96,0.85,31,235,0.96,0.86,33,253,0.96,0.87,36,274,0.96,0.88,39,298,0.96,0.89,43,327,0.96,0.9,48,362,0.96,0.91,53,404,0.96,0.92,60,457,0.96,0.93,69,525,0.96,0.94,81,616,0.96,0.95,98,743,0.96,0.96,123,933,0.96,0.97,166,1251,0.96,0.98,250,1885,0.96,0.99,503,3789,
    0.97,0.02,2,13,0.97,0.03,2,14,0.97,0.04,2,15,0.97,0.06,3,18,0.97,0.07,3,19,0.97,0.08,3,20,0.97,0.09,3,21,0.97,0.11,4,22,0.97,0.12,4,23,0.97,0.13,4,24,0.97,0.14,4,25,0.97,0.15,4,26,0.97,0.16,4,27,0.97,0.17,4,28,0.97,0.18,5,29,0.97,0.19,5,30,0.97,0.21,5,31,0.97,0.22,5,32,0.97,0.23,6,33,0.97,0.24,6,34,0.97,0.25,6,35,0.97,0.26,6,36,0.97,0.27,6,37,0.97,0.28,6,38,0.97,0.29,7,39,0.97,0.3,7,41,0.97,0.31,7,42,0.97,0.32,7,43,0.97,0.33,7,44,0.97,0.34,8,45,
    0.97,0.35,8,46,0.97,0.36,8,48,0.97,0.37,8,49,0.97,0.38,9,50,0.97,0.39,9,52,0.97,0.4,9,53,0.97,0.41,9,55,0.97,0.42,10,56,0.97,0.43,10,58,0.97,0.44,10,59,0.97,0.45,11,61,0.97,0.46,11,63,0.97,0.47,11,64,0.97,0.48,12,66,0.97,0.49,12,68,0.97,0.5,12,70,0.97,0.51,13,72,0.97,0.52,13,74,0.97,0.53,13,76,0.97,0.54,14,79,0.97,0.55,14,81,0.97,0.56,15,84,0.97,0.57,15,86,0.97,0.58,16,89,0.97,0.59,16,92,0.97,0.6,17,95,0.97,0.61,17,98,0.97,0.62,18,101,0.97,0.63,19,105,0.97,0.64,19,109,
    0.97,0.65,20,112,0.97,0.66,21,117,0.97,0.67,22,121,0.97,0.68,22,126,0.97,0.69,23,130,0.97,0.7,24,136,0.97,0.71,25,141,0.97,0.72,26,147,0.97,0.73,28,154,0.97,0.74,29,161,0.97,0.75,30,168,0.97,0.76,32,176,0.97,0.77,33,185,0.97,0.78,35,195,0.97,0.79,37,205,0.97,0.8,39,217,0.97,0.81,41,229,0.97,0.82,44,243,0.97,0.83,47,259,0.97,0.84,50,277,0.97,0.85,54,297,0.97,0.86,58,320,0.97,0.87,63,347,0.97,0.88,69,378,0.97,0.89,75,414,0.97,0.9,83,458,0.97,0.91,93,512,0.97,0.92,106,579,0.97,0.93,121,665,0.97,0.94,142,780,
    0.97,0.95,172,940,0.97,0.96,216,1182,0.97,0.97,290,1583,0.97,0.98,437,2387,0.97,0.99,880,4798,0.98,0.01,3,15,0.98,0.02,4,18,0.98,0.03,5,20,0.98,0.04,5,22,0.98,0.05,5,23,0.98,0.06,6,25,0.98,0.07,6,26,0.98,0.08,7,27,0.98,0.09,7,29,0.98,0.1,7,30,0.98,0.11,8,31,0.98,0.12,8,33,0.98,0.13,8,34,0.98,0.14,9,35,0.98,0.15,9,36,0.98,0.16,9,38,0.98,0.17,10,39,0.98,0.18,10,40,0.98,0.19,10,41,0.98,0.2,11,43,0.98,0.21,11,44,0.98,0.22,11,45,0.98,0.23,12,47,0.98,0.24,12,48,0.98,0.25,12,49,
    0.98,0.26,13,51,0.98,0.27,13,52,0.98,0.28,13,54,0.98,0.29,14,55,0.98,0.3,14,57,0.98,0.31,15,58,0.98,0.32,15,60,0.98,0.33,16,62,0.98,0.34,16,63,0.98,0.35,16,65,0.98,0.36,17,67,0.98,0.37,17,69,0.98,0.38,18,71,0.98,0.39,18,73,0.98,0.4,19,75,0.98,0.41,19,77,0.98,0.42,20,79,0.98,0.43,21,81,0.98,0.44,21,83,0.98,0.45,22,86,0.98,0.46,22,88,0.98,0.47,23,90,0.98,0.48,24,93,0.98,0.49,24,96,0.98,0.5,25,98,0.98,0.51,26,101,0.98,0.52,27,104,0.98,0.53,28,107,0.98,0.54,28,111,0.98,0.55,29,114,
    0.98,0.56,30,118,0.98,0.57,31,121,0.98,0.58,32,125,0.98,0.59,33,129,0.98,0.6,34,133,0.98,0.61,36,138,0.98,0.62,37,143,0.98,0.63,38,147,0.98,0.64,39,153,0.98,0.65,41,158,0.98,0.66,42,164,0.98,0.67,44,170,0.98,0.68,46,177,0.98,0.69,47,183,0.98,0.7,49,191,0.98,0.71,51,199,0.98,0.72,54,207,0.98,0.73,56,216,0.98,0.74,59,226,0.98,0.75,61,237,0.98,0.76,64,248,0.98,0.77,68,260,0.98,0.78,71,274,0.98,0.79,75,289,0.98,0.8,79,305,0.98,0.81,84,323,0.98,0.82,89,343,0.98,0.83,95,365,0.98,0.84,102,390,0.98,0.85,109,418,
    0.98,0.86,118,451,0.98,0.87,127,488,0.98,0.88,139,532,0.98,0.89,152,583,0.98,0.9,168,645,0.98,0.91,188,720,0.98,0.92,213,815,0.98,0.93,245,936,0.98,0.94,287,1098,0.98,0.95,346,1324,0.98,0.96,436,1664,0.98,0.97,584,2230,0.98,0.98,881,3362,0.98,0.99,1770,6757,0.99,0.01,10,28,0.99,0.02,12,32,0.99,0.03,14,36,0.99,0.04,15,39,0.99,0.05,16,42,0.99,0.06,17,45,0.99,0.07,18,47,0.99,0.08,19,50,0.99,0.09,20,52,0.99,0.1,21,55,0.99,0.11,22,57,0.99,0.12,23,59,0.99,0.13,24,62,0.99,0.14,25,64,0.99,0.15,26,66,0.99,0.16,27,69,
    0.99,0.17,28,71,0.99,0.18,29,73,0.99,0.19,30,76,0.99,0.2,31,78,0.99,0.21,32,80,0.99,0.22,33,83,0.99,0.23,34,85,0.99,0.24,35,88,0.99,0.25,36,90,0.99,0.26,37,93,0.99,0.27,38,96,0.99,0.28,39,98,0.99,0.29,40,101,0.99,0.3,41,104,0.99,0.31,42,107,0.99,0.32,44,110,0.99,0.33,45,113,0.99,0.34,46,116,0.99,0.35,47,119,0.99,0.36,49,123,0.99,0.37,50,126,0.99,0.38,51,129,0.99,0.39,53,133,0.99,0.4,54,137,0.99,0.41,56,140,0.99,0.42,57,144,0.99,0.43,59,148,0.99,0.44,61,152,0.99,0.45,62,157,0.99,0.46,64,161,
    0.99,0.47,66,166,0.99,0.48,68,170,0.99,0.49,70,175,0.99,0.5,72,180,0.99,0.51,74,186,0.99,0.52,76,191,0.99,0.53,79,197,0.99,0.54,81,203,0.99,0.55,83,209,0.99,0.56,86,216,0.99,0.57,89,222,0.99,0.58,92,229,0.99,0.59,95,237,0.99,0.6,98,245,0.99,0.61,101,253,0.99,0.62,105,261,0.99,0.63,108,270,0.99,0.64,112,280,0.99,0.65,116,290,0.99,0.66,120,301,0.99,0.67,125,312,0.99,0.68,130,324,0.99,0.69,135,337,0.99,0.7,140,350,0.99,0.71,146,365,0.99,0.72,152,380,0.99,0.73,159,397,0.99,0.74,166,415,0.99,0.75,174,434,0.99,0.76,182,455,
    0.99,0.77,192,478,0.99,0.78,202,502,0.99,0.79,213,530,0.99,0.8,225,559,0.99,0.81,238,592,0.99,0.82,253,629,0.99,0.83,269,670,0.99,0.84,287,716,0.99,0.85,308,768,0.99,0.86,332,827,0.99,0.87,360,896,0.99,0.88,392,976,0.99,0.89,430,1071,0.99,0.9,476,1184,0.99,0.91,0,0,0.99,0.93,0,0
    ),byrow=T,ncol=4)
  colnames(res) <- c("delta","lambda","a","b")
  res
  }

# generate search grid (internal use only)
searchGrid <- function(mings,maxgs,minwd,maxld,lag.type,gammaTab,tol) {
  if(lag.type=="gamm.lag") {
    auxind <- which(gammaTab[,"a"]>=mings-tol & gammaTab[,"a"]<=maxgs+tol & gammaTab[,"b"]<=maxld+tol & gammaTab[,"b"]-gammaTab[,"a"]>=minwd-tol)
    if(length(auxind)>0) {
      res <- gammaTab[auxind,1:2]
      if(length(auxind)>500) {
        outind <- seq(1,length(auxind),by=ceiling(length(auxind)/500))  ## <---------- thinning
        res[outind,]
        } else {
        res  
        }
      }
    } else {
    auxmat <- c()
    for(i in 0:maxld) {
      for(j in i:maxld) { 
        if(i>=mings & i<=maxgs & j-i>=minwd)
        auxmat <- rbind(auxmat,c(i,j))
        }   
      }
    auxmat
    }
  }

# create lm formula (internal use only)
creatForm <- function(y,X,group,type,theta) {
  xnam <- c()
  if(length(X)>0) {
    for(i in 1:length(X)) {
      if(type[i]=="none") {
        xnam[i] <- X[i]
        } else {
        if(is.null(group)) {
          xnam[i] <- paste(type[i],"(",X[i],",",theta[[i]][1],",",theta[[i]][2],")",sep="")
          } else {
          xnam[i] <- paste(type[i],"(",X[i],",",theta[[i]][1],",",theta[[i]][2],",",group,")",sep="")
          }
        }
      }
    }
  if(is.null(group)) {
    if(length(X)>0) {
      res <- paste(y,"~",paste(xnam,collapse="+"),sep="")    
      } else {
      res <- paste(y,"~1",sep="")
      }
    } else {
    if(length(X)>0) {
      res <- paste(y,"~-1+",group,"+",paste(xnam,collapse="+"),sep="")    
      } else {
      res <- paste(y,"~-1+",group,sep="")
      }
    }
  formula(res)
  }

# extract the name from a lag shape (internal use only)
extrName <- function(x) {
  if(identical("quec.lag(",substr(x,1,9))) {                           
    gsub("\\)","",gsub("quec\\.lag\\(","",strsplit(x,",")[[1]]))[1]
    } else if(identical("qdec.lag(",substr(x,1,9))) {                            
    gsub("\\)","",gsub("qdec\\.lag\\(","",strsplit(x,",")[[1]]))[1]
    } else if(identical("gamm.lag(",substr(x,1,9))) {
    gsub("\\)","",gsub("gamm\\.lag\\(","",strsplit(x,",")[[1]]))[1]
    } else {
    x
    }
  }

# extract lag parameters from a formula (internal use only)
extrLPar <- function(x) {
  auxform <- scanForm(x)
  nomi <- auxform$X
  if(length(nomi)>0) {
    res <- matrix(nrow=length(nomi),ncol=2)
    for(i in 1:length(nomi)) {
      inam <- nomi[i]
      ilab <- extrName(inam)
      if(ilab %in% names(auxform$ltype)) {
        ityp <- auxform$ltype[ilab]
        ipar <- auxform$lpar[[ilab]]
        } else {
        ityp <- "none"
        ipar <- NA
        }
      if(ityp=="none") {
        res[i,] <- c(0,0)
        } else if(ityp=="gamm.lag") {
        res[i,] <- gamlim(ipar[1],ipar[2])
        } else {
        res[i,] <- c(ipar[1],ipar[2]) 
        }
      }
    rownames(res) <- sapply(nomi,extrName)
    colnames(res) <- c("a","b") 
    res
    }
  }

# get levels of variables in data (internal use only)
getLev <- function(data) {
  auxq <- sapply(data,isQuant)
  res <- list()
  for(i in 1:length(auxq)) {
    if(auxq[i]==T) {
      res[[i]] <- NA
      } else {
      res[[i]] <- paste(colnames(data)[i],levels(factor(data[,i])),sep="")
      }
    }
  names(res) <- colnames(data)
  res
  }


# perform ols estimation (internal use only)
doLS <- function(formula,group,data) {
  formOK <- formula
  Xm0 <- model.matrix(formOK,data=data)
  auxdel <- names(which(apply(Xm0,2,var)==0))
  if(length(auxdel)>0) {
    auxlev <- getLev(data)
    x2del <- c()
    for(i in 1:length(auxdel)) {
      if(auxdel[i] %in% colnames(data)) {
        x2del <- c(x2del,auxdel[i])
        } else {
        auxf <- sapply(auxlev,function(z){auxdel[i] %in% z})
        x2del <- c(x2del,names(which(auxf==T)))
        }
      }
    x2del <- setdiff(x2del,group)
    if(length(x2del)>0) {
      auxform <- scanForm(formula)  
      if(is.null(group)) auxsep <- "" else auxsep <- "-1+"
      formOK <- formula(paste(auxform$y,"~",auxsep,paste(setdiff(auxform$X,x2del),collapse="+"),sep=""))
      }
    }
  res <- lm(formOK,data=data)
  res$call$formula <- formOK
  res
  }

# fit a distributed-lag linear regression model (internal use only)
dlaglm <- function(formula,group,data,adapt,no.select,min.gestation,max.gestation,min.width,max.lead,sign,selection,ndiff,gammaTab) {
  auxscan <- scanForm(formula)
  y <- auxscan$y
  if(length(auxscan$X)>0) {
    lagPar <- auxscan$lpar
    lagType <- auxscan$ltype
    lagNam <- names(lagPar)
    if(length(lagNam)==0) adapt <- F
    laglimit <- findLagLim(data[,c(y,lagNam,group)],group=group)-ndiff
    if(adapt==F) {
      for(i in 1:length(lagPar)) {
        if(!is.na(lagPar[[i]][1]) && lagPar[[i]][1]>laglimit) lagPar[[i]][1] <- laglimit
        if(!is.na(lagPar[[i]][2]) && lagPar[[i]][2]>laglimit) lagPar[[i]][2] <- laglimit
        }
      formOK <- creatForm(y,names(lagPar),group,lagType,lagPar)
      modOK <- doLS(formula=formOK,group=group,data=data)
      } else {
      bestPar <- vector("list",length=length(lagNam))
      names(bestPar) <- lagNam
      xOK <- no.select  #####
      fine <- 0
      while(fine==0) {
        xtest <- setdiff(lagNam,xOK)
        ntest <- length(xtest)
        if(ntest>0) {
          currentAIC <- rep(NA,ntest) 
          names(currentAIC) <- xtest
          for(i in 1:ntest) {
            if(xtest[i] %in% names(sign)) {
              isign <- sign[xtest[i]]
              } else {
              isign <- NULL
              }
            if(lagType[xtest[i]]!="none") {
              if(xtest[i] %in% names(min.gestation)) {
                mings <- min.gestation[xtest[i]]
                } else {
                mings <- 0
                }
              if(xtest[i] %in% names(max.gestation)) {
                maxgs <- max.gestation[xtest[i]]
                } else {
                maxgs <- laglimit
                }
              if(xtest[i] %in% names(min.width)) {
                minwd <- min(laglimit,min.width[xtest[i]])  #####
                } else {
                minwd <- 0
                }
              if(xtest[i] %in% names(max.lead)) {
                maxld <- min(laglimit,max.lead[xtest[i]])  #####
                } else {
                maxld <- laglimit
                }
              #for(j in 1:length(lagPar)) {
              #  if(sum(is.na(lagPar[[j]]))==0) {
              #    if(lagType[names(lagPar)[j]]=="gamm.lag") {
              #      tab0 <- which(gammaTab[,"a"]==mings & gammaTab[,"b"]==maxld)
              #      if(length(tab0)>0) lagPar[[j]] <- gammaTab[rev(tab0)[1],1:2]  ## <---------- scelta iniziale dei lag gamma
              #      #if(length(tab0)>0) lagPar[[j]] <- c(0.93,0.18)
              #      } else {
              #      lagPar[[j]] <- c(mings,maxld)
              #      }
              #    }
              #  }
              auxcons <- searchGrid(mings,maxgs,minwd,maxld,lagType[xtest[i]],gammaTab,tol=1)
              if(nrow(auxcons)==0) auxcons <- matrix(lagPar[[xtest[i]]],nrow=1)
              aic0 <- bhat0 <- c()
              for(j in 1:nrow(auxcons)) {
                testType <- lagType[c(xOK,xtest[i])]     
                testPar <- lagPar[c(xOK,xtest[i])]                   
                testPar[[xtest[i]]] <- auxcons[j,]
                form0 <- creatForm(y,names(testPar),group,testType,testPar)
                mod0 <- doLS(formula=form0,group=group,data=data)
                summ0 <- summary(mod0)$coefficients
                ixall <- rownames(summ0)
                iauxlab <- sapply(ixall,extrName)
                ixlab <- ixall[which(iauxlab==xtest[i])]
                #
                if(length(ixlab)>0 && ixlab %in% rownames(summ0)) { 
                  bhat0[j] <- summ0[ixlab,1]
                  if(selection=="aic") {
                    aic0[j] <- AIC(mod0)
                    } else {
                    aic0[j] <- BIC(mod0)
                    }
                  } else {
                  bhat0[j] <- NA
                  aic0[j] <- Inf
                  }
                }
              if(!is.null(isign)) {
                if(isign=="+") {
                  auxsign <- which(bhat0>0)
                  } else {
                  auxsign <- which(bhat0<0)                  
                  }
                if(length(auxsign)>0) {
                  auxbest <- auxsign[which.min(aic0[auxsign])]
                  } else {
                  auxbest <- which.min(aic0)
                  }
                } else {
                auxbest <- which.min(aic0)
                }
              bestPar[[xtest[i]]] <- auxcons[auxbest,]
              currentAIC[xtest[i]] <- aic0[auxbest]
              } else {
              testType <- lagType[c(xOK,xtest[i])]     
              testPar <- lagPar[c(xOK,xtest[i])]
              form0 <- creatForm(y,names(testPar),group,testType,testPar)
              mod0 <- doLS(formula=form0,group=group,data=data)
              auxsumm <- summary(mod0)$coefficients
              if(xtest[i] %in% rownames(auxsumm)) {
                if(selection=="aic") {
                  currentAIC[xtest[i]] <- AIC(mod0)
                  } else {
                  currentAIC[xtest[i]] <- BIC(mod0)  
                  }
                } else {
                currentAIC[xtest[i]] <- Inf     
                }
              }
            }
          xOK <- c(xOK,names(currentAIC)[which.min(currentAIC)])
          } else {
          fine <- 1
          }
        }
      formOK <- creatForm(y,names(bestPar),group,lagType,bestPar)
      modOK <- doLS(formula=formOK,group=group,data=data)
      }
    } else {
    if(is.null(group)) {
      formOK <- formula(paste(y,"~1",sep=""))
      } else {
      formOK <- formula(paste(y,"~-1+",group,sep=""))        
      }
    modOK <- doLS(formula=formOK,group=group,data=data)
    }
  modOK
  }

# hac weights (internal use only)
W_hac <- function(Xmat,res,maxlag) {
  n <- nrow(Xmat)
  p <- ncol(Xmat)
  W <- matrix(0,nrow=p,ncol=p)
  for(i in 1:n) {
    W <- W+res[i]^2*Xmat[i,]%*%t(Xmat[i,])
    }
  if(maxlag>0) {
    for(j in 1:maxlag) {
      wi <- 0
      for(i in (j+1):n) {
        wi <- wi+res[i]*res[i-j]*(Xmat[i,]%*%t(Xmat[i-j,])+Xmat[i-j,]%*%t(Xmat[i,]))
        }
      W <- W+(1-j/(1+maxlag))*wi
      }
    }
  W
  }

# newey-west hac covariance matrix
vcovHAC <- function(x,group=NULL) {
  if(("lm" %in% class(x))==F & ("dlsem" %in% class(x))==F) stop("Argument 'x' must be an object of class 'lm' or 'dlsem'",call.=F)
  if("lm" %in% class(x)) {
    doHAC(x=x,group=group)
    } else {
    lapply(x$estimate,doHAC,group=group)
    }
  }

# hac for class lm (internal use only)
doHAC <- function(x,group) {
  Xmat <- model.matrix(x)
  n <- nrow(Xmat)
  p <- ncol(Xmat)
  res <- residuals(x)
  if(!is.null(group) && is.na(group)) group <- NULL
  if(is.null(group)) {
    maxlag <- ar(res)$order
    W <- W_hac(Xmat,res,maxlag)
    } else {
    if(length(group)!=1) stop("Argument 'group' must contain a single variable name",call.=F)
    if((group %in% names(x$xlevels))==F) stop("Unknown variable '",group,"' provided to argument 'group'",call.=F)
    glev <- x$xlevels[[group]]
    if(length(glev)<2) stop("The group factor must have at least 2 unique values",call.=F)
    gnam <- paste(group,glev,sep="")
    if("(Intercept)" %in% colnames(Xmat)) gnam[1] <- "(Intercept)"
    Wsum <- matrix(0,nrow=ncol(Xmat),ncol=ncol(Xmat))
    W <- matrix(0,nrow=p,ncol=p)
    maxlag <- c()
    for(i in 1:length(gnam)) {
      iind <- names(which(Xmat[,gnam[i]]==1))
      maxlag[i] <- ar(res[iind])$order
      W <- W+W_hac(Xmat[iind,],res[iind],maxlag[i])
      }
    names(maxlag) <- gnam
    }
  Imat <- solve(t(Xmat)%*%Xmat)
  out <- n/(n-p)*Imat%*%W%*%Imat
  attr(out,"max.lag") <- maxlag
  out
  }

# vcov method for class 'hac'
vcov.hac <- function(object,...)  {
  object$vcov
  }

# summary method for class 'hac'
summary.hac <- function(object,...)  {
  res <- summary.lm(object)
  res$coefficients[,2] <- sqrt(diag(object$vcov))
  res$coefficients[,3] <- res$coefficients[,1]/res$coefficients[,2]
  res$coefficients[,4] <- 2*pt(-abs(res$coefficients[,3]),object$df.residual)
  res
  }

# compute lag effects of a covariate (internal use only)
lagEff <- function(model,x,cumul,conf,lag) {
  formstr <- strsplit(gsub(" ","",as.character(model$call$formula)[3]),"\\+")[[1]]
  xall <- names(model$coefficients)
  auxlab <- sapply(xall,extrName)
  xlab <- xall[which(auxlab==x)]
  auxscan <- scanForm(model$call)
  if(auxscan$ltype[x]=="quec.lag") {
    sx <- auxscan$lpar[[x]][1]
    dx <- auxscan$lpar[[x]][2]
    imu <- model$coefficients[xlab]
    icov <- vcov(model)[xlab,xlab]
    if(is.null(lag)) {
      xgrid <- 0:dx
      } else {
      xgrid <- lag
      }
    iH <- matrix(lagwei(c(sx,dx),xgrid,"quec.lag"),ncol=1)
    } else if(auxscan$ltype[x]=="qdec.lag") {
    sx <- auxscan$lpar[[x]][1]
    dx <- auxscan$lpar[[x]][2]
    imu <- model$coefficients[xlab]
    icov <- vcov(model)[xlab,xlab]
    if(is.null(lag)) {
      xgrid <- 0:dx
      } else {
      xgrid <- lag
      }
    iH <- matrix(lagwei(c(sx,dx),xgrid,"qdec.lag"),ncol=1)
    } else if(auxscan$ltype[x]=="gamm.lag") {
    delta <- auxscan$lpar[[x]][1]
    lambda <- auxscan$lpar[[x]][2]
    ilim <- gamlim(delta,lambda)
    imu <- model$coefficients[xlab]
    icov <- vcov(model)[xlab,xlab]
    if(is.null(lag)) {
      xa <- 0  #####
      #maxgam <- qgamma(0.99,1/(1-delta),-log(lambda))+xa
      xgrid <- 0:ilim[2]
      } else {
      xgrid <- lag
      }
    iH <- matrix(lagwei(c(delta,lambda),xgrid,"gamm.lag"),ncol=1)
    idel <- setdiff(xgrid,ilim[1]:ilim[2])
    if(length(idel)>0) iH[sapply(idel,function(z){which(xgrid==z)}),] <- 0
    } else {  
    xgrid <- 0  
    imu <- model$coefficients[x]
    icov <- matrix(diag(rep(1,length(imu))),nrow=length(imu),ncol=length(imu))
    iH <- matrix(1,nrow=1,ncol=1)
    }
  ibhat <- iH%*%imu
  ibse <- sqrt(diag(iH%*%icov%*%t(iH)))
  quan <- -qnorm((1-conf)/2)
  out <- cbind(ibhat,ibhat-quan*ibse,ibhat+quan*ibse)
  if(cumul==F) {
    rownames(out) <- xgrid
    colnames(out) <- c("estimate",paste(c("lower ","upper "),conf*100,"%",sep=""))
    out
    } else {
    outC <- out
    for(j in 1:ncol(out)) {
      for(i in 1:nrow(out)) {
        outC[i,j] <- sum(out[1:i,j])
        }
      }
    rownames(outC) <- xgrid
    colnames(outC) <- c("estimate",paste(c("lower ","upper "),conf*100,"%",sep=""))
    outC
    }
  }

# adf test (internal use only)
adft <- function(x,k) {
  k <- k+1
  x <- as.vector(x,mode="double")
  y <- diff(x)
  n <- length(y)
  z <- embed(y,k)
  yt <- z[,1]
  xt1 <- x[k:n]
  tt <- k:n
  if(k>1) {
    yt1 <- z[,2:k,drop=F]
    res <- lm(yt~xt1+tt+yt1)
    } else {
    res <- lm(yt~xt1+tt)
    }
  res.sum <- summary(res)$coefficients
  if(nrow(res.sum)>=2) {
    STAT <- res.sum[2,1]/res.sum[2,2]
    table <- -1*cbind(c(4.38, 4.15, 4.04, 3.99, 3.98, 3.96), c(3.95, 
      3.8, 3.73, 3.69, 3.68, 3.66), c(3.6, 3.5, 3.45, 3.43, 
      3.42, 3.41), c(3.24, 3.18, 3.15, 3.13, 3.13, 3.12), c(1.14, 
      1.19, 1.22, 1.23, 1.24, 1.25), c(0.8, 0.87, 0.9, 0.92, 
      0.93, 0.94), c(0.5, 0.58, 0.62, 0.64, 0.65, 0.66), c(0.15, 
      0.24, 0.28, 0.31, 0.32, 0.33))
    tablen <- dim(table)[2]
    tableT <- c(25, 50, 100, 250, 500, 1e+05)
    tablep <- c(0.01, 0.025, 0.05, 0.1, 0.9, 0.95, 0.975, 0.99)
    tableipl <- numeric(tablen)
    for(i in (1:tablen)) {
      tableipl[i] <- approx(tableT,table[,i],n,rule=2)$y
      }
    PVAL <- approx(tableipl,tablep,STAT,rule=2)$y
    } else {
    STAT <- PVAL <- NA
    }
  list(statistic=STAT,p.value=PVAL,'lag.order'=k-1)
  }

# apply differentiation (internal use only)
applyDiff <- function(x,group,data,k) {
  if(is.null(x)) x <- setdiff(colnames(data),group)
  deltaFun <- function(z,k) {
    if(k>0 & k<length(z)) {
      zd <- c(rep(NA,k),z[1:(length(z)-k)])
      z-zd
      } else if(k<=0) {
      z
      } else {
      rep(NA,length(z))
      }
    }
  diffdat <- data
  if(is.null(group)) {
    for(w in 1:length(x)) {
      if(isQuant(data[,x[w]])) {
        diffdat[,x[w]] <- deltaFun(data[,x[w]],k[w])      
        }
      }
    } else {
    data[,group] <- factor(data[,group])
    gruppi <- levels(factor(data[,group]))
    for(i in 1:length(gruppi)) {
      auxind <- which(data[,group]==gruppi[i])
      for(w in 1:length(x)) {
        if(isQuant(data[,x[w]])) {
          diffdat[auxind,x[w]] <- deltaFun(data[auxind,x[w]],k[w])
          }
        }
      } 
    }
  diffdat
  }

# unit root test
unirootTest <- function(x=NULL,group=NULL,time=NULL,data,combine="choi",k=0,log=FALSE) {
  if(!identical(class(data),"data.frame")) stop("Argument 'data' must be a data.frame",call.=F)
  if(length(log)!=1 || !is.logical(log)) stop("Argument 'log' must be a logical value",call.=F)
  if(!is.null(time) && is.na(time)) time <- NULL
  if(!is.null(time) && (time %in% colnames(data))==F) stop("Unknown variable '",time,"' provided to argument 'time'",call.=F)
  if(!is.null(time) && length(time)!=1) stop("Argument 'time' must contain a single variable name",call.=F)
  if(!is.null(group) && (group %in% colnames(data))==F) stop("Unknown variable '",group,"' provided to argument 'group'",call.=F)
  if(!is.null(group) && length(group)!=1) stop("Argument 'group' must contain a single variable name",call.=F)
  if(!is.null(group) && is.na(group)) group <- NULL
  if(!is.null(x)) { 
    for(i in 1:length(x)) {
      if(isQuant(data[,x[i]])==F) stop("'",x[i],"' is not a numerical variable",call.=F)
      }
    } else {
    allnam <- setdiff(colnames(data),c(group,time))
    for(i in 1:length(allnam)) {
      if(isQuant(data[,allnam[i]])) x <- c(x,allnam[i])
      }
    }
  auxvar <- setdiff(c(x,group,time),colnames(data))
  if(length(auxvar)>0) stop("Unknown variable: '",auxvar[1],"'",sep="",call.=F)
  if(log==T) {
    nolog <- c()
    for(i in 1:length(x)) {
      if(sum(data[,x[i]]<=0,na.rm=T)>0) {
        nolog <- c(nolog,x[i])        
        } else {
        data[,x[i]] <- log(data[,x[i]])
        }
      }
    }
  if(!is.null(group)) {
    data[,group] <- factor(data[,group])
    gruppi <- levels(data[,group])
    if(length(gruppi)<2) stop("The group factor must have at least 2 unique values",call.=F)
    g.id <- as.numeric(data[,group])
    glab <- gruppi
    if(is.null(k)) k <- trunc((min(table(g.id))-1)^(1/3))
    } else {
    g.id <- glab <- rep(1,nrow(data))  
    if(is.null(k)) k <- trunc((nrow(data)-1)^(1/3))
    }  
  if(length(combine)!=1 || (combine %in% c("choi","demetrescu"))==F) stop("Argument 'combine' must be either 'choi' or 'demetrescu'",call.=F)
  if(length(k)!=1 || !is.numeric(k) || k<0 || k!=round(k)) stop("Argument 'k' must be an non-negative integer",call.=F)
  data[which(abs(data[,x])==Inf),x] <- NA
  if(!is.null(time)) {
    for(i in 1:length(g.id)) {
      auxind <- which(g.id==gruppi[i])
      idat <- data[auxind,]
      data[auxind,] <- idat[order(idat[,time]),]
      }
    }
  res <- vector("list",length=length(x))
  for(i in 1:length(x)) {
    res[[i]] <- doADFtest(data[,x[i]],g.id=g.id,combine=combine,k=k,glab=glab)
    }
  names(res) <- x
  if(!is.null(group)) attr(res,"combine") <- combine else attr(res,"combine") <- NULL
  attr(res,"k") <- k
  class(res) <- "unirootTest"
  res
  }

# unit root test for a variable (internal use only)  
doADFtest <- function(x,g.id,combine,k,glab) {
  gruppi <- sort(unique(g.id))
  auxstat <- auxp <- nwm <- c()
  options(warn=-1)
  for(i in 1:length(gruppi)) {
    auxind <- which(g.id==gruppi[i])
    auxdat <- na.omit(x[auxind])
    nwm[i] <- length(auxdat)
    if(length(auxdat)>4 && var(auxdat)>0) {
      auxadf <- adft(auxdat,k=k)
      auxstat[i] <- auxadf$statistic
      auxp[i] <- auxadf$p.value
      } else {
      auxstat[i] <- NA
      auxp[i] <- NA
      }
    }
  if(length(auxstat)>1) names(auxstat) <- names(nwm) <- glab
  if(length(auxp)>0) {
    auxpStar <- auxp[which(auxp<1)]
    if(length(auxpStar)>0) {
      m <- length(auxpStar)
      logp <- qnorm(auxpStar)
      rhat <- 1-var(logp)
      rstar <- max(rhat,-1/(m-1))
      if(combine=="demetrescu") {
        auxz <- sum(logp)/sqrt(m*(1+(m-1)*(rstar+0.2*sqrt(2/(m+1))*(1-rstar))))
        } else if(combine=="choi") {
        auxz <- sum(logp)/sqrt(m)
        }
      auxpval <- 2*pnorm(-abs(auxz))
      } else {
      auxz <- NA
      auxpval <- NA
      }
    res <- list(statistic=auxstat,n=nwm,z.value=auxz,p.value=auxpval)
    } else {
    res <- list(statistic=NULL,n=nwm,z.value=NULL,p.value=NULL)
    }
  options(warn=0)
  res
  }

# print method for class 'unirootTest'
print.unirootTest <- function(x,...) {
  if(!is.null(attr(x,"combine"))) {
    if(attr(x,"combine")=="choi") {
      auxcom <- "Choi"
      } else {
      auxcom <- "Demetrescu"
      }
    auxp <- paste(auxcom,"'s p-values",sep="")
    } else {
    auxp <- "p-values"
    }
  auxH0 <- "(null hypothesis is unit root)"
  if(attr(x,"k")==0) {
    auxtest <- "Standard Dickey-Fuller test" 
    } else {
    auxtest <- paste("Augmented Dickey-Fuller test (lag order: ",attr(x,"k"),")",sep="")
    }
  cat(auxtest,"\n")  
  cat(auxp,auxH0,"\n")
  res <- sapply(x,function(z){z$p.value})
  print(round(res,4))
  }

# estimate parameters of the imputation model (internal use only)
impuFit <- function(xcont,xqual,group,data) {
  res <- G <- list()
  logL <- 0
  z.names <- c(group,xqual)
  for(i in 1:length(xcont)) {
    if(i==1) {
      if(is.null(z.names)) ipar <- "1" else ipar <- z.names  
      } else {
      ipar <- c(xcont[1:(i-1)],z.names)        
      }
    iform <- formula(paste(xcont[i],"~",paste(ipar,collapse="+"),sep=""))
    imod <- lm(iform,data=data)
    imod$call$formula <- iform
    logL <- logL-0.5*AIC(imod,k=0)
    res[[i]] <- imod
    G[[i]] <- ipar
    }
  names(res) <- names(G) <- xcont
  list(hat=res,G=G,logL=logL)
  }

# predict missing values from the imputation model (internal use only)
impuPred <- function(mod,data) {
  est <- mod$hat
  G <- mod$G
  res <- data
  options(warn=-1)
  for(i in 1:length(est)) {
    iest <- est[[i]]
    inam <- names(est)[i]
    ipar <- G[[inam]]
    ina <- which(is.na(data[,inam]))
    if(length(ina)>0) {res[ina,inam] <- predict(iest,res[ina,ipar,drop=F])}
    }
  options(warn=0)
  res
  }

# linear interpolation (internal use only)
linImp <- function(x) {
  res <- x
  auxNA <- which(is.na(x))
  if(length(auxNA)>0) {
    naL <- split(auxNA,cumsum(c(1,diff(auxNA)!=1)))
    for(i in 1:length(naL)) {
      ina <- naL[[i]]
      x1 <- min(ina)-1
      x2 <- max(ina)+1
      y1 <- x[x1]
      y2 <- x[x2]
      b <- (y2-y1)/(x2-x1)
      a <- y1-b*x1
      res[ina] <- a+b*ina
      }
    }
  res
  }

# spline reconstruction (internal use only)
splinImp <- function(x,group,data) {
  #
  doImp <- function(x) {
    res <- x
    auxNA <- which(is.na(x))
    if(length(auxNA)>0&length(auxNA)<length(x)) {
      auxO <- which(!is.na(x))
      auxI <- intersect(auxNA,min(auxO):max(auxO))
      yI <- spline(1:length(x),x,xout=1:length(x))
      res[auxI] <- yI$y[auxI]
      }
    res
    }
  #
  if(!is.null(group)) {
    data[,group] <- factor(data[,group])
    gruppi <- levels(data[,group])    
    res <- data
    for(w in 1:length(gruppi)) {
      auxind <- which(data[,group]==gruppi[w])
      for(i in 1:length(x)) {
        res[auxind,x[i]] <- doImp(data[auxind,x[i]])
        }
      }    
    } else {
    res <- data
    for(i in 1:length(x)) {
      res[,x[i]] <- doImp(data[,x[i]])
      }
    }
  res
  }

# imputation of missing data (internal use only)
EM.imputation <- function(xcont,xqual,group,data,tol,maxiter,quiet=F) {
  #isNA <- matrix(0,nrow(data),ncol=length(xcont))
  #colnames(isNA) <- xcont
  #currentDat <- data
  #for(i in 1:length(xcont)) {
  #  ina <- which(is.na(data[,xcont[i]]))
  #   if(length(ina)>0) {
  #    isNA[ina,xcont[i]] <- 1
  #    #currentDat[ina,xcont[i]] <- 0
  #    }
  #  }
  nmiss <- apply(data[,xcont],2,function(v){sum(is.na(v))})
  xcont <- xcont[order(nmiss)]
  currentDat <- data
  for(i in 1:length(xcont)) {
    currentDat[which(is.na(currentDat[,xcont[i]])),xcont[i]] <- mean(data[,xcont[i]],na.rm=T)
    }
  currentFit <- impuFit(xcont=xcont,xqual=xqual,group=group,data=currentDat)
  currentLik <- -Inf
  #currentLik <- currentFit$logL
  fine <- forcend <- 0
  count <- 1
  if(quiet==F) {
    cat("Starting EM...")
    flush.console() 
    }
  while(fine==0) {
    newDat <- impuPred(currentFit,data)
    newFit <- impuFit(xcont=xcont,xqual=xqual,group=group,data=newDat)
    newLik <- newFit$logL
    if(quiet==F) {
      cat('\r',"EM iteration ",count,". Log-likelihood: ",newLik,sep="")
      flush.console() 
      }
    if(newLik<currentLik) {
      newLik <- currentLik
      newDat <- currentDat
      #warning("Forced stop of EM algorithm because likelihood has decreased",call.=F)
      fine <- 1
      } else {
      if(newLik-currentLik>tol & count<maxiter) {
        currentFit <- newFit
        currentLik <- newLik
        currentDat <- newDat
        count <- count+1
        } else {
        fine <- 1
        if(count>=maxiter) forcend <- 1
        }
      }
    }
  if(quiet==F) {
    if(forcend==0) {
      cat('\r',"EM converged after ",count," iterations. Log-likelihood: ",newLik,sep="","\n")
      } else {
      cat('\r',"EM stopped after ",maxiter," iterations. Log-likelihood: ",newLik,sep="","\n")      
      }
    }
  newDat
  }

# function to plot a lag shape (internal use only)
makeShape <- function(bmat,maxlag,cumul,bcum,conf,ylim,title) {
  if(!is.null(maxlag)) {
    ymlag <- max(as.numeric(rownames(bmat)))
    if(maxlag>=ymlag) {
      if(cumul==F) {
        addmat <- matrix(0,nrow=maxlag-ymlag+1,ncol=3)
        } else {
        addmat <- matrix(bmat[nrow(bmat),],nrow=maxlag-ymlag+1,ncol=3,byrow=T)
        }
      bmat <- rbind(bmat,addmat)
      rownames(bmat) <- -1:(maxlag+1)
      } else {                
      bmat <- bmat[1:(which(rownames(bmat)==as.character(maxlag))+1),]
      }
    auxNZ <- which(bmat[,1]!=0)
    } else {
    auxNZ <- which(bmat[,1]!=0)
    if(cumul==F) {
      if(nrow(bmat)>max(auxNZ)+1) {
        bmat <- bmat[1:(max(auxNZ)+1),]
        }
      } else {                          
      auxCm <- min(intersect(which(diff(bmat[,1])==0)+1,auxNZ))
      if(nrow(bmat)>auxCm) {
        bmat <- bmat[1:auxCm,]
        }
      auxNZ <- which(bmat[,1]!=0)
      }
    }
  #xaux <- (1:nrow(bmat))-2
  xaux <- as.numeric(rownames(bmat))  #####
  xaux <- c(xaux,max(xaux)+1)
  #
  upLim <- 1.05*max(bmat)
  if(is.null(ylim)) {
    lowLim <- 1.05*min(bmat)
    upLim <- max(abs(c(upLim,lowLim)))
    lowLim <- -max(abs(c(upLim,lowLim)))
    } else {
    lowLim <- ylim[1]
    upLim <- ylim[2]
    }
  auxs <- which(bmat[,1]!=0)
  bmat_s <- bmat[c(max(1,min(auxs)-1):min(nrow(bmat),max(auxs)+1)),]
  bval <- as.numeric(rownames(bmat_s))
  xgrid <- sort(unique(c(bval,seq(min(bval),max(bval),length=100))))
  ygrid <- cbind(spline(bval,bmat_s[,1],xout=xgrid)$y,spline(bval,bmat_s[,2],xout=xgrid)$y,spline(bval,bmat_s[,3],xout=xgrid)$y)
  #####
  auxsgn <- sign(bmat[auxs,1])
  if(sum(auxsgn==-1)==0) {
    auxdel <- which(ygrid[,1]<0)
    } else if(sum(auxsgn==1)==0) {
    auxdel <- which(ygrid[,1]>0)
    } else {
    auxdel <- c()  
    }
  if(length(auxdel)>0) {
    auxInt <- c()
    for(i in 1:length(bval)) {
      auxInt[i] <- which(xgrid==bval[i])
      }
    for(i in 1:length(auxdel)) {
      isx <- auxInt[max(which(auxInt<=auxdel[i]))]
      idx <- auxInt[min(which(auxInt>=auxdel[i]))]
      ygrid[isx:idx,] <- NA
      }
    ygrid[c(1,nrow(ygrid)),] <- 0
    for(j in 1:ncol(ygrid)) {
      ygrid[,j] <- linImp(ygrid[,j])
      }
    }
  #####
  plot(0,type="n",xlim=c(min(xaux),max(xaux)),ylim=c(lowLim,upLim),yaxs="i",xaxs="i",cex.lab=1.2,
    lwd=2,xaxt="n",yaxt="n",xlab="Lag",ylab="Coefficient",main=title,cex.main=1.2) 
  if(cumul==T) mtext("cumulative lag shape",cex=0.9)
  polygon(c(xgrid,rev(xgrid)),c(ygrid[,2],rev(ygrid[,3])),border=NA,col="grey80")
  yaxaux <- seq(lowLim,upLim,length=21)
  ylabaux <- signif(yaxaux,3)
  ylabaux[11] <- 0
  xaxaux <- seq(min(xaux),max(xaux))
  auxby <- max(1,round((max(xaux)-min(xaux)+1)/30))
  xlabaux1 <- xlabaux2 <- seq(min(xaux),max(xaux),by=auxby)
  xlabaux2[c(1,length(xlabaux1))] <- NA
  abline(h=yaxaux,v=seq(min(xaux),max(xaux),by=auxby),col="grey75",lty=2)
  abline(h=0,lty=2,col="grey35")                                        
  lines(ygrid[,1]~xgrid,col="grey40",lty=2)
  #
  xpoi <- as.numeric(names(bmat[,1]))
  ypoi <- c()
  for(i in 1:length(xpoi)) {
    if(xpoi[i] %in% xgrid) ypoi[i] <- ygrid[which(xgrid==xpoi[i]),1]  #####
    }
  points(ypoi~xpoi,col="grey35",lty=2,cex=0.6)
  #
  axis(1,at=xlabaux1,labels=xlabaux2,cex.axis=1.1)
  axis(2,at=yaxaux,labels=ylabaux,cex.axis=1.1)
  confLeg <- paste("   ",conf*100,"% CI: (",bcum[2],", ",bcum[3],")",sep="")      
  if(max(bmat[,1])>0) {
    legpos <- "bottomright"
    } else {
    legpos <- "topright"
    }                                       
  est <- bmat[,1]
  if(cumul==T) {
    newest <- est[1]
    for(i in 2:length(est)) {
      newest[i] <- est[i]-est[i-1]
      } 
    est <- newest
    }
  minlag <- min(as.numeric(rownames(bmat)[which(est!=0)]))
  maxlag <- max(as.numeric(rownames(bmat)[which(est!=0)]))             
  legend(legpos,legend=c(paste("Effective lags: ",minlag," to ",maxlag,sep=""),paste("Cumulative coefficient: ",bcum[1],sep=""),confLeg),bty="n",cex=1.1)
  box()
  }

# plot the lag shape associated to an overall causal effect or a path
lagPlot <- function(x,from=NULL,to=NULL,path=NULL,maxlag=NULL,cumul=FALSE,conf=0.95,use.ns=FALSE,ylim=NULL,title=NULL) {
  if(("dlsem" %in% class(x))==F) stop("Argument 'x' must be an object of class 'dlsem'",call.=F)
  if(!is.null(maxlag) && (length(maxlag)!=1 || !is.numeric(maxlag) || maxlag<=0 || maxlag!=round(maxlag))) stop("Argument 'maxlag' must be a positive integer number",call.=F)
  if(length(cumul)!=1 || !is.logical(cumul)) stop("Arguent 'cumul' must be a logical value",call.=F)
  if(length(use.ns)!=1 || !is.logical(use.ns)) stop("Argument 'use.ns' must be a logical value",call.=F)
  if(!is.null(ylim) && (length(ylim)!=2 || ylim[1]>=ylim[2])) stop("Invalid argument 'ylim'",call.=F)
  #
  if(!is.null(from) && !is.na(from) && (is.null(to) & is.null(path))) {
    path <- from
    auxstr <- strsplit(path,"\\*")[[1]]
    if(length(auxstr)<2) {
      stop("Argument 'to' is missing",call.=F)
      } else {  
      from <- NULL
      }
    } else {
    if(is.null(from)||is.null(to)) {
      auxstr <- strsplit(path,"\\*")[[1]]
      if(length(auxstr)<2) stop("Invalid path length",call.=F)
      from <- to <- NULL
      } else {
      path <- NULL
      }
    }
  #
  if(is.null(path) && (is.null(from) || is.na(from))) stop("Argument 'from' is missing",call.=F)
  if(is.null(path) && (is.null(to) || is.na(to))) stop("Argument 'to' is missing",call.=F)
  xedgF <- edgeMat(x,conf=conf,full=T)
  xedg <- edgeMat(x,conf=conf,full=F)
  if(is.null(path)) {                        
    auxpa <- causalEff(x,from=from,to=to,lag=NULL,cumul=cumul,conf=conf,use.ns=use.ns)$overall
    } else {
    auxstr <- strsplit(path,"\\*")[[1]]
    pathchk <- setdiff(auxstr,names(x$estimate))
    if(length(pathchk)>0) stop("Unknown variable '",pathchk[1],"' in the path",call.=F)
    from <- auxstr[1]
    to <- rev(auxstr)[1] 
    isIn <- isInF <- 1
    for(i in 2:length(auxstr)) {
      isIn <- isIn*length(which(xedg[,1]==auxstr[i-1] & xedg[,2]==auxstr[i]))
      isInF <- isInF*length(which(xedgF[,1]==auxstr[i-1] & xedgF[,2]==auxstr[i]))
      }
    if((use.ns==T & isInF>0) | isIn>0) {
      auxpa <- causalEff(x,from=from,to=to,cumul=cumul,conf=conf,use.ns=use.ns)
      auxpa <- auxpa[[path]] 
      } else {
      #if(isInF>0) {
      #  stop("Path not found. Try to reduce 'conf' or to set 'use.ns' to TRUE",call.=F)
      #  } else {
      #  stop("Inexistent path",call.=F)
      #  }
      auxpa <- NULL
      }
    }     
  if(!is.null(auxpa)) {
    yaux <- rbind(rep(0,ncol(auxpa)),auxpa)
    rownames(yaux) <- c(-1:(nrow(yaux)-2))
    if(is.null(title)) {
      if(is.null(path)) {
        title <- paste(to," ~ ",from,sep="")
        } else {
        title <- paste(auxstr,collapse=" * ")
        }
      }
    bmat <- yaux[,c(1,3,4)]
    if(cumul==F) {
      auxbcum <- causalEff(x,from=from,to=to,cumul=T,conf=conf,use.ns=use.ns)$overall
      bcum <- signif(auxbcum[nrow(auxbcum),c(1,3,4)],5)
      } else {
      bcum <- signif(bmat[nrow(bmat),],5)
      }
    makeShape(bmat=bmat,maxlag=maxlag,cumul=cumul,bcum=bcum,conf=conf,ylim=ylim,title=title)
    } else {
    NULL  
    }
  }

# check if a vector is named (internal use only)
hasname <- function(x) {
  out <- sum(nchar(names(x))>0)==length(x)
  if(is.na(out)) out <- F
  out
  }
  
# check control options (internal use only)
controlCheck <- function(parstr,control,parSets,is.sign=F) {
  if(parstr %in% names(control)) {
    if(!is.list(control[[parstr]]) || hasname(control[[parstr]])==F) {
      stop("Component '",parstr,"' of argument 'control' is not a named list",call.=F)
      }
    for(i in 1:length(control[[parstr]])) {
      if(hasname(control[[parstr]][[i]])==F) stop("Component '",parstr,"' of argument 'control' is not a named list: ",names(control[[parstr]])[i],call.=F)
      }
    #auxch <- setdiff(names(control[[parstr]]),names(parSets))
    #if(length(auxch)>0) stop("Unknown variable in component '",parstr,"' of argument 'control': ",auxch[1],call.=F)
    auxG <- lapply(control[[parstr]],names)
    for(i in 1:length(control[[parstr]])) { 
      ival <- control[[parstr]][[i]]
      for(j in 1:length(ival)) {
        if(is.sign==F) {
          if(!is.numeric(ival[j]) || ival[j]<0 || ival[j]!=round(ival[j])) {
            stop("Invalid control options in component '",parstr,"' of argument 'control': ",names(control[[parstr]])[i]," | ",names(control[[parstr]][[i]])[j],call.=F)
            }
          } else {
          if((ival[j] %in% c("+","-"))==F) {
            stop("Invalid control options in component '",parstr,"' of argument 'control': ",names(control[[parstr]])[i]," | ",names(control[[parstr]][[i]])[j],call.=F)
            }
          }
        }
      }
    }
  }

# check missing values (internal use only)
checkNA <- function(x,group,data) {
  if(sum(!is.na(data[,x]))<3) stop("Variable '",x,"' has less than 3 observed values",call.=F)
  if(!is.null(group)) {
    gruppi <- levels(factor(data[,group]))
    for(i in 1:length(gruppi)) {
      auxind <- which(data[,group]==gruppi[i])
      if(sum(!is.na(data[auxind,x]))<1) {
        stop("Variable '",x,"' has no observed values in group '",gruppi[i],"'",call.=F)
        }
      }
    }
  }

# fit a dlsem
dlsem <- function(model.code,group=NULL,time=NULL,exogenous=NULL,data,log=FALSE,hac=FALSE,diff.options=list(combine="choi",k=0,maxdiff=3),
  imput.options=list(tol=0.0001,maxiter=500,recons=TRUE,no.imput=NULL),
  global.control=NULL,local.control=NULL,quiet=FALSE) {
  #
  if(!is.list(model.code) || length(model.code)==0 || sum(sapply(model.code,class)!="formula")>0) stop("Argument 'model code' must be a list of formulas",call.=F)
  if(!identical(class(data),"data.frame")) stop("Argument 'data' must be an object of class 'data.frame'",call.=F)  
  if(nrow(data)<3) stop("There must be at least 3 observations",call.=F)
  nameOfData <- deparse(substitute(data))
  if(!is.null(group) && length(group)!=1) stop("Argument 'group' must contain a single variable name",call.=F)
  if(!is.null(group) && is.na(group)) group <- NULL
  if(!is.null(group)) {
    if(length(group)!=1) stop("Argument 'group' must be of length 1",call.=F)
    if((group %in% colnames(data))==F) stop("Unknown variable '",group,"' provided to argument 'group'",call.=F)
    gruppi <- levels(factor(data[,group]))
    if(length(gruppi)<2) stop("The group factor must have at least 2 unique values",call.=F)
    data[,group] <- factor(data[,group])
    }
  if(!is.null(time)) {
    if(length(time)!=1) stop("Argument 'time' must be of length 1",call.=F)
    if((time %in% colnames(data))==F) stop("Unknown variable '",time,"' provided to argument 'time'",call.=F)
    if(!is.null(group)) {
      for(i in 1:length(gruppi)) {
        iind <- which(data[,group]==gruppi[i])
        idat <- data[iind,]
        data[iind,] <- idat[order(idat[,time]),]
        }
      } else {
      data <- data[order(data[,time]),]
      }
    }
  if(!is.null(exogenous) && identical(NA,exogenous)) exogenous <- NULL
  if(!is.null(group) && length(group)!=1) stop("Argument 'group' must contain a single variable name",call.=F)
  if(!is.null(group) && (group %in% colnames(data))==F) stop("Variable '",group,"' not found in data",sep="",call.=F)
  if(length(log)!=1 || !is.logical(log)) stop("Argument 'log' must be a logical value",call.=F)
  if(length(hac)!=1 || !is.logical(hac)) stop("Argument 'hac' must be a logical value",call.=F)
  if(length(quiet)!=1 || !is.logical(quiet)) stop("Argument 'quiet' must be a logical value",call.=F)
  if(!is.null(diff.options) && !is.list(diff.options)) stop("Argument 'diff.options' must be a list",call.=F)
  if(!is.null(imput.options) && !is.list(imput.options)) stop("Argument 'imput.options' must be a list",call.=F)
  if(!is.null(global.control) && !is.list(global.control)) stop("Argument 'global.control' must be a list",call.=F) 
  if(!is.null(local.control) && !is.list(local.control)) stop("Argument 'local.control' must be a list",call.=F)
  if(is.null(diff.options)) diff.options <- list(combine="choi",k=0,maxdiff=3)
  auxch <- setdiff(names(diff.options),c("combine","k","maxdiff"))
  if(length(auxch)>0) stop("Invalid component '",auxch[1],"' in argument 'diff.options'",sep="",call.=F)
  if("combine" %in% names(diff.options)) {
    combine <- diff.options$combine
    } else {
    combine <- "choi"
    }
  if("k" %in% names(diff.options)) {
    k <- diff.options$k
    } else {
    k <- 0
    }
  if("maxdiff" %in% names(diff.options)) {
    maxdiff <- diff.options$maxdiff
    } else {
    maxdiff <- 3
    }
  if(length(combine)!=1 || (combine %in% c("choi","demetrescu"))==F) stop("Argument 'combine' must be either 'choi' or 'demetrescu'",call.=F)
  if(length(maxdiff)!=1 || !is.numeric(maxdiff) || maxdiff<0 || maxdiff!=round(maxdiff)) stop("Argument 'maxdiff' must be a non-negative integer number",call.=F)
  if(!is.null(k)) {
    if(length(k)!=1 || !is.numeric(k) || k<0 || k!=round(k)) stop("Argument 'k' must either NULL or a non-negative integer",call.=F)
    }
  #
  if(is.null(imput.options)) imput.options <- list(tol=0.0001,maxiter=500,recons=T,no.imput=NULL)
  auxch <- setdiff(names(imput.options),c("tol","maxiter","recons","no.imput"))
  if(length(auxch)>0) stop("Invalid component '",auxch[1],"' in argument 'imput.options'",sep="",call.=F)
  if("tol" %in% names(imput.options)) {
    tol <- imput.options$tol
    } else {
    tol <- 0.0001
    }
  if("maxiter" %in% names(imput.options)) {
    maxiter <- imput.options$maxiter
    } else {
    maxiter <- 500
    }
  if("no.imput" %in% names(imput.options)) {
    no.imput <- imput.options$no.imput
    } else {
    no.imput <- NULL
    }
  if("recons" %in% names(imput.options)) {
    recons <- imput.options$recons
    } else {
    recons <- T
    }
  if(length(maxiter)!=1 || maxiter<0 || maxiter!=round(maxiter)) stop("Argument 'maxiter' must be a non-negative integer number",call.=F)
  if(length(tol)!=1 || tol<=0) stop("Argument 'tol' must be a positive real number",call.=F)              
  #
  if(is.null(global.control)) {
    global.control <- list(adapt=F,selection="bic")
    } else {
    if(("adapt" %in% names(global.control))==F) global.control$adapt <- F
    if(("selection" %in% names(global.control))==F) global.control$selection <- "bic"
    }                                              
  auxch <- setdiff(names(global.control),c("adapt","min.gestation","max.gestation","min.width","max.lead","sign","selection"))
  if(length(auxch)>0) stop("Invalid component '",auxch[1],"' in argument 'global.control'",sep="",call.=F)  
  if(!is.logical(global.control$adapt)) stop("Component 'adapt' in argument 'global.control' must be a logical value",call.=F)
  if((global.control$selection %in% c("aic","bic"))==F) stop("Component 'selection' in argument 'global.control' must be one among 'aic' and 'bic'",call.=F)
  if(!is.null(global.control$min.gestation) && (global.control$min.gestation<0 || global.control$min.gestation!=round(global.control$min.gestation))) {
    stop("Component 'min.gestation' in argument 'global.control' must be a non-negative integer value",call.=F)
    }
  if(!is.null(global.control$max.gestation) && (global.control$max.gestation<0 || global.control$max.gestation!=round(global.control$max.gestation))) {
    stop("Component 'max.gestation' in argument 'global.control' must be a non-negative integer value",call.=F)
    }
  if(!is.null(global.control$min.width) && (global.control$min.width<0 || global.control$min.width!=round(global.control$min.width))) {
    stop("Component 'min.width' in argument 'global.control' must be a non-negative integer value",call.=F)
    }
  if(!is.null(global.control$max.lead) && (global.control$max.lead<0 || global.control$max.lead!=round(global.control$max.lead))) {
    stop("Component 'max.lead' in argument 'global.control' must be a non-negative integer value",call.=F)
    }
  if(!is.null(global.control$sign) && ((global.control$sign %in% c("+","-"))==F)) {
    stop("Component 'sign' in argument 'global.control' must be either '+' or '-'",call.=F)
    }
  if(!is.null(global.control$min.width) && !is.null(global.control$max.lead) && global.control$min.width>global.control$max.lead) {
    stop("Component 'min.width' greater than component 'max.lead' in argument 'global.control'",call.=F)
    }
  if(!is.null(global.control$min.gestation) && !is.null(global.control$max.lead) && global.control$min.gestation>global.control$max.lead) {
    stop("Component 'min.gestation' greater than component 'max.lead' in argument 'global.control'",call.=F)
    }
  if(!is.null(global.control$min.gestation) && !is.null(global.control$max.gestation) && global.control$min.gestation>global.control$max.gestation) {
    stop("Component 'min.gestation' greater than component 'max.gestation' in argument 'global.control'",call.=F)
    }
  #
  rownames(data) <- 1:nrow(data)
  res <- pset <- list()
  messlen <- 0
  for(i in 1:length(model.code)) {
    if(sum(grepl("\\-",model.code[[i]]))>0) stop("Invalid character '-' in 'model.code', regression for '",model.code[[i]][2],"'",call.=F)
    if(sum(grepl("\\:",model.code[[i]]))>0) stop("Invalid character ':' in 'model.code', regression for '",model.code[[i]][2],"'",call.=F) #####
    if(sum(grepl("\\*",model.code[[i]]))>0) stop("Invalid character '*' in 'model.code', regression for '",model.code[[i]][2],"'",call.=F) #####
    }
  anygamma <- 0
  for(i in 1:length(model.code)) {
    icheck <- scanForm(model.code[[i]])
    ilagpar <- icheck$lpar
    for(j in 1:length(ilagpar)) {
      if(!is.null(icheck$ltype[[j]]) && icheck$ltype[[j]]!="none") {
        if(sum(is.na(ilagpar[[j]]))>0) stop("Missing argument in ",icheck$y," ~ ",icheck$X[[j]],call.=F)
        if(icheck$ltype[[j]]=="gamm.lag") anygamma <- 1
        }
      }
    ipar <- names(ilagpar)
    if(length(ipar)>0) {
      if(!is.null(group)) {
        if(group %in% ipar) stop("Variable '",group,"' is defined as a group factor and appears in 'model.code'",call.=F) 
        }
      auxfun <- setdiff(ipar,colnames(data))
      if(length(auxfun)>0) {
        if(identical(substr(auxfun[1],1,11),"uncons.lag(")) {
          stop("Unconstrained lag shape not allowed in 'model.code', regression for '",model.code[[i]][2],"'",call.=F)
          } else if(identical(substr(auxfun[1],1,10),"almon.lag(")) {
          stop("Almon's lag shape not allowed in 'model.code', regression for '",model.code[[i]][2],"'",call.=F)
          } else {
          stop("Invalid expression for '",icheck$y,"': ",auxfun[1],call.=F) #####
          }
        }
      auxexo <- intersect(ipar,exogenous)
      if(length(auxexo)>0) stop("Variable '",auxexo[1],"' appears both in 'model.code' and in 'exogenous'",call.=F)
      auxdupl <- duplicated(ipar)
      if(sum(auxdupl)>0) stop("Duplicated covariate '",ipar[auxdupl][1],"' in 'model.code', regression for '",model.code[[i]][2],"'",call.=F)
      }
    auxstr <- as.character(model.code[[i]])[-1]
    ynam <- gsub(" ","",auxstr[1])
    names(model.code)[i] <- ynam
    xnam <- gsub(" ","",strsplit(auxstr[2],"\\+")[[1]])  
    xnam <- xnam[which(nchar(xnam)>0)]
    for(j in 1:length(xnam)) {
      if(grepl("\\(",xnam[j])) xnam[j] <- strsplit(strsplit(xnam[j],",")[[1]][1],"\\(")[[1]][2]
      }
    if(identical(xnam,"1") | length(xnam)==0) {
      pset[[i]] <- character(0)
      } else {
      pset[[i]] <- xnam
      }
    names(pset)[i] <- ynam 
    }   
  codnam <- c()
  for(i in 1:length(model.code)) {
    codnam[i] <- as.character(model.code[[i]])[2]
    }
  auxdupl <- duplicated(codnam)
  if(sum(auxdupl)>0) stop("Duplicated response variable '",codnam[auxdupl][1],"' in 'model.code'",call.=F)
  nodenam <- unique(c(names(pset),unlist(pset)))
  auxadd <- setdiff(nodenam,codnam)
  if(length(auxadd)>0) {
    for(i in length(auxadd):1) {
      model.code <- c(formula(paste(auxadd[i],"~1",sep="")),model.code)
      names(model.code)[1] <- auxadd[i]
      pset[[length(pset)+1]] <- character(0) 
      names(pset)[length(pset)] <- auxadd[i]
      }
    }
  if(length(nodenam)<2) stop("The model cannot contain less than 2 endogenous variables",call.=F)
  auxvar <- setdiff(c(nodenam,exogenous),colnames(data))
  if(length(auxvar)>0) {
    stop("Variable '",auxvar[1],"' not found in data",sep="",call.=F)
    }
  for(i in 1:length(nodenam)) {
    if(isQuant(data[,nodenam[i]])==F) stop("Qualitative variables cannot appear in 'model.code': ",nodenam[i],sep="",call.=F)
    }
  if(!is.null(exogenous)) {
    for(i in 1:length(exogenous)) {
      if(isQuant(data[,exogenous[i]])==F) {
        if(sum(is.na(data[,exogenous[i]]))>0) stop("Qualitative variables cannot contain missing values: ",exogenous[i],sep="",call.=F)
        }
      }                                                                                                                                                                          
    }
  G <- new("graphNEL",nodes=names(pset),edgemode="directed")    
  for(i in 1:length(pset)) {
    if(length(pset[[i]])>0) {
      for(j in 1:length(pset[[i]])) {
        G <- addEdge(pset[[i]][j],names(pset)[i],G,1) 
        }
      }
    }          
  topG <- topOrder(G)
  if(is.null(topG)) stop("The DAG contains directed cycles",call.=F)  
  auxch <- setdiff(no.imput,topG)
  if(length(auxch)>0) stop("Unknown variable '",auxch[1],"' in component 'no.imput' of argument 'imput.options'",call.=F)  
  auxch <- setdiff(names(local.control),c("adapt","min.gestation","max.gestation","min.width","max.lead","sign"))  #,"L"
  if(length(auxch)>0) stop("Unknown component '",auxch[1],"' in argument 'local.control'",call.=F)
  if("adapt" %in% names(local.control)) {
    if(!is.logical(local.control[["adapt"]]) || hasname(local.control[["adapt"]])==F) stop("Component 'adapt' of argument 'local.control' must be a named logical vector",call.=F)
    auxch <- setdiff(names(local.control[["adapt"]]),topG)
    if(length(auxch)>0) stop("Unknown variables in component 'adapt' of argument 'local.control': ",paste(auxch,collapse=", "),call.=F)
    }
  if("min.gestation" %in% names(local.control)) controlCheck("min.gestation",local.control,pset)
  if("max.gestation" %in% names(local.control)) controlCheck("max.gestation",local.control,pset)
  if("min.width" %in% names(local.control)) controlCheck("min.width",local.control,pset)
  if("max.lead" %in% names(local.control)) controlCheck("max.lead",local.control,pset)
  if("sign" %in% names(local.control)) controlCheck("sign",local.control,pset,is.sign=T)
  auxch <- intersect(names(local.control[["min.width"]]),names(local.control[["max.lead"]]))  
  if(length(auxch)>0) {
    for(i in 1:length(auxch)) {
      iming <- local.control[["min.gestation"]][[auxch[i]]]
      imaxg <- local.control[["max.gestation"]][[auxch[i]]]
      iminw <- local.control[["min.width"]][[auxch[i]]]
      imaxw <- local.control[["max.lead"]][[auxch[i]]]
      ich1 <- intersect(names(iminw),names(imaxw))
      if(length(ich1)>0) {
        for(j in 1:length(ich1)) {
          if(iminw[ich1[j]]>imaxw[ich1[j]]) stop("Component 'min.width' greater than component 'max.lead' in argument 'local.control': ",auxch[i],"~",ich1[j],call.=F)
          }
        }
      ich2 <- intersect(names(iming),names(imaxw))
      if(length(ich2)>0) {
        for(j in 1:length(ich2)) {
          if(iming[ich2[j]]>imaxw[ich2[j]]) stop("Component 'min.gestation' greater than component 'max.lead' in argument 'local.control': ",auxch[i],"~",ich2[j],call.=F)
          }
        }
      ich3 <- intersect(names(iming),names(imaxg))
      if(length(ich3)>0) {
        for(j in 1:length(ich3)) {
          if(iming[ich3[j]]>imaxg[ich3[j]]) stop("Component 'min.gestation' greater than component 'max.gestation' in argument 'local.control': ",auxch[i],"~",ich3[j],call.=F)
          }
        }
      }
    }
  nodenam <- c(exogenous,topG)
  xfact <- c()
  for(i in 1:length(nodenam)) {
    if(isQuant(data[,nodenam[i]])==F) {
      xfact <- c(xfact,nodenam[i])
      data[,nodenam[i]] <- factor(data[,nodenam[i]])
      }
    }
  origdat <- data
  if(log==T) {
    nolog <- xfact
    logtest <- setdiff(nodenam,nolog)            
    if(length(logtest)>0) {
      for(i in 1:length(logtest)) {
        if(sum(data[,logtest[i]]<=0,na.rm=T)>0) {
          nolog <- c(nolog,logtest[i])        
          } else {
          data[,logtest[i]] <- log(data[,logtest[i]])
          }
        }
      }
    if(quiet==F) if(length(nolog)>0) cat("Logarithm not applied to variables: ",paste(nolog,collapse=", "),sep="","\n")
    }
  difftest <- setdiff(nodenam,xfact)
  if(recons==T) {
    x2recons <- setdiff(difftest,no.imput)
    if(length(x2recons)>0) data <- splinImp(x2recons,group,data)
    }
  ndiff <- 0
  if(length(difftest)>0 & maxdiff>0) {
    fine <- 0
    if(quiet==F) cat("Checking unit root...")
    flush.console()
    while(fine==0) {
      auxp <- c()
      urtList <- unirootTest(difftest,group=group,data=data,combine=combine,k=k)
      for(i in 1:length(difftest)) {
        ipvl <- urtList[[i]]$p.value
        if(is.null(ipvl)) {
          auxp[i] <- 0
          } else {
          if(is.na(ipvl)) {
            auxp[i] <- 0
            } else {
            auxp[i] <- ipvl
            }
          }
        }
      nUR <- length(which(auxp>0.05))
      if(nUR>0) {
        if(ndiff<maxdiff) {
          ndiff <- ndiff+1
          data <- applyDiff(x=difftest,group=group,data=data,k=rep(1,length(difftest)))                                        
          } else {
          fine <- 1
          }
        } else {
        fine <- 1
        }
      }
    } else {
    urtList <- NULL  
    }
  if(quiet==F) {
    if(ndiff==0) {
      cat('\r',"No differentiation performed")
      } else {
      cat('\r',"Order",ndiff,"differentiation performed")    
      }
    cat("\n")
    }
  if(is.null(group)) {
    nK <- length(nodenam)+1
    } else {
    nK <- length(nodenam)+nlevels(factor(data[,group]))
    }
  auxna <- apply(data[,nodenam],1,function(x){sum(is.na(x))})
  if(sum(auxna)>0) {
    auxOK <- unname(which(auxna<length(nodenam))) 
    if(ndiff>0) {
      if(is.null(group)) {
        auxind <- 1:ndiff
        } else {
        auxind <- c()
        for(i in 1:length(gruppi)) {
          auxind <- c(auxind,which(data[,group]==gruppi[i])[1:ndiff])          
          }
        }                              
      auxOK <- setdiff(auxOK,auxind)                    
      }
    if(sum(is.na(data[auxOK,]))>0 && maxiter>0) {
      x2imp <- setdiff(nodenam,c(no.imput,xfact))
      if(length(x2imp)>0) {
        for(i in 1:length(x2imp)) {
          nachk <- checkNA(x2imp[i],group,data[auxOK,])
          }
        nIm <- sum(is.na(data[,x2imp]))
        if(nIm>0) {
          data[auxOK,c(group,xfact,x2imp)] <- EM.imputation(xcont=x2imp,xqual=xfact,group=group,data=data[auxOK,c(group,xfact,x2imp),drop=F],tol=tol,maxiter=maxiter,quiet=quiet)
          }
        }
      }
    }
  nomi <- c()
  optList <- list()
  if(quiet==F) cat("Starting estimation...")
  flush.console()
  if(anygamma==1) glTab <- gammaTabFun() else glTab <- NULL  #####
  for(i in 1:length(model.code)) {
    nomi[i] <- as.character(model.code[[i]])[2]
    auxmess <- paste("Estimating regression model ",i,"/",length(model.code)," (",nomi[i],")",sep="")
    auxdel <- messlen-nchar(auxmess)+1
    if(quiet==F) {
      if(auxdel>0) {
        cat('\r',auxmess,rep(" ",auxdel))
        } else {
        cat('\r',auxmess)
        }
      flush.console()
      }
    if(is.null(exogenous)) {
      iform <- model.code[[i]]
      } else {
      iform <- formula(paste(as.character(model.code[[i]])[2],"~",paste(exogenous,collapse="+"),"+",as.character(model.code[[i]])[3],sep=""))
      }       
    iad <- global.control$adapt
    if("adapt" %in% names(local.control)) {
      if(nomi[i] %in% names(local.control[["adapt"]])) {
        iad <- local.control[["adapt"]][nomi[i]]
        }
      }
    if(length(pset[[nomi[i]]])>0) {
      #if(!is.null(global.control$L)) {
      #  iL <- rep(global.control$L,length(pset[[nomi[i]]]))
      #  names(iL) <- pset[[nomi[i]]]
      #  } else {
      #  iL <- c()
      #  }
      if(!is.null(global.control$min.gestation)) {
        iming <- rep(global.control$min.gestation,length(pset[[nomi[i]]]))
        names(iming) <- pset[[nomi[i]]]
        } else {
        iming <- c()
        }
      if(!is.null(global.control$max.gestation)) {
        iges <- rep(global.control$max.gestation,length(pset[[nomi[i]]]))
        names(iges) <- pset[[nomi[i]]]
        } else {
        iges <- c()
        }
      if(!is.null(global.control$min.width)) {
        iwd <- rep(global.control$min.width,length(pset[[nomi[i]]]))
        names(iwd) <- pset[[nomi[i]]]
        } else {
        iwd <- c()
        }      
      if(!is.null(global.control$max.lead)) {
        ilead <- rep(global.control$max.lead,length(pset[[nomi[i]]]))       
        names(ilead) <- pset[[nomi[i]]]
        } else {
        ilead <- c()
        }
      if(!is.null(global.control$sign)) {
        isg <- rep(global.control$sign,length(pset[[nomi[i]]]))
        names(isg) <- pset[[nomi[i]]]
        } else {
        isg <- c()
        }      
      } else {
      iming <- iges <- iwd <- ilead <- isg <- c()
      #iL <- c()
      }   
    #if("L" %in% names(local.control)) {
    #  if(nomi[i] %in% names(local.control[["L"]])) {
    #    iauxL <- local.control[["L"]][nomi[i]]
    #    iL[names(iauxL)] <- iauxL
    #    }
    #  }
    if("min.gestation" %in% names(local.control)) {
      if(nomi[i] %in% names(local.control[["min.gestation"]])) {
        iauxming <- local.control[["min.gestation"]][[nomi[i]]]
        iming[names(iauxming)] <- iauxming
        }
      }
    if("max.gestation" %in% names(local.control)) {
      if(nomi[i] %in% names(local.control[["max.gestation"]])) {
        iauxges <- local.control[["max.gestation"]][[nomi[i]]]
        iges[names(iauxges)] <- iauxges
        }
      }
    if("min.width" %in% names(local.control)) {
      if(nomi[i] %in% names(local.control[["min.width"]])) {
        iauxwd <- local.control[["min.width"]][[nomi[i]]]
        iwd[names(iauxwd)] <- iauxwd
        }
      }
    if("max.lead" %in% names(local.control)) {
      if(nomi[i] %in% names(local.control[["max.lead"]])) {
        iauxlead <- local.control[["max.lead"]][[nomi[i]]]
        ilead[names(iauxlead)] <- iauxlead
        }
      }
    if("sign" %in% names(local.control)) {
      if(nomi[i] %in% names(local.control[["sign"]])) {
        iauxsg <- local.control[["sign"]][[nomi[i]]]
        isg[names(iauxsg)] <- iauxsg
        }
      }
    optList[[i]] <- list(adapt=iad,min.gestation=iming,max.gestation=iges,min.width=iwd,max.lead=ilead,sign=isg)
    imod <- dlaglm(iform,group=group,data=data,adapt=iad,no.select=exogenous,min.gestation=iming,max.gestation=iges,min.width=iwd,max.lead=ilead,sign=isg,selection=global.control$selection,ndiff=ndiff,gammaTab=glTab)  ### L=iL,
    if(hac==T) {
      imod$vcov <- vcovHAC(imod,group=group)
      class(imod) <- c("hac","lm")
      }
    res[[i]] <- imod
    messlen <- nchar(auxmess)
    }
  names(optList) <- nomi
  auxmess <- "Estimation completed"
  auxdel <- messlen-nchar(auxmess)+1
  if(quiet==F) {
    if(auxdel>0) {
      cat('\r',auxmess,rep(" ",auxdel),sep="")
      } else {
      cat('\r',auxmess)
      }
    cat("\n")
    }
  names(res) <- nomi
  callList <- lapply(res,function(z){z$call})
  out <- list(estimate=res[topG],call=callList,exogenous=exogenous,group=group,log=log,ndiff=ndiff,
    diff.options=diff.options,imput.options=imput.options,selection=global.control$selection,adaptation=optList,
    Rsq=RsqCalc(res[topG]),data.orig=origdat[,c(group,time,nodenam)],data.used=data[,c(group,time,nodenam)])
  class(out) <- "dlsem"
  out
  }

# automated plots of lag shapes
auto.lagPlot <- function(x,cumul=FALSE,conf=0.95,plotDir=NULL) {
  if(("dlsem" %in% class(x))==F) stop("Argument 'x' must be an object of class 'dlsem'",call.=F)
  if(length(cumul)!=1 || !is.logical(cumul)) stop("Arguent 'cumul' must be a logical value",call.=F)
  if(length(conf)!=1 || !is.numeric(conf) || conf<=0 || conf>=1) stop("Arguent 'conf' must be a real number greater than 0 and less than 1",call.=F)
  if(is.null(plotDir)) plotDir <- getwd()
  for(i in 1:length(x$estimate)) {  
    scan0 <- scanForm(x$estimate[[i]]$call$formula)
    inam <- scan0$y
    isumm <- summary(x$estimate[[inam]])$coefficients
    ilagged <- names(scan0$ltype)[which(scan0$ltype!="none")]
    ilab <- sapply(rownames(isumm),extrName)
    if(length(ilagged)>0) {                                  
      for(j in 1:length(ilagged)) {
        ijlab <- names(ilab)[which(ilab==ilagged[j])]
        if(isumm[ijlab,4]<=1-conf) {
          pdf(file.path(plotDir,paste(inam,"~",ilagged[j],".pdf",sep="")))
          lagPlot(x,path=paste(ilagged[j],"*",inam,sep=""),cumul=cumul,conf=conf)
          dev.off()
          }
        }
      }
    }
  cat("Plots saved in ",plotDir,"\n",sep="")
  }

# print method for class dlsem
print.dlsem <- function(x,...) {
  cat("A distributed-lag linear structural equation model","\n")
  n.e <- sum(sapply(chldsets(makeGraph(x)$graph),length))
  N.e <- sum(sapply(chldsets(makeGraph(x)$full.graph),length))
  if(!is.null(x$group)) {
    cat(" Number of groups: ",nlevels(x$data.used[,x$group]),"\n",sep="")
    } else {
    cat(" No groups","\n")
    }
  cat(" Number of endogenous variables: ",length(x$estimate),"\n",sep="")
  if(!is.null(x$exogenous)) {
    cat(" Number of exogenous variables: ",length(x$exogenous),"\n",sep="")
    } else {
    cat(" No exogenous variables","\n")
    }
  if(n.e>0) {
    cat(" ",n.e,"/",N.e," significant edges at 5% level","\n",sep="")
    } else {
    cat(" No significant edges at 5% level","\n")  
    }
  }

# format summary table (internal use only)
formatSumm <- function(x,newnam) {
  if(newnam==T) {
    colnam <- c("theta","se(theta)","t value","Pr(>|t|)","")
    } else {
    colnam <- c(colnames(x),"")
    }
  auxp <- rep("",nrow(x))
  pval <- x[,ncol(x)]
  auxp[which(pval<0.1)] <- "."
  auxp[which(pval<0.05)] <- "*"
  auxp[which(pval<0.01)] <- "**"
  auxp[which(pval<0.001)] <- "***"
  res <- data.frame(x,auxp)
  colnames(res) <- colnam
  res
  }

# summary method for class dlsem
summary.dlsem <- function(object,...) {
  elev <- glev <- c()
  enam <- object$exogenous 
  if(!is.null(enam)) {
    for(i in 1:length(enam)) {
      if(!is.factor(object$data.used[,enam[i]])) {
        elev <- c(elev,enam[i]) 
        } else {
        elev <- c(elev,paste(enam[i],levels(object$data.used[,enam[i]]),sep=""))
        }
      }
    }
  if(!is.null(object$group)) {
    glev <- paste(object$group,levels(object$data.used[,object$group]),sep="")
    }
  estim <- object$estimate
  fitI <- c()
  fitI[1] <- RsqCalc(estim)["(overall)"]
  fitI[2] <- AIC(object)["(overall)"]
  fitI[3] <- BIC(object)["(overall)"]
  names(fitI) <- c("Rsq","AIC","BIC")
  summList <- summList_e <- summList_g <- vector("list",length=length(estim))
  names(summList) <- names(summList_e) <- names(summList_g) <- names(estim)
  summS <- matrix(nrow=length(estim),ncol=2)
  rownames(summS) <- names(estim)
  colnames(summS) <- c("Std. Dev.","df")
  for(i in 1:length(estim)) {
    iB <- summary(estim[[i]])$coefficients
    inam <- setdiff(rownames(iB),c("(Intercept)",elev,glev))
    if(length(inam)>0) {
      isumm <- formatSumm(iB[inam,,drop=F],newnam=F)
      ilag <- lagspan(estim[[i]]$call)[gsub(" ","",inam),,drop=F]
      iendo <- cbind(ilag,isumm)
      rownames(iendo) <- inam
      colnames(iendo) <- c(colnames(ilag),colnames(isumm))
      summList[[i]] <- iendo
      }
    if(length(elev)>0) {
      elevOK <- intersect(rownames(iB),elev)
      if(length(elevOK)>0) {
        summList_e[[i]] <- formatSumm(iB[elevOK,,drop=F],newnam=F)
        } else {
        #summList_e[[i]] <- NULL
        }
      }
    if(length(glev)>0) {
      summList_g[[i]] <- formatSumm(iB[intersect(glev,rownames(iB)),,drop=F],newnam=F)   
      } else {
      summList_g[[i]] <- formatSumm(iB["(Intercept)",,drop=F],newnam=F)          
      }
    summS[i,] <- c(summary(estim[[i]])$sigma,estim[[i]]$df.residual)
    }
  OUT <- list(endogenous=summList,exogenous=summList_e,group=summList_g,errors=summS,gof=fitI)
  class(OUT) <- "summary.dlsem"
  OUT
  }

# print method for class summary.dlsem
print.summary.dlsem <- function(x,...) {
  cat("ENDOGENOUS PART","\n","\n")
  for(i in 1:length(x$endogenous)) {
    cat("Response: ",names(x$endogenous)[i],sep="","\n")
    isumm <- x$endogenous[[i]]
    if(!is.null(isumm)) {
      print(isumm)
      } else {
      cat("-","\n")
      }
    if(i<length(x$endogenous)) cat("\n")
    }
  cat("\n","\n")
  cat("EXOGENOUS PART","\n")
  if(sum(sapply(x$exogenous,is.null))==0) {
    cat("\n")
    for(i in 1:length(x$exogenous)) {
      cat("Response: ",names(x$exogenous)[i],sep="","\n")
      print(x$exogenous[[i]])
      cat("\n")
      }
    } else {
    cat(" -","\n","\n")
    }
  fitI <- x$gof
  cat("\n")
  cat("INTERCEPTS","\n")
  if(sum(sapply(x$group,is.null))==0) {
    cat("\n")
    for(i in 1:length(x$group)) {
      cat("Response: ",names(x$group)[i],sep="","\n")
      print(x$group[[i]])
      cat("\n")
      }
    } else {
    cat(" -","\n","\n")
    }
  fitI <- x$gof
  cat("\n")
  cat("ERRORS","\n")
  print(x$errors)
  fitI <- x$gof
  cat("\n","\n")
  cat("GOODNESS OF FIT","\n","\n")
  cat("R-squared: ",round(fitI[1],4),"\n",sep="")
  cat("AIC: ",fitI[2],"\n",sep="")
  cat("BIC: ",fitI[3],"\n",sep="")
  }

# nobs method for class dlsem
nobs.dlsem <- function(object,...) {
  sapply(object$estimate,nobs)
  }

# coef method for class dlsem
coef.dlsem <- function(object,...) {
  lapply(object$estimate,coef)
  }

# vcov method for class dlsem
vcov.dlsem <- function(object,...) {
  lapply(object$estimate,vcov)
  }

# logLik method for class dlsem
logLik.dlsem <- function(object,...) {
  lapply(object$estimate,logLik)
  }

# AIC method for class dlsem
AIC.dlsem <- function(object,...) {
  res <- sapply(object$estimate,AIC)
  OUT <- c(res,sum(res))
  names(OUT) <- c(names(res),"(overall)")
  OUT
  }

# BIC method for class dlsem
BIC.dlsem <- function(object,...) {
  res <- sapply(object$estimate,BIC)
  OUT <- c(res,sum(res))
  names(OUT) <- c(names(res),"(overall)")
  OUT
  }

# extract the number of parameters of an object of class dlsem
npar <- function(x) {
  if(("dlsem" %in% class(x))==F) stop("Argument 'x' must be an object of class 'dlsem'",call.=F)
  sapply(x$estimate,function(z){attributes(logLik(z))$df})
  }

# extract gestation and lead lags (internal use only)
lagspan <- function(x) {
  iscan <- scanForm(x)
  if(length(iscan$X)>0) {
    itype <- iscan$ltype
    inam <- names(itype)
    ires <- matrix(nrow=length(itype),ncol=2)
    colnames(ires) <- c("a","b")
    rownames(ires) <- iscan$X #inam
    for(j in 1:length(inam)) {
      if(itype[inam[j]]=="none") {
        ires[j,] <- c(0,0)
        } else if(itype[inam[j]]=="gamm.lag") {
        ires[j,] <- gamlim(iscan$lpar[[inam[j]]][1],iscan$lpar[[inam[j]]][2])
        } else {
        ires[j,] <- iscan$lpar[[inam[j]]]
        }
      }
    ires
    }
  }

# format fitted or residuals of an object of class dlsem (internal use only)
formatFit <- function(object,pred) {
  ind <- lapply(pred,names)
  maxind <- max(as.numeric(unlist(ind)))
  res <- matrix(nrow=maxind,ncol=length(pred))
  colnames(res) <- names(pred)
  rownames(res) <- 1:maxind
  for(i in 1:length(pred)) {
    inam <- names(pred[[i]])
    for(j in 1:length(inam)) {      
      res[as.numeric(inam[j]),i] <- pred[[i]][inam[j]]
      }
    }
  if(!is.null(object$group)) {
    out <- data.frame(object$data.orig[,object$group],res)
    colnames(out)[1] <- object$group
    } else {
    out <- data.frame(res)
    }
  out
  }

# fitted method for class dlsem
fitted.dlsem <- function(object,...) {
  pred <- list()
  for(i in 1:length(object$estimate)) {
    pred[[i]] <- fitted(object$estimate[[i]],...)
    }
  names(pred) <- names(object$estimate)
  formatFit(object,pred)
  }

# residuals method for class dlsem
residuals.dlsem <- function(object,...) {
  pred <- list()
  for(i in 1:length(object$estimate)) {
    pred[[i]] <- residuals(object$estimate[[i]],...)
    }
  names(pred) <- names(object$estimate)
  formatFit(object,pred)
  }

# predict method for class dlsem
predict.dlsem <- function(object,newdata=NULL,...) {  
  pred <- list()
  if(is.null(newdata)) {
    for(i in 1:length(object$estimate)) {
      pred[[i]] <- predict(object$estimate[[i]],...)
      }
    if(!is.null(object$group)) {
      igrou <- as.character(object$data.used[,object$group])
      names(igrou) <- rownames(object$data.used)
      pred <- c(list(igrou),pred)
      }
    names(pred) <- c(object$group,names(object$estimate))
    } else {
    Z <- newdata
    if(ncol(Z)>0) {
      Xq <- c()
      for(i in 1:ncol(Z)) {
        if(isQuant(Z[,i])) Xq <- c(Xq,colnames(Z)[i])
        }
      if(length(Xq)>0) {
        if(object$log==T) {
          for(i in 1:length(Xq)) {
            if(sum(Z[,Xq[i]]<=0,na.rm=T)==0) Z[,Xq[i]] <- log(Z[,Xq[i]])
            }
          }
        if(object$ndiff>0) Z <- applyDiff(x=Xq,group=object$group,data=Z,k=rep(object$ndiff,length(Xq))) 
        }
      }
    #G <- makeGraph(object)$full.graph
    #nomi <- topOrder(G)
    nomi <- names(object$estimate)
    #
    pred <- list()
    for(i in 1:length(nomi)) {
      iP <- predict(object$estimate[[nomi[i]]],newdata=Z,...)
      pred[[i]] <- iP
      }
    if(!is.null(object$group)) {
      igrou <- as.character(Z[,object$group])  ####
      names(igrou) <- rownames(Z)
      pred <- c(list(igrou),pred)
      }    
    names(pred) <- c(object$group,nomi)
    }
  auxind <- lapply(pred,names)
  ind <- sort(unique(as.numeric(unlist(auxind))))
  res <- data.frame(matrix(nrow=length(ind),ncol=length(pred)))
  colnames(res) <- names(pred)
  rownames(res) <- ind
  for(i in 1:length(pred)) {
    inam <- names(pred[[i]])
    for(j in 1:length(inam)) {
      res[inam[j],i] <- pred[[i]][inam[j]]
      }
    }
  if(!is.null(object$group)) {
    res[,object$group] <- factor(res[,object$group],levels=levels(object$data.used[,object$group]))
    }
  res
  }

# compute R-squared (internal use only)
RsqCalc <- function(xfit) {
  Rsq <- n <- c()
  for(i in 1:length(xfit)) {
    n[i] <- nobs(xfit[[i]])
    Rsq[i] <- summary(xfit[[i]])$'r.squared'
    }
  OUT <- c(Rsq,sum(Rsq*n)/sum(n))
  names(OUT) <- c(names(xfit),"(overall)")
  OUT
  }

# create graph object from dlsem (internal use only)
makeGraph <- function(x,conf=0.95) {
  if(("dlsem" %in% class(x))==F) stop("Argument 'x' must be an object of class 'dlsem'",call.=F)
  if(length(conf)!=1 || !is.numeric(conf) || conf<=0 || conf>=1) stop("Arguent 'conf' must be a real number greater than 0 and less than 1",call.=F)
  nomi <- names(x$estimate)
  G0 <- G <- new("graphNEL",nodes=nomi,edgemode="directed")
  eSign <- c()
  for(i in 1:length(nomi)) {
    isumm <- summary(x$estimate[[nomi[i]]])$coefficients
    auxnam <- rownames(isumm)
    for(j in 1:length(auxnam)) {
      auxsg <- sign(isumm[auxnam[j],1])
      ijnam <- extrName(auxnam[j])
      if(ijnam %in% nomi) {        
        G0 <- addEdge(ijnam,nomi[i],G0,1)
        if(isumm[auxnam[j],4]<1-conf) {
          G <- addEdge(ijnam,nomi[i],G,1)
          if(auxsg>0) {
            eSign <- c(eSign,"+")
            } else {
            eSign <- c(eSign,"-")
            }
          } else {
          eSign <- c(eSign,"")
          }
        names(eSign)[length(eSign)] <- paste(ijnam,"~",nomi[i],sep="")              
        }
      }
    }
  list(graph=G,full.graph=G0,sign=eSign)
  }

# convert into class 'graphNEL'
as.graphNEL <- function(x,conf=0.95,use.ns=FALSE) {
  if(use.ns==F) {
    makeGraph(x,conf=conf)$graph
    } else {
    makeGraph(x,conf=conf)$full.graph
    }
  }

# compute the coefficient associated to each edge at different time lags (internal use only)
edgeCoeff <- function(x,lag=NULL,conf=0.95) {
  nomi <- names(x$estimate)
  laglen <- c()
  for(i in 1:length(nomi)) {
    isumm <- summary(x$estimate[[nomi[i]]])$coefficients
    auxnam <- rownames(isumm)
    for(j in 1:length(auxnam)) {
      ijnam <- extrName(auxnam[j])
      if(ijnam %in% nomi) {
        cumb <- lagEff(model=x$estimate[[nomi[i]]],x=ijnam,cumul=F,conf=conf,lag=NULL)
        laglen <- c(laglen,rownames(cumb)[nrow(cumb)])
        }
      }
    }
  meL <- max(as.numeric(laglen))
  lagOK <- sort(unique(c(lag,0:meL)))
  bList <- vector("list",length=length(lagOK))
  for(i in 1:length(nomi)) {
    isumm <- summary(x$estimate[[nomi[i]]])$coefficients
    auxnam <- rownames(isumm)
    for(j in 1:length(auxnam)) {
      ijnam <- extrName(auxnam[j])
      if(ijnam %in% nomi) {
        for(w in 1:length(lagOK)) {
          bList[[w]] <- rbind(bList[[w]],lagEff(model=x$estimate[[nomi[i]]],x=ijnam,cumul=F,conf=conf,lag=lagOK[w]))
          rownames(bList[[w]])[nrow(bList[[w]])] <- paste(ijnam,"~",nomi[i],sep="")
          }
        }
      }
    if(!is.null(bList)) names(bList) <- lagOK
    }
  for(i in 1:length(bList)) {
    auxnam <- rownames(bList[[i]])
    newnam <- c()
    for(j in 1:length(auxnam)) {
      newnam[j] <- paste(rev(strsplit(auxnam[j],"~")[[1]]),collapse="~")
      }
    rownames(bList[[i]]) <- newnam
    }
  for(i in 1:length(bList)) {
    colnames(bList[[i]]) <- c("estimate",paste(c("lower ","upper "),conf*100,"%",sep=""))
    }
  if(is.null(lag)) {
    bList
    } else {
    voidB <- bList[[1]]
    voidB[] <- 0
    bList2 <- vector("list",length=length(lag))
    names(bList2) <- lag
    for(i in 1:length(bList2)) {
      if(lag[i]<=max(lagOK)) {
        bList2[[i]] <- bList[[as.character(lag[i])]]
        } else {
        bList2[[i]] <- voidB
        }
      }
    bList2
    }
  }

# plot method for class dlsem
plot.dlsem <- function(x,conf=0.95,style=2,node.col=NULL,font.col=NULL,border.col=NULL,edge.col=NULL,edge.lab=NULL,...) {
  if((style %in% c(0,1,2))==F) stop("Argument 'style' must be either '0' (plain), '1' (significance shown), or '2' (signs shown)",call.=F)
  G <- makeGraph(x,conf=conf)
  nomi <- nodes(G$full.graph)  
  #####
  cutString <- function(x,l) {
    n <- nchar(x)
    k <- ceiling(n/l)
    res <- c()
    for(i in 1:k) {
      res[i] <- substr(x,1+(i-1)*l,i*l)
      }
    paste(res,collapse="\n")
    }
  #####
  nAttr <- list()
  nAttr$shape <- rep("ellipse",length(nomi))
  nAttr$fontsize <- rep(14,length(nomi))
  nAttr$height <- rep(2.5,length(nomi))
  nAttr$width <- rep(4,length(nomi))
  nAttr$label <- sapply(nomi,cutString,l=12)  ##### maximum number of characters: 12
  for(i in 1:length(nAttr)) {
    names(nAttr[[i]]) <- nomi
    }
  if(!is.null(node.col)) nAttr$fillcolor <- node.col
  if(!is.null(font.col)) nAttr$fontcolor <- font.col
  if(!is.null(border.col)) nAttr$color <- border.col
  eAttr <- list()                                                     
  if(!is.null(edge.lab)) {
    eAttr$label <- edge.lab
    #eAttr$labelfontsize <- rep(14,length(edge.lab))
    #names(eAttr$labelfontsize) <- names(edge.lab)
    }
  if(!is.null(edge.col)) {
    eAttr$color <- edge.col     
    eAttr$color[which(G$sign=="")] <- NA
    } else {
    eCol <- G$sign
    if(style==1) {
      eCol[which(G$sign=="+")] <- "grey30"
      eCol[which(G$sign=="-")] <- "grey30"
      eCol[which(G$sign=="")] <- "grey75"
      } else if(style==2) {
      eCol[which(G$sign=="+")] <- "green4"
      eCol[which(G$sign=="-")] <- "tomato3"
      eCol[which(G$sign=="")] <- "grey75"
      } else {
      eCol[] <- "grey30"
      }
    eAttr[[length(eAttr)+1]] <- eCol
    }
  names(eAttr)[length(eAttr)] <- "color"
  #eStyl <- G$sign
  #eStyl[which(G$sign=="+")] <- "solid"
  #eStyl[which(G$sign=="-")] <- "dashed"
  #eAttr[[length(eAttr)+1]] <- eStyl
  #names(eAttr)[length(eAttr)] <- "style"
  #
  #
  par(xpd=T)
  plot(G$full.graph,"dot",nodeAttrs=nAttr,edgeAttrs=eAttr,attrs=list(edge=list(color="grey25",arrowsize=0.4)),...)
  par(defpar)
  }   

# find the child sets (internal use only)
chldsets <- function(x) {
  pset <- inEdges(x)
  findchld <- function(xname,ps) {names(which(sapply(ps,function(z){xname %in% z})==T))}
  nomi <- names(pset)
  res <- lapply(nomi,findchld,ps=pset)
  names(res) <- nomi
  res
  }

# node markov blanket (internal use only)
nodeMB <- function(nodeName,G) {
  pset <- inEdges(G)
  auxch <- chldsets(G)[[nodeName]]
  auxpar <- pset[[nodeName]]
  auxp <- c()
  if(length(auxch)>0) {
    for(i in 1:length(auxch)) {
      auxp <- c(auxp,pset[[auxch[i]]])
      }
    }
  setdiff(unique(c(auxpar,auxch,auxp)),nodeName)
  }

# node ancestors (internal use only)
nodeAnces <- function(nodeName,G) {
  eList <- inEdges(G)
  xpar <- aux.xpar <- eList[[nodeName]]
  while(length(aux.xpar)>0) {
    newpar <- c()
    for(i in 1:length(aux.xpar)) {
      newpar <- c(newpar,eList[[aux.xpar[i]]])
      }
    xpar <- unique(c(xpar,newpar))
    aux.xpar <- newpar
    }
  unique(xpar)
  }

# node descendants (internal use only)
nodeDescen <- function(nodeName,G) {
  eList <- chldsets(G)
  xpar <- aux.xpar <- eList[[nodeName]]
  while(length(aux.xpar)>0) {
    newpar <- c()
    for(i in 1:length(aux.xpar)) {
      newpar <- c(newpar,eList[[aux.xpar[i]]])
      }
    xpar <- c(xpar,newpar)
    aux.xpar <- newpar
    }
  unique(xpar)
  }

# find topological order (internal use only)
topOrder <- function(G) {
  parSet <- inEdges(G)
  nomi <- names(parSet)
  L <- c()
  S <- nomi[which(sapply(parSet,length)==0)] 
  while(length(S)>0) {
    xaux <- S[1]
    S <- setdiff(S,xaux)
    L <- c(L,xaux)
    sch <- c()
    for(j in 1:length(parSet)) {
      if(xaux %in% parSet[[j]]) sch <- c(sch,nomi[j])
      }
    if(length(sch)>0) {
      for(j in 1:length(sch)) {
        parSet[[sch[j]]] <- setdiff(parSet[[sch[j]]],xaux)
        if(length(parSet[[sch[j]]])==0) S <- c(S,sch[j])  
        }
      }
    }
  if(sum(sapply(parSet,length))==0) L else NULL
  }

# ancestral graph (internal use only)
angraph <- function(x,G) {
  xanc <- unlist(sapply(x,nodeAnces,G))
  xOK <- unique(c(x,xanc))
  subGraph(xOK,G)
  }

# moral graph (internal use only)
morgraph <- function(G) {
  pset <- inEdges(G)
  nomi <- names(pset)
  for(i in 1:length(nomi)) {
    ipar <- pset[[nomi[i]]]
    if(length(ipar)>2) {
      for(j in 1:(length(ipar)-1)) {
        for(w in (j+1):length(ipar)) {
          if((ipar[j] %in% pset[[ipar[w]]])==F) G <- addEdge(ipar[j],ipar[w],G,1)
          }
        }
      }
    }
  newpset <- inEdges(G)  
  W <- new("graphNEL",nodes=nomi,edgemode="undirected")    
  for(i in 1:length(nomi)) {
    ips <- newpset[[i]]
    if(length(ips)>0) {
      for(j in 1:length(ips)) {
        W <- addEdge(ips[j],nomi[i],W,1) 
        }
      }
    }  
  W
  }

# check conditional independence
isIndep <- function(x,var1=NULL,var2=NULL,given=NULL,conf=0.95,use.ns=FALSE) {
  if(("dlsem" %in% class(x))==F) stop("Argument 'x' must be an object of class 'dlsem'",call.=F)
  if(is.null(var1) || is.na(var1)) stop("Argument 'var1' is missing",call.=F)
  if(!is.null(var1) && length(var1)!=1) stop("Argument 'var1' must be of length 1",call.=F)
  if(is.null(var2) || is.na(var2)) stop("Argument 'var2' is missing",call.=F)  
  if(!is.null(var2) && length(var2)!=1) stop("Argument 'var2' must be of length 1",call.=F)
  if(!is.null(given) && is.na(given)) stop("Argument 'given' is missing",call.=F)  
  if(length(use.ns)!=1 || !is.logical(use.ns)) stop("Argument 'use.ns' must be a logical value",call.=F)
  Gobj <- makeGraph(x,conf=conf)
  if(use.ns==F) {
    G <- Gobj$graph
    } else {
    G <- Gobj$full.graph
    }
  nomi <- nodes(G)
  auxcheck <- setdiff(c(var1,var2,given),nomi)
  if(length(auxcheck)>0) stop("Unknown variable '",auxcheck[1],"'",sep="",call.=F)
  if(length(unlist(chldsets(G)))>0) {
    Gm <- morgraph(angraph(c(var1,var2,given),G))
    pset <- chldsets(Gm)                      
    xedg <- c()
    for(i in 1:length(pset)) {
      if(length(pset[[i]])>0) {
        for(j in 1:length(pset[[i]])) {
          xedg <- rbind(xedg,c(names(pset)[i],pset[[i]][j])) 
          }
        }
      }                              
    if(!is.null(xedg)) {
      xedg <- rbind(xedg,xedg[,2:1])
      nomi <- sort(unique(xedg))
      borde <- var1
      reached <- c()
      neighb <- function(x,node) {
        Ne <- c(x[which(x[,1]==node),2],x[which(x[,2]==node),1])
        sort(unique(Ne))
        }                     
      while(length(borde)>0) {
        reached <- c(reached,borde)
        fan_borde <- c()
        for(i in 1:length(borde)) {                      
          auxne <- neighb(xedg,borde[i])                 
          fan_borde <- c(fan_borde,auxne) 
          }                              
        borde <- setdiff(fan_borde,c(reached,given))   
        if(length(intersect(borde,var2))>0) break()
        }
      ifelse(length(borde)>0, res <- F, res <- T)
      } else {
      res <- T
      }
    } else {
    res <- T
    }
  res
  }

# matrix of edges (internal use only)
edgeMat <- function(x,conf,full) {
  if(full==T) {
    xedg <- chldsets(makeGraph(x,conf=conf)$full.graph)
    } else {
    xedg <- chldsets(makeGraph(x,conf=conf)$graph)
    }
  res <- c()
  if(length(xedg)>0) {
    for(i in 1:length(xedg)) {
      if(length(xedg[[i]])>0) {
        res <- rbind(res,cbind(rep(names(xedg)[i],length(xedg[[i]])),xedg[[i]]))
        }
      }
    }
  res
  }

# find directed path (internal use only)
dpathFind <- function(G,from,to) {
  auxedg <- chldsets(G)
  if(to %in% nodeDescen(from,G)) {
    auxmat <- c()
    for(i in 1:length(auxedg)) {
      if(length(auxedg[[i]])>0) {
        for(j in 1:length(auxedg[[i]])) {
          auxmat <- rbind(auxmat,c(names(auxedg)[i],auxedg[[i]][j]))
          }
        }
      }
    auxchld <- auxmat[which(auxmat[,1]==from),2]
    pathList <- list()
    for(i in 1:length(auxchld)) pathList[[i]] <- c(from,auxchld[i])
    endcheck <- function(x) {
      res <- rep(0,length(x))                 
      for(i in 1:length(x)) {                
        if(rev(x[[i]])[1]==to) res[i] <- 1
        }
      res
      }                        
    isOK <- endcheck(pathList)               
    while(sum(isOK)<length(isOK)) {
      auxind <- which(isOK==0)[1]         
      auxchld <- auxmat[which(auxmat[,1]==rev(pathList[[auxind]])[1]),2]
      auxpath <- pathList[[auxind]]
      if(length(auxchld)>0) {
        pathList[[auxind]] <- c(auxpath,auxchld[1])
        if(length(auxchld)>1) {
          for(i in 2:length(auxchld)) {
            pathList[[length(pathList)+1]] <- c(auxpath,auxchld[i])
            }
          }   
        } else {
        pathList <- pathList[-auxind]
        }                       
      isOK <- endcheck(pathList)  
      }
    pathList
    } else {
    NULL
    }
  } 

# find lag to sum (internal use only)
findlag2sum <- function(x,lag) {
  g <- length(x)                             
  out <- list()
  for(w in 1:length(lag)) {
    mycode <- "res <- c(); "
    for(i in 1:g) {
      mycode <- paste(mycode,"for(k",i," in 1:length(x[[",i,"]])) { ; ",sep="")
      }
    mycode <- paste(mycode,paste("xaux <- c(",paste("x[[",1:g,"]][k",1:g,"]",collapse=",",sep=""),"); ",sep=""),sep="")
    mycode <- paste(mycode,"if(sum(xaux)==lag[w]) { res <- rbind(res,xaux) }; ",sep="")
    mycode <- paste(mycode,paste(rep("}",length(x)),collapse="; "),sep="")
    eval(parse(text=mycode))
    if(is.null(res)) {
      res <- matrix(nrow=0,ncol=g)
      } else {
      rownames(res) <- NULL
      }
    colnames(res) <- names(x)             
    out[[w]] <- res
    }
  names(out) <- lag
  out
  }

# computation of causal effects
causalEff <- function(x,from=NULL,to=NULL,lag=NULL,cumul=FALSE,conf=0.95,use.ns=FALSE) {
  if(("dlsem" %in% class(x))==F) stop("Argument 'x' must be an object of class 'dlsem'",call.=F)
  if(is.null(from) || is.na(from)) stop("Argument 'from' is missing",call.=F)
  if(is.null(to) || is.na(to)) stop("Argument 'to' is missing",call.=F)
  if(!is.character(from)) stop("Invalid argument 'from'",call.=F)
  if(!is.character(to)) stop("Invalid argument 'to'",call.=F)
  if(length(to)!=1) stop("Argument 'to' must be of length 1",call.=F)
  if(length(cumul)!=1 || !is.logical(cumul)) stop("Arguent 'cumul' must be a logical value",call.=F)
  if(length(use.ns)!=1 || !is.logical(use.ns)) stop("Argument 'use.ns' must be a logical value",call.=F)
  if(!is.null(lag)) {
    for(i in 1:length(lag)) {
      if(!is.numeric(lag[i]) || is.na(lag[i]) || lag[i]<0 || lag[i]!=round(lag[i])) stop("Argument 'lag' must contain non-negative integer numbers only",call.=F)
      }
    }
  auxcheck <- setdiff(c(from,to),names(x$estimate))
  if(length(auxcheck)>0) {
    auxcntx <- intersect(auxcheck,x$exogenous)
    if(length(auxcntx)>0) {
      stop("Variable '",auxcntx[1],"' is exogenous and cannot appear in argument 'from' or 'to'",call.=F)
      } else {
      stop("Unknown variable '",auxcheck[1],"'",sep="",call.=F)
      }
    }
  Gobj <- makeGraph(x,conf=conf)
  if(use.ns==F) {
    G <- Gobj$graph
    } else {
    G <- Gobj$full.graph
    }
  nomi <- nodes(G)
  if(length(setdiff(from,nomi))>0) stop("Unknown variable '",setdiff(from,nomi)[1],"'",sep="",call.=F)
  if((to %in% nomi)==F) stop("Unknown variable '",to,"'",sep="",call.=F)
  isOK <- rep(1,length(from))
  for(i in 1:length(from)) {
    if((to %in% nodeDescen(from[i],G))==F) isOK[i] <- 0
    }
  ###
  pset <- inEdges(G)
  for(i in 1:length(from)) {
    ipa <- pset[[from[[i]]]]
    if(length(ipa)>0) {
      for(j in 1:length(ipa)) {
        G <- removeEdge(ipa[j],from[i],G)
        }
      }
    }  
  ###
  if(sum(isOK)>0) {
    from <- from[which(isOK==1)]
    mycol1 <- mycol2 <- rep(NA,length(nomi))
    names(mycol1) <- names(mycol2) <- nomi
    nodemed <- c()
    pathList <- vector("list",length=length(from))
    for(i in 1:length(from)) {
      pathList[[i]] <- dpathFind(G,from=from[i],to=to)
      nodemed <- c(nodemed,setdiff(unlist(pathList[[i]]),c(from,to)))
      }
    nodemed <- unique(nodemed)
    names(pathList) <- from
    auxdel <- which(sapply(pathList,is.null)==T)
    if(length(auxdel)>0) {
      pathList <- pathList[-auxdel]
      from <- from[-auxdel]
      }
    nodecond <- setdiff(unlist(pset[c(to,nodemed)]),c(from,nodemed))
    nodebarr <- setdiff(nomi,c(from,to,nodemed,nodecond))
    mycol1[c(from,to)] <- mycol2[c(from,to)] <- "grey20"
    mycol1[nodemed] <- mycol2[nodemed] <- "grey20"
    mycol1[nodecond] <- "navy"
    mycol2[nodecond] <- "grey70"
    mycol1[nodebarr] <- mycol2[nodebarr] <- "grey70"
    xedg <- chldsets(G)
    ednam <- list()
    for(i in 1:length(xedg)) {
      if(length(xedg[[i]])>0) ednam[[i]] <- paste(names(xedg)[i],"~",xedg[[i]],sep="")
      }
    ednam <- unlist(ednam)
    eddel <- c()
    for(i in 1:length(nodebarr)) {
      eddel <- c(eddel,paste(nodebarr[i],"~",setdiff(nomi,nodebarr[i]),sep=""),paste(setdiff(nomi,nodebarr[i]),"~",nodebarr[i],sep=""))
      }
    for(i in 1:length(nodecond)) {
      eddel <- c(eddel,paste(nodecond[i],"~",setdiff(nomi,nodecond[i]),sep=""),paste(setdiff(nomi,nodecond[i]),"~",nodecond[i],sep=""))
      }
    edcol <- rep("grey70",length(Gobj$sign))
    names(edcol) <- names(Gobj$sign)
    edcol[intersect(setdiff(ednam,eddel),names(which(Gobj$sign=="+")))] <- "green4"
    edcol[intersect(setdiff(ednam,eddel),names(which(Gobj$sign=="-")))] <- "tomato3"
    newPathList <- list()
    for(i in 1:length(pathList)) {
      newPathList <- c(newPathList,pathList[[i]])
      }                                                        
    laglen <- list()
    for(i in 1:length(newPathList)) {
      jlaglen <- list()
      for(j in 2:length(newPathList[[i]])) {
        auxnam <- paste(newPathList[[i]][j],"~",newPathList[[i]][j-1],sep="")
        auxeff <- lagEff(model=x$estimate[[newPathList[[i]][j]]],x=newPathList[[i]][j-1],cumul=F,conf=conf,lag=NULL)
        auxpos <- which(auxeff[,1]!=0)
        if(length(auxpos)>0) {
          jlaglen[[j-1]] <- as.numeric(rownames(auxeff)[auxpos])
          } else {
          jlaglen[[j-1]] <- NA
          }
        }
      laglen[[i]] <- c(min(jlaglen[[1]],na.rm=T),sum(sapply(jlaglen,max,na.rm=T)))
      }
    meL <- max(unlist(laglen))+1        
    lagOK <- 0:meL
    mycoeff <- edgeCoeff(x,lag=lagOK,conf=conf)         
    quan <- -qnorm((1-conf)/2)
    #
    sd_calc <- function(muvet,sdvet) { sqrt(prod(muvet^2+sdvet^2)-prod(muvet^2)) }
    #
    sd_sum <- function(sdvet) {
      res <- c()
      for(i in 1:length(sdvet)) {
        res[i] <- sqrt(sum(sdvet[1:i]^2))
        }
      res
      }    
    #
    bhat <- list()
    for(i in 1:length(mycoeff)) {
      bhat[[i]] <- matrix(nrow=nrow(mycoeff[[i]]),ncol=2)
      for(j in 1:nrow(mycoeff[[i]])) {
        auxeval <- mycoeff[[i]][j,1]
        auxsd <- (mycoeff[[i]][j,3]-mycoeff[[i]][j,1])/quan
        bhat[[i]][j,] <- c(auxeval,auxsd)
        }
      rownames(bhat[[i]]) <- rownames(mycoeff[[i]])
      }
    names(bhat) <- names(mycoeff)
    outList <- list()                       
    for(i in 1:length(newPathList)) {           
      outList[[i]] <- matrix(nrow=length(lagOK),ncol=2)
      rownames(outList[[i]]) <- lagOK
      colnames(outList[[i]]) <- c("estimate","std. error")
      auxbetalag <- list()
      for(j in 2:length(newPathList[[i]])) {
        auxnam <- paste(newPathList[[i]][j],"~",newPathList[[i]][j-1],sep="")
        auxeff <- lagEff(model=x$estimate[[newPathList[[i]][j]]],x=newPathList[[i]][j-1],cumul=F,conf=conf,lag=lagOK)
        auxpos <- which(auxeff[,1]!=0)
        if(length(auxpos)>0) {                                 
          auxbetalag[[j-1]] <- as.numeric(rownames(auxeff)[auxpos])
          } else {
          auxbetalag[[j-1]] <- 0
          }
        names(auxbetalag)[j-1] <- auxnam
        }
      lagsumMat <- findlag2sum(auxbetalag,lagOK)
      for(j in 1:length(lagsumMat)) {
        auxlag <- as.character(lagOK[j])                           
        auxind <- lagsumMat[[j]]                          
        if(nrow(auxind)>0) {
          auxres <- array(dim=c(nrow(auxind),ncol(auxind),2))
          for(w1 in 1:nrow(auxind)) {                          
            for(w2 in 1:ncol(auxind)) {
              auxres[w1,w2,1] <- bhat[[as.character(auxind[w1,w2])]][colnames(auxind)[w2],1]
              auxres[w1,w2,2] <- bhat[[as.character(auxind[w1,w2])]][colnames(auxind)[w2],2]
              }
            }                                                           
          muprod <- sdprod <- c()
          for(w in 1:nrow(auxind)) {                               
            muprod[w] <- prod(auxres[w,,1])                              
            sdprod[w] <- sd_calc(auxres[w,,1],auxres[w,,2])
            }
          mupath <- sum(muprod)
          sdpath <- sqrt(sum(sdprod^2))
          outList[[i]][j,] <- c(mupath,sdpath)
          } else {
          outList[[i]][j,] <- rep(0,2)
          }
        }
      }
    names(outList) <- sapply(newPathList,function(x){paste(x,collapse="*")})
    out <- matrix(nrow=length(lagOK),ncol=2)
    rownames(out) <- lagOK
    colnames(out) <- c("estimate","std. error")
    for(j in 1:length(lagOK)) {
      auxover <- rep(0,2)
      for(i in 1:length(outList)) {
        auxover <- auxover+outList[[i]][j,]
        }
      out[j,] <- auxover
      }
    outList[[length(outList)+1]] <- out
    names(outList)[[length(outList)]] <- "overall"
    if(cumul==T) {
      for(i in 1:length(outList)) {
        if(nrow(outList[[i]])>1) {
          outList[[i]][,1] <- cumsum(outList[[i]][,1])
          outList[[i]][,2] <- sd_sum(outList[[i]][,2])
          }
        }    
      }
    #
    for(i in 1:length(outList)) {
      imu <- outList[[i]][,1]
      isd <- outList[[i]][,2]
      outList[[i]] <- cbind(imu,isd,imu-quan*isd,imu+quan*isd)
      colnames(outList[[i]]) <- c("estimate","std. err.",paste(c("lower ","upper "),conf*100,"%",sep=""))
      }
    #
    if(is.null(lag)) {
      outList
      } else {
      lagSel <- lag
      auxOlag <- which(lag>max(lagOK))
      if(length(auxOlag)>0) lagSel[auxOlag] <- max(lagOK)  
      outList2 <- outList
      for(i in 1:length(outList)) {
        outList2[[i]] <- outList[[i]][as.character(lagSel),,drop=F]
        rownames(outList2[[i]]) <- lag
        }
      outList2
      }
    } else {
    auxanc <- nodeAnces(to,makeGraph(x)$full.graph)
    if(length(intersect(from,auxanc))>0) {
      #stop("No paths found connecting the selected variables. Try to reduce 'conf' or to set 'use.ns' to TRUE",call.=F)
      #} else {
      #stop("No paths exist connecting the selected variables",call.=F)
      NULL
      }
    }
  }
