% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posixt.R
\name{posixt-rounding}
\alias{posixt-rounding}
\alias{date_floor.POSIXt}
\alias{date_ceiling.POSIXt}
\alias{date_round.POSIXt}
\title{Rounding: date-time}
\usage{
\method{date_floor}{POSIXt}(
  x,
  precision,
  ...,
  n = 1L,
  origin = NULL,
  nonexistent = NULL,
  ambiguous = x
)

\method{date_ceiling}{POSIXt}(
  x,
  precision,
  ...,
  n = 1L,
  origin = NULL,
  nonexistent = NULL,
  ambiguous = x
)

\method{date_round}{POSIXt}(
  x,
  precision,
  ...,
  n = 1L,
  origin = NULL,
  nonexistent = NULL,
  ambiguous = x
)
}
\arguments{
\item{x}{\verb{[POSIXct / POSIXlt]}

A date-time vector.}

\item{precision}{\verb{[character(1)]}

One of:
\itemize{
\item \code{"week"}
\item \code{"day"}
\item \code{"hour"}
\item \code{"minute"}
\item \code{"second"}
}

\code{"week"} is an alias for \code{"day"} with \code{n * 7}.}

\item{...}{These dots are for future extensions and must be empty.}

\item{n}{\verb{[positive integer(1)]}

A single positive integer specifying a multiple of \code{precision} to use.}

\item{origin}{\verb{[POSIXct(1) / POSIXlt(1) / NULL]}

An origin to start counting from.

\code{origin} must have exactly the same time zone as \code{x}.

\code{origin} will be floored to \code{precision}. If information is lost when
flooring, a warning will be thrown.

If \code{NULL}, defaults to midnight on 1970-01-01 in the time zone of \code{x}.}

\item{nonexistent}{\verb{[character / NULL]}

One of the following nonexistent time resolution strategies, allowed to be
either length 1, or the same length as the input:
\itemize{
\item \code{"roll-forward"}: The next valid instant in time.
\item \code{"roll-backward"}: The previous valid instant in time.
\item \code{"shift-forward"}: Shift the nonexistent time forward by the size of
the daylight saving time gap.
\item \verb{"shift-backward}: Shift the nonexistent time backward by the size of
the daylight saving time gap.
\item \code{"NA"}: Replace nonexistent times with \code{NA}.
\item \code{"error"}: Error on nonexistent times.
}

Using either \code{"roll-forward"} or \code{"roll-backward"} is generally
recommended over shifting, as these two strategies maintain the
\emph{relative ordering} between elements of the input.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{nonexistent} must be supplied
and cannot be \code{NULL}. This is a convenient way to make production code
robust to nonexistent times.}

\item{ambiguous}{\verb{[character / zoned_time / POSIXct / list(2) / NULL]}

One of the following ambiguous time resolution strategies, allowed to be
either length 1, or the same length as the input:
\itemize{
\item \code{"earliest"}: Of the two possible times, choose the earliest one.
\item \code{"latest"}: Of the two possible times, choose the latest one.
\item \code{"NA"}: Replace ambiguous times with \code{NA}.
\item \code{"error"}: Error on ambiguous times.
}

Alternatively, \code{ambiguous} is allowed to be a zoned_time (or POSIXct) that
is either length 1, or the same length as the input. If an ambiguous time
is encountered, the zoned_time is consulted. If the zoned_time corresponds
to a naive_time that is also ambiguous \emph{and} uses the same daylight saving
time transition point as the original ambiguous time, then the offset of
the zoned_time is used to resolve the ambiguity. If the ambiguity cannot be
resolved by consulting the zoned_time, then this method falls back to
\code{NULL}.

Finally, \code{ambiguous} is allowed to be a list of size 2, where the first
element of the list is a zoned_time (as described above), and the second
element of the list is an ambiguous time resolution strategy to use when
the ambiguous time cannot be resolved by consulting the zoned_time.
Specifying a zoned_time on its own is identical to \verb{list(<zoned_time>, NULL)}.

If \code{NULL}, defaults to \code{"error"}.

If \code{getOption("clock.strict")} is \code{TRUE}, \code{ambiguous} must be supplied and
cannot be \code{NULL}. Additionally, \code{ambiguous} cannot be specified as a
zoned_time on its own, as this implies \code{NULL} for ambiguous times that the
zoned_time cannot resolve. Instead, it must be specified as a list
alongside an ambiguous time resolution strategy as described above. This is
a convenient way to make production code robust to ambiguous times.}
}
\value{
\code{x} rounded to the specified \code{precision}.
}
\description{
These are POSIXct/POSIXlt methods for the
\link[=date-and-date-time-rounding]{rounding generics}.
\itemize{
\item \code{date_floor()} rounds a date-time down to a multiple of
the specified \code{precision}.
\item \code{date_ceiling()} rounds a date-time up to a multiple of
the specified \code{precision}.
\item \code{date_round()} rounds up or down depending on what is closer,
rounding up on ties.
}

You can group by irregular periods such as \code{"month"} or \code{"year"} by using
\code{\link[=date_group]{date_group()}}.
}
\details{
When rounding by \code{"week"}, remember that the \code{origin} determines the "week
start". By default, 1970-01-01 is the implicit origin, which is a
Thursday. If you would like to round by weeks with a different week start,
just supply an origin on the weekday you are interested in.
}
\examples{
x <- as.POSIXct("2019-03-31", "America/New_York")
x <- add_days(x, 0:5)

# Flooring by 2 days, note that this is not tied to the current month,
# and instead counts from the specified `origin`, so groups can cross
# the month boundary
date_floor(x, "day", n = 2)

# Compare to `date_group()`, which groups by the day of the month
date_group(x, "day", n = 2)

# Note that daylight saving time gaps can throw off rounding
x <- as.POSIXct("1970-04-26 01:59:59", "America/New_York") + c(0, 1)
x

# Rounding is done in naive-time, which means that rounding by 2 hours
# will attempt to generate a time of 1970-04-26 02:00:00, which doesn't
# exist in this time zone
try(date_floor(x, "hour", n = 2))

# You can handle this by specifying a nonexistent time resolution strategy
date_floor(x, "hour", n = 2, nonexistent = "roll-forward")
}
