% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weightit2glm.R
\name{method_glm}
\alias{method_glm}
\title{Propensity Score Weighting Using Generalized Linear Models}
\description{
This page explains the details of estimating weights from generalized linear model-based propensity scores by setting \code{method = "glm"} in the call to \code{\link[=weightit]{weightit()}} or \code{\link[=weightitMSM]{weightitMSM()}}. This method can be used with binary, multi-category, and continuous treatments.

In general, this method relies on estimating propensity scores with a parametric generalized linear model and then converting those propensity scores into weights using a formula that depends on the desired estimand. For binary and multi-category treatments, a binomial or multinomial regression model is used to estimate the propensity scores as the predicted probability of being in each treatment given the covariates. For ordinal treatments, an ordinal regression model is used to estimate generalized propensity scores. For continuous treatments, a generalized linear model is used to estimate generalized propensity scores as the conditional density of treatment given the covariates.
\subsection{Binary Treatments}{

For binary treatments, this method estimates the propensity scores using \code{\link[=glm]{glm()}}. An additional argument is \code{link}, which uses the same options as \code{link} in \code{\link[=family]{family()}}. The default link is "logit", but others, including "probit", are allowed. The following estimands are allowed: ATE, ATT, ATC, ATO, ATM, and ATOS. Weights can also be computed using marginal mean weighting through stratification for the ATE, ATT, and ATC. See \code{\link[=get_w_from_ps]{get_w_from_ps()}} for details.
}

\subsection{Multi-Category Treatments}{

For multi-category treatments, the propensity scores are estimated using multinomial regression from one of a few functions depending on the argument supplied to \code{multi.method} (see Additional Arguments below). The following estimands are allowed: ATE, ATT, ATC, ATO, and ATM. The weights for each estimand are computed using the standard formulas or those mentioned above. Weights can also be computed using marginal mean weighting through stratification for the ATE, ATT, and ATC. See \code{\link[=get_w_from_ps]{get_w_from_ps()}} for details. Ordinal treatments are treated exactly the same as non-order multi-category treatments except that additional models are available to estimate the generalized propensity score (e.g., ordinal logistic regression).
}

\subsection{Continuous Treatments}{

For continuous treatments, the generalized propensity score is estimated using linear regression. The conditional density can be specified as normal or another distribution. In addition, kernel density estimation can be used instead of assuming a specific density for the numerator and denominator of the generalized propensity score by setting \code{density = "kernel"}. Other arguments to \code{\link[=density]{density()}} can be specified to refine the density estimation parameters. \code{plot = TRUE} can be specified to plot the density for the numerator and denominator, which can be helpful in diagnosing extreme weights.
}

\subsection{Longitudinal Treatments}{

For longitudinal treatments, the weights are the product of the weights estimated at each time point.
}

\subsection{Sampling Weights}{

Sampling weights are supported through \code{s.weights} in all scenarios except for multi-category treatments with \code{link = "bayes.probit"} and for binary and continuous treatments with \code{missing = "saem"} (see below). Warning messages may appear otherwise about non-integer successes, and these can be ignored.
}

\subsection{Missing Data}{

In the presence of missing data, the following value(s) for \code{missing} are allowed:
\describe{
\item{\code{"ind"} (default)}{First, for each variable with missingness, a new missingness indicator variable is created which takes the value 1 if the original covariate is \code{NA} and 0 otherwise. The missingness indicators are added to the model formula as main effects. The missing values in the covariates are then replaced with the covariate medians (this value is arbitrary and does not affect estimation). The weight estimation then proceeds with this new formula and set of covariates. The covariates output in the resulting \code{weightit} object will be the original covariates with the \code{NA}s.
}
\item{\code{"saem"}}{For binary treatments with \code{link = "logit"} or continuous treatments, a stochastic approximation version of the EM algorithm (SAEM) is used via the \CRANpkg{misaem} package. No additional covariates are created. See Jiang et al. (2019) for information on this method. In some cases, this is a suitable alternative to multiple imputation.
}
}
}

\subsection{M-estimation}{

For binary treatments, M-estimation is supported when \code{link} is neither \code{"flic"} nor \code{"flac"} (see below). For multi-category treatments, M-estimation is supported when \code{multi.method} is \code{"weightit"} (the default for non-ordered treatments) or \code{"glm"}. For continuous treatments, M-estimation is supported when \code{density} is not \code{"kernel"}. The conditional treatment variance and unconditional treatment mean and variance are included as parameters to estimate, as these all go into calculation of the weights. For all treatment type, M-estimation is not supported when \code{missing = "saem"}. See \code{\link[=glm_weightit]{glm_weightit()}} and \code{vignette("estimating-effects")} for details. For longitudinal treatments, M-estimation is supported whenever the underlying methods are.
}
}
\section{Additional Arguments}{

For binary treatments, the following additional argument can be specified:
\describe{
\item{\code{link}}{the link used in the generalized linear model for the propensity scores. \code{link} can be any of those allowed by \code{\link[=binomial]{binomial()}}. A \code{br.} prefix can be added (e.g., \code{"br.logit"}); this changes the fitting method to the bias-corrected generalized linear models implemented in the \CRANpkg{brglm2} package. \code{link} can also be either \code{"flic"} or \code{"flac"} to fit the corresponding Firth corrected logistic regression models implemented in the \CRANpkg{logistf} package.}
}

For multi-category treatments, the following additional arguments can be specified:
\describe{
\item{\code{multi.method}}{the method used to estimate the generalized propensity scores. Allowable options include \code{"weightit"} to use an M-estimation-based method of multinomial logistic regression implemented in \pkg{WeightIt}, \code{"glm"} to use a series of binomial models using \code{\link[=glm]{glm()}}, \code{"mclogit"} to use multinomial logistic regression as implemented in \pkgfun{mclogit}{mblogit}, \code{"mnp"} to use Bayesian multinomial probit regression as implemented in \pkgfun{MNP}{MNP}, and \code{"brmultinom"} to use bias-reduced multinomial logistic regression as implemented in \pkgfun{brglm2}{brmultinom}. For ordered treatments, \code{"polr"} can be supplied to use ordinal regression as implemented in \pkgfun{MASS}{polr} unless \code{link} is \code{"br.logit"}, in which case bias-reduce ordinal logistic regression as implemented in \pkgfun{brglm2}{bracl} is used. \code{"weightit"} and \code{"mclogit"} should give near-identical results, the main difference being increased robustness and customizability when using \code{"mclogit"} at the expense of not being able to use M-estimation to compute standard errors after weighting. The default is \code{"weightit"} for un-ordered treatments and \code{"polr"} for ordered treatments. Ignored when \code{missing = "saem"}.}
\item{\code{link}}{The link used in the multinomial, binomial, or ordered regression model for the generalized propensity scores depending on the argument supplied to \code{multi.method}. When \code{multi.method = "glm"}, \code{link} can be any of those allowed by \code{\link[=binomial]{binomial()}}. When treatment is ordered and \code{multi.method = "polr"}, \code{link} can be any of those allowed by \code{MASS::polr()} or \code{"br.logit"}. Otherwise, \code{link} should be \code{"logit"} or not specified.}
}

For continuous treatments, the following additional arguments may be supplied:
\describe{
\item{\code{density}}{A function corresponding the conditional density of the treatment. The standardized residuals of the treatment model will be fed through this function to produce the numerator and denominator of the generalized propensity score weights. If blank, \code{\link[=dnorm]{dnorm()}} is used as recommended by Robins et al. (2000). This can also be supplied as a string containing the name of the function to be called. If the string contains underscores, the call will be split by the underscores and the latter splits will be supplied as arguments to the second argument and beyond. For example, if \code{density = "dt_2"} is specified, the density used will be that of a t-distribution with 2 degrees of freedom. Using a t-distribution can be useful when extreme outcome values are observed (Naimi et al., 2014).

Can also be \code{"kernel"} to use kernel density estimation, which calls \code{\link[=density]{density()}} to estimate the numerator and denominator densities for the weights. (This used to be requested by setting \code{use.kernel = TRUE}, which is now deprecated.)}
\item{\code{bw}, \code{adjust}, \code{kernel}, \code{n}}{If \code{density = "kernel"}, the arguments to \code{\link[=density]{density()}}. The defaults are the same as those in \code{density()} except that \code{n} is 10 times the number of units in the sample.}
\item{\code{plot}}{If \code{density = "kernel"}, whether to plot the estimated densities.}
\item{\code{link}}{The link used to fit the linear model for the generalized propensity score. Can be any allowed by \code{\link[=gaussian]{gaussian()}}.
}
}

Additional arguments to \code{glm()} can be specified as well when it is used for fitting. The \code{method} argument in \code{glm()} is renamed to \code{glm.method}. This can be used to supply alternative fitting functions, such as those implemented in the \CRANpkg{glm2} package. Other arguments to \code{weightit()} are passed to \code{...} in \code{glm()}. In the presence of missing data with \code{link = "logit"} and \code{missing = "saem"}, additional arguments are passed to \pkgfun{misaem}{miss.glm} and \pkgfun{misaem}{predict.miss.glm}, except the \code{method} argument in \pkgfun{misaem}{predict.miss.glm} is replaced with \code{saem.method}.

For continuous treatments in the presence of missing data with \code{missing = "saem"}, additional arguments are passed to \pkgfun{misaem}{miss.lm} and \pkgfun{misaem}{predict.miss.lm}.
}

\section{Additional Outputs}{

\describe{
\item{\code{obj}}{When \code{include.obj = TRUE}, the (generalized) propensity score model fit. For binary treatments, the output of the call to \code{\link[=glm]{glm()}} or the requested fitting function. For multi-category treatments, the output of the call to the fitting function (or a list thereof if \code{multi.method = "glm"}). For continuous treatments, the output of the call to \code{glm()} for the predicted values in the denominator density.
}
}
}

\examples{
library("cobalt")
data("lalonde", package = "cobalt")

#Balancing covariates between treatment groups (binary)
(W1 <- weightit(treat ~ age + educ + married +
                  nodegree + re74, data = lalonde,
                method = "glm", estimand = "ATT",
                link = "probit"))
summary(W1)
bal.tab(W1)

#Balancing covariates with respect to race (multi-category)
(W2 <- weightit(race ~ age + educ + married +
                  nodegree + re74, data = lalonde,
                method = "glm", estimand = "ATE"))
summary(W2)
bal.tab(W2)

#Balancing covariates with respect to re75 (continuous)
#with kernel density estimate
(W3 <- weightit(re75 ~ age + educ + married +
                  nodegree + re74, data = lalonde,
                method = "glm", density = "kernel"))
summary(W3)
bal.tab(W3)
}
\references{
\subsection{Binary treatments}{
\itemize{
\item \code{estimand = "ATO"}
}

Li, F., Morgan, K. L., & Zaslavsky, A. M. (2018). Balancing covariates via propensity score weighting. \emph{Journal of the American Statistical Association}, 113(521), 390–400. \doi{10.1080/01621459.2016.1260466}
\itemize{
\item \code{estimand = "ATM"}
}

Li, L., & Greene, T. (2013). A Weighting Analogue to Pair Matching in Propensity Score Analysis. \emph{The International Journal of Biostatistics}, 9(2). \doi{10.1515/ijb-2012-0030}
\itemize{
\item \code{estimand = "ATOS"}
}

Crump, R. K., Hotz, V. J., Imbens, G. W., & Mitnik, O. A. (2009). Dealing with limited overlap in estimation of average treatment effects. \emph{Biometrika}, 96(1), 187–199. \doi{10.1093/biomet/asn055}
\itemize{
\item Other estimands
}

Austin, P. C. (2011). An Introduction to Propensity Score Methods for Reducing the Effects of Confounding in Observational Studies. \emph{Multivariate Behavioral Research}, 46(3), 399–424. \doi{10.1080/00273171.2011.568786}
\itemize{
\item Marginal mean weighting through stratification
}

Hong, G. (2010). Marginal mean weighting through stratification: Adjustment for selection bias in multilevel data. \emph{Journal of Educational and Behavioral Statistics}, 35(5), 499–531. \doi{10.3102/1076998609359785}
\itemize{
\item Bias-reduced logistic regression
}

See references for the \CRANpkg{brglm2} package.
\itemize{
\item Firth corrected logistic regression
}

Puhr, R., Heinze, G., Nold, M., Lusa, L., & Geroldinger, A. (2017). Firth’s logistic regression with rare events: Accurate effect estimates and predictions? \emph{Statistics in Medicine}, 36(14), 2302–2317. \doi{10.1002/sim.7273}
\itemize{
\item SAEM logistic regression for missing data
}

Jiang, W., Josse, J., & Lavielle, M. (2019). Logistic regression with missing covariates — Parameter estimation, model selection and prediction within a joint-modeling framework. \emph{Computational Statistics & Data Analysis}, 106907. \doi{10.1016/j.csda.2019.106907}
}

\subsection{Multi-Category Treatments}{
\itemize{
\item \code{estimand = "ATO"}
}

Li, F., & Li, F. (2019). Propensity score weighting for causal inference with multiple treatments. \emph{The Annals of Applied Statistics}, 13(4), 2389–2415. \doi{10.1214/19-AOAS1282}
\itemize{
\item \code{estimand = "ATM"}
}

Yoshida, K., Hernández-Díaz, S., Solomon, D. H., Jackson, J. W., Gagne, J. J., Glynn, R. J., & Franklin, J. M. (2017). Matching weights to simultaneously compare three treatment groups: Comparison to three-way matching. \emph{Epidemiology} (Cambridge, Mass.), 28(3), 387–395. \doi{10.1097/EDE.0000000000000627}
\itemize{
\item Other estimands
}

McCaffrey, D. F., Griffin, B. A., Almirall, D., Slaughter, M. E., Ramchand, R., & Burgette, L. F. (2013). A Tutorial on Propensity Score Estimation for Multiple Treatments Using Generalized Boosted Models. \emph{Statistics in Medicine}, 32(19), 3388–3414. \doi{10.1002/sim.5753}
\itemize{
\item Marginal mean weighting through stratification
}

Hong, G. (2012). Marginal mean weighting through stratification: A generalized method for evaluating multivalued and multiple treatments with nonexperimental data. \emph{Psychological Methods}, 17(1), 44–60. \doi{10.1037/a0024918}
}

\subsection{Continuous treatments}{

Robins, J. M., Hernán, M. Á., & Brumback, B. (2000). Marginal Structural Models and Causal Inference in Epidemiology. \emph{Epidemiology}, 11(5), 550–560.
\itemize{
\item Using non-normal conditional densities
}

Naimi, A. I., Moodie, E. E. M., Auger, N., & Kaufman, J. S. (2014). Constructing Inverse Probability Weights for Continuous Exposures: A Comparison of Methods. \emph{Epidemiology}, 25(2), 292–299. \doi{10.1097/EDE.0000000000000053}
\itemize{
\item SAEM linear regression for missing data
}

Jiang, W., Josse, J., & Lavielle, M. (2019). Logistic regression with missing covariates — Parameter estimation, model selection and prediction within a joint-modeling framework. \emph{Computational Statistics & Data Analysis}, 106907. \doi{10.1016/j.csda.2019.106907}
}
}
\seealso{
\code{\link[=weightit]{weightit()}}, \code{\link[=weightitMSM]{weightitMSM()}}, \code{\link[=get_w_from_ps]{get_w_from_ps()}}
}
