\name{pairs.copuladata}
\alias{pairs.copuladata}

\title{Pairs Plot of Copula Data}

\description{
  This function provides pair plots for copula data. Using default setting it plots bivariate contour plots on the lower panel, scatter plots and correlations on the upper panel and histograms on the diagonal panel.
}

\usage{
\method{pairs}{copuladata}(x, labels = names(x), ...,
      lower.panel = lp.copuladata,
      upper.panel = up.copuladata,
      diag.panel = dp.copuladata,
      label.pos = 0.85, cex.labels = 1, gap = 0,
      method = "kendall", ccols = terrain.colors(30),
      margins = "norm", margins.par = 0)
}

\arguments{
  \item{x}{\code{copuladata} object.}
  \item{labels}{variable names/labels.}
  \item{\dots}{other graphical parameters (see \code{\link[graphics]{par}}).}
  \item{lower.panel}{panel function to be used on the lower diagonal panels (if not supplied, a default function is used)}
  \item{upper.panel}{panel function to be used on the upper diagonal panels (if not supplied, a default function is used)}
  \item{diag.panel}{panel function to be used on the diagonal panels (if not supplied, a default function is used)}
  \item{label.pos}{y position of labels in the diagonal panel; default: \code{label.pos = 0.85}.}
  \item{cex.labels}{magnification to be used for the labels of the diagonal panel; default: \code{cex.labels = 1}.}
  \item{gap}{distance between subplots, in margin lines; default: \code{gap = 0}.}
If the default panel function \code{upper.panel} is used, the following parameters can be set additionally:
\item{method}{a character string indicating which correlation coefficients are computed. One of \code{"pearson"}, \code{"kendall"} (default), or \code{"spearman"}}
If the default panel function \code{lower.panel} (see \code{\link{BiCopMetaContour}}) is used, the following parameters can be set additionally:
\item{ccols}{colour to be used for the contour plots; default: \code{ccols = terrain.colors(30)}.}
\item{margins}{character; margins for the contour plots. Possible margins are:\cr
                 \code{"norm"} = standard normal margins (default)\cr
                 \code{"t"} = Student t margins with degrees of freedom as specified by \code{margins.par}\cr
                 \code{"gamma"} = Gamma margins with shape and scale as specified by \code{margins.par}\cr
                 \code{"exp"} = Exponential margins with rate as specified by \code{margins.par}\cr
                 \code{"unif"} = uniform margins}
  \item{margins.par}{parameter(s) of the distribution of the margins (of the contour plots) if necessary (default: \code{margins.par = 0}), i.e., 
                     \itemize{
                       \item a positive real number for the degrees of freedom of Student t margins (see \code{\link{dt}}),
                        \item a 2-dimensional vector of positive real numbers for the shape and scale parameters of Gamma margins (see \code{\link{dgamma}}),
                        \item a positive real number for the rate parameter of exponential margins (see \code{\link{dexp}}). 
                     }}
}

\note{
If the default panel functions are used \cr
\itemize{
\item \code{col} changes only the colour of the points in the scatter plot (\code{upper.panel}) \cr
\item \code{cex} changes only the magnification of the points in the scatter plot (\code{upper.panel})
}
}


\author{Tobias Erhardt}

\seealso{\code{\link[graphics]{pairs}}, \code{\link{as.copuladata}}, \code{\link{BiCopMetaContour}}}

\examples{
data(daxreturns)

data <- as.copuladata(daxreturns)
sel <- c(4,5,14,15)

## pairs plot with default settings
pairs(data[sel])

## pairs plot with custom settings
nlevels <- 20
pairs(data[sel], cex = 2, pch = 1, col = "black",
      diag.panel = NULL, label.pos = 0.5,
      cex.labels = 2.5, gap = 1,
      method = "pearson", ccols = heat.colors(nlevels),
      margins = "gamma", margins.par = c(1,1))
      
## pairs plot with own panel functions
up <- function(x, y) {
  # upper panel: empirical contour plot
  op <- par(usr = c(-3, 3, -3, 3), new = TRUE)
  BiCopMetaContour(x, y, bw = 2, levels = c(0.01, 0.05, 0.1, 0.15, 0.2),
                    # exponential margins
                    margins = "exp", margins.par = 1,
                    axes = FALSE)
  on.exit(par(op))
}

lp <- function(x, y) {
  # lower panel: scatter plot (copula data) and correlation
  op <- par(usr = c(0, 1, 0, 1), new = TRUE)
  points(x, y, pch = 1, col = "black")
  r <- cor(x, y, method = "spearman") # Spearman's rho
  txt <- format(x = r, digits = 3, nsmall = 3)[1]
  text(x = 0.5, y = 0.5, labels = txt, cex = 1 + abs(r) * 2, col = "blue")
  on.exit(par(op))
}

dp <- function(x) {
  # diagonal panel: histograms (copula data)
  op <- par(usr = c(0, 1, 0, 1.5), new = TRUE)
  hist(x, freq = FALSE, add = TRUE, col = "brown", border = "black", main = "")
  abline(h = 1, col = "black", lty = 2)
  on.exit(par(op))
}

nlevels <- 20
pairs(data[sel],
      lower.panel = lp, upper.panel = up, diag.panel = dp, gap = 0.5)
}
