\name{C.FRNN.FRST}
\alias{C.FRNN.FRST}
\title{The fuzzy-rough nearest neighbour algorithm}
\usage{
  C.FRNN.FRST(decision.table, newdata, control = list())
}
\arguments{
  \item{decision.table}{a \code{"DecisionTable"} class
  representing the decision table. See
  \code{\link{SF.asDecisionTable}}.  It should be noted
  that the data must be numeric values instead of
  string/char.}

  \item{newdata}{a \code{"DecisionTable"} class
  representing data for the test process.

  See \code{\link{SF.asDecisionTable}}.}

  \item{control}{a list of other parameters as follows.
  \itemize{ \item \code{type.LU}: a type of lower and upper
  approximations. See Section \code{Details}. The default
  value is \code{type.LU = "implicator.tnorm"}.  \item
  \code{k}: the number of neighbours. It should be taken
  into account that this value could affect the accuracy.
  The default value is 5.  \item \code{type.aggregation}:
  the type of aggregation operator. See
  \code{\link{BC.IND.relation.FRST}}.  The default value is
  \code{type.aggregation = c("t.tnorm", "lukasiewicz")}.
  \item \code{type.relation}: the type of relation. See
  \code{\link{BC.LU.approximation.FRST}}.

  The default value is \code{c("tolerance", "eq.1")}.
  \item \code{type.implicator}: the type of implicator
  operator.

  See \code{\link{BC.LU.approximation.FRST}}. The default
  value is \code{"lukasiewicz"}.  \item \code{q.some}: a
  vector of values of alpha and beta parameter of VQRS.

  See \code{\link{BC.LU.approximation.FRST}}. The default
  value is \code{c(0.1, 0.6)}.  \item \code{q.most}: a
  vector of values of alpha and beta parameter of VQRS.

  See \code{\link{BC.LU.approximation.FRST}}. The default
  value is \code{c(0.2, 1)}.  }}
}
\value{
  A matrix of predicted classes of newdata.
}
\description{
  It is used to predict new datasets/patterns based on the
  fuzzy-rough nearest neighbour algorithm (FRNN) proposed
  by (R. Jensen and C. Cornelis, 2011).
}
\details{
  This method uses the fuzzy lower and upper approximation
  to improve the fuzzy nearest neighbour (FNN) algorithm.
  This algorithm assigns a class to a target instance
  \eqn{t} as follows. \itemize{ \item Determine \eqn{k}
  nearest neighbors considering their similarity to new
  patterns. \item Assign new patterns to the class based on
  maximal value of fuzzy lower and upper approximation. If
  a value of fuzzy lower approximation is high, it shows
  that neighbours of newdata belong to a particular class,
  e.g. \code{C}. On the other hand, a high value of fuzzy
  upper approximation means that at least one neighbour
  belongs to that class. }

  In this function, we provide two approaches based on
  types of fuzzy lower and upper approximations. The
  following is a list of the considered approximations:
  \itemize{ \item \code{"implicator.tnorm"}: It refers to
  lower and upper approximations based on implicator/t-norm
  approach.  For more detail, it can be seen in
  \code{\link{BC.LU.approximation.FRST}}. When using this
  approach, we need to assign the \code{control} parameter
  as follows:

  \code{control <- list(type.LU = "implicator.tnorm", k,}

  \code{type.aggregation, type.relation, t.implicator)}

  The detailed description of the components in the
  \code{control} parameter can be seen in

  \code{\link{BC.LU.approximation.FRST}}.

  \item \code{"vqrs"}: It refers to lower and upper
  approximations based on vaquely quantified rough sets.
  For more detail, it can be seen in
  \code{\link{BC.LU.approximation.FRST}}. When using this
  approach, we need to assign the \code{control} parameter
  as follows:

  \code{control <- list(type.LU = "vqrs", k, q.some,
  q.most,}

  \code{type.relation, type.aggregation)}

  The detailed description of the components in the
  \code{control} parameter can be seen in

  \code{\link{BC.LU.approximation.FRST}}. }
}
\examples{
#############################################################
## In this example, we are using Iris dataset.
## It should be noted that since the values of the decision attribute are strings,
## they should be transformed into numeric values using unclass()
#############################################################
data(iris)
## shuffle the data
set.seed(2)
irisShuffled <- iris[sample(nrow(iris)),]

## transform values of the decision attribute into numerics
irisShuffled[,5] <- unclass(irisShuffled[,5])

## split the data into training and testing data
iris.training <- irisShuffled[1:105,]
iris.testing <- irisShuffled[106:nrow(irisShuffled),1:4]

colnames(iris.training) <- c("Sepal.Length", "Sepal.Width", "Petal.Length",
                       "Petal.Width", "Species")

## convert into a standard decision table
decision.table <- SF.asDecisionTable(dataset = iris.training, decision.attr = 5,
                                     indx.nominal = c(5))
tst.iris <- SF.asDecisionTable(dataset = iris.testing)

###### FRNN algorithm using lower/upper approximation:
###### Implicator/tnorm based approach
control <- list(type.LU = "implicator.tnorm", k = 20,
                type.aggregation = c("t.tnorm", "lukasiewicz"),
                type.relation = c("tolerance", "eq.1"), t.implicator = "lukasiewicz")
\dontrun{res.1 <- C.FRNN.FRST(decision.table = decision.table, newdata = tst.iris,
                             control = control)}

###### FRNN algorithm using VQRS
control <- list(type.LU = "vqrs", k = 20, q.some = c(0.1, 0.6), q.most = c(0.2, 1),
                 type.relation = c("tolerance", "eq.1"),
                 type.aggregation = c("t.tnorm","lukasiewicz"))
\dontrun{res.2 <- C.FRNN.FRST(decision.table = decision.table, newdata = tst.iris,
                             control = control)}
}
\references{
  R. Jensen and C. Cornelis, "Fuzzy-rough Nearest Neighbour
  Classification and Prediction", Theoretical Computer
  Science, vol. 412, p. 5871 - 5884 (2011).
}
\seealso{
  \code{\link{C.FRNN.O.FRST}}, \code{\link{C.POSNN.FRST}}
}

