% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getter_functions.R
\name{retrieve}
\alias{retrieve}
\alias{retrieve.PubChemInstance}
\alias{retrieve.PubChemInstanceList}
\alias{retrieve.PC_Substance}
\alias{retrieve.PugViewInstance}
\alias{retrieve.PugViewSection}
\title{Retrieve Information from PubChem Instances}
\usage{
retrieve(object, ...)

\method{retrieve}{PubChemInstance}(object, .slot = NULL, .to.data.frame = TRUE, .verbose = FALSE, ...)

\method{retrieve}{PubChemInstanceList}(
  object,
  .which = NULL,
  .slot = NULL,
  .to.data.frame = TRUE,
  .combine.all = FALSE,
  ...
)

\method{retrieve}{PC_Substance}(
  object,
  .slot = NULL,
  .idx = 1,
  .to.data.frame = TRUE,
  .verbose = FALSE,
  ...
)

\method{retrieve}{PugViewInstance}(object, .slot = NULL, .to.data.frame = TRUE, ...)

\method{retrieve}{PugViewSection}(object, .slot = NULL, .to.data.frame = FALSE, ...)
}
\arguments{
\item{object}{An object returned from a PubChem request.}

\item{...}{Additional arguments passed to other methods.}

\item{.slot}{A string specifying which slot to return. Should not be NULL or length of >1 with some exceptions. See the notes for details.}

\item{.to.data.frame}{A logical value. If TRUE, the returned object will be converted into a data.frame (or tibble).
If conversion to a data.frame fails, a list will be returned with a warning. Be cautious with complex lists
(i.e., many elements nested within each other) as it may be time-consuming to convert such lists into a data frame.
Additionally, \code{.to.data.frame} is ignored in specific scenarios.}

\item{.verbose}{A logical value. Should the resulting object be printed to the R console? If TRUE, the object is returned invisibly
and the output is printed nicely to the R console. This option may not be available for some slots (or classes).
See Notes/Details.}

\item{.which}{A character value. This is the identifier of the PubChem request that will be extracted from the complete list. It is ignored if \code{.combine.all = TRUE}.}

\item{.combine.all}{a logical value. If TRUE, the properties of all requested instances are combined into a single data frame (or a list if \code{.to.data.frame = FALSE}).}

\item{.idx}{An integer indicating which substance result should be returned. A PubChem request may return multiple
substances in the output. \code{.idx} specifies the index of the substance to be extracted from the complete list.}
}
\description{
This generic function extracts a specific slot from a PubChem instance.
}
\note{
If the object is from the \code{'PC_Properties'} class, the \code{.slot} can be defined as NULL. If \code{.slot = NULL}, \code{retrieve()} will return all available properties. If \code{'object'} is of class other than \code{'PC_Properties'}, \code{.slot} should be length of 1.

\subsection{Extracting multiple slots.}{
In some cases, it may be practical to extract multiple slots from \code{'object'}. For example, one may wish to extract properties from the output of \link{get_properties} by running the functions in a loop. See codes below for a practical example:

\preformatted{
  library(dplyr)

  props <- get_properties(
    properties = c("MolecularWeight", "MolecularFormula", "HBondDonorCount",
                   "HBondAcceptorCount", "InChIKey", "InChI"),
    identifier = 2244,
    namespace = "cid",
    propertyMatch = list(
      .ignore.case = TRUE,
      type = "contain"
    )
  )

  bind_columns <- function(x, ...){
    part1 <- x[[1]][ ,"Identifier"]
    part2 <- lapply(x, "[", 2) %>%
      bind_cols()

    bind_cols(part1, part2)
  }

  propsToExtract <- c("MolecularWeight", "MolecularFormula", "HBondDonorCount")
  tmp <- lapply(propsToExtract, retrieve, object = props, .which = "2244")
  bind_columns(tmp)
}}


\subsection{Use of the \code{'.verbose'} argument}{
\code{retrieve} returns output silently (invisibly) when \code{.verbose = TRUE}. However, the function behaves differently
under the following scenarios:
\itemize{
  \item{\code{.verbose} is ignored if \code{.combine.all = TRUE}. The output is returned silently.}
  \item{\code{.verbose} is ignored if the requested slot is not printable to the R console because it is too complicated to print.}
}
}
}
\section{Details on \code{'PugViewInstance'} and \code{'PugViewSection'}}{

The PugView API returns a detailed list related to PubChem requests. The 'Section' slot in this list is structured into
a sub-class called \code{'PugViewSection'}. This object contains information organized through several sections (or sub-sections),
which can be retrieved using \emph{section-specific} functions such as \link{section} and \link{sectionList}.

The function argument \code{.to.data.frame} is ignored if the "Section" slot is being extracted from the complete list.
For other slots, \code{.to.data.frame} is considered as usual. See examples for usage.
}

\examples{
compounds <- get_compounds(
  identifier = c("aspirin", "ibuprofen", "rstudio"),
  namespace = "name"
 )

# Extract information for "aspirin"
aspirin <- instance(compounds, "aspirin")
# print(aspirin)

# Extract a specific slot from the "aspirin" compound.
retrieve(aspirin, "props", .to.data.frame = TRUE)

# Examples (PubChemInstanceList)
retrieve(compounds, "aspirin", "props", .to.data.frame = TRUE)

# Verbose Assay References to R Console
assays <- get_assays(identifier = c(1234, 7815), namespace = "aid")

instance(assays, "7815")
retrieve(assays, "7815", "xref", .verbose = TRUE)

# Print assay protocol to R console (if available)
# Note that it may be too long to print for some assays.
# retrieve(assays, "1234", "protocol", .verbose = TRUE)

# No protocol is available for assay "1234".
# retrieve(assays, "7815", "protocol", .verbose = TRUE)

# Ignores ".verbose" and ".which" if ".combine.all = TRUE".
retrieve(assays, .slot = "xref", .verbose = TRUE, .combine.all = TRUE)

### PUG VIEW EXAMPLES ###
pview <- get_pug_view(identifier = "2244", annotation = "data", domain = "compound")

# PugViewSectionList object.
# This object contains all the section information related to the PubChem request.
sect <- retrieve(pview, .slot = "Section")
print(sect)

retrieve(pview, .slot = "RecordType", .to.data.frame = TRUE)

}
