"_PACKAGE"

# Import
#' @importFrom Rcpp sourceCpp
#' @importFrom stats runif
#' @importFrom foreach %dopar%
#' @importFrom foreach foreach
#' @importFrom doParallel registerDoParallel
#' @importFrom doParallel stopImplicitCluster
#' @importFrom stats optim


# Package documentation
#' @useDynLib GPpenalty, .registration = TRUE
#' @name GPpenalty-package
#' @title GPpenalty
#'
#' @description Implements maximum likelihood estimation for Gaussian processes, supporting both isotropic and anisotropic models with predictive capabilities.
#' Includes penalized likelihood estimation using score-based metrics that account for uncertainty and cross validation techniques for tuning parameter selection.
#' Designed specifically for small datasets.
#'
#' @section Functions:
#' \itemize{
#'   \item \code{\link{mle_gp}}: The function computes maximum likelihood estimates for the lengthscale, scale, mu, and nugget (g) parameters using \code{optim},
#'              with options to fix or assume zero for certain parameters.
#'   \item \code{\link{predict_gp}}: Computes the posterior mean and covariance matrix for a given set of
#'              input locations based on a fitted model.
#'   \item \code{\link{gp_cv}}: Performs cross-validation to select an optimal tuning parameter for penalized MLE of the lengthscale parameter in Gaussian processes.
#'   \item \code{\link{mle_penalty}}: Computes penalized maximum likelihood estimates for the lengthscale parameter using \code{optim}.
#'   \item \code{\link{score}}: Calculates a score value. Higher score values indicate better fits.
#' }
#'
#' @examples
#' \donttest{
#' #### define function ###
#' f_x <- function(x) {
#' return(sin(2*pi*x) + x^2)
#' }
#'
#' ### x and y ###
#' x <- runif(8, min=0, max=1)
#' y <- f_x(x)
#' x.test <- runif(100, min=0, max=1)
#' y.test <- f_x(x.test)
#'
#' ### no penalization ###
#' # fit
#' fit <- mle_gp(y, x)
#' # prediction
#' pred <- predict_gp(fit, x.test)
#'
#' # evaluate the predictive performance with score
#' score_value <- score(y.test, pred$mup, pred$Sigmap)
#'
#' ### penalization ###
#' # leave-one-out cross validation
#' loocv.lambda <- gp_cv(y, x)
#' # fit
#' fit.loocv <- mle_penalty(loocv.lambda)
#' # prediction
#' pred.loocv <- predict_gp(fit.loocv, x.test)
#'
#' # k-fold cross validation with the score metric
#' kfold.score <- gp_cv(y, x, k=4)
#' # fit
#' fit.kfold.score <- mle_penalty(kfold.score)
#' # prediction
#' pred.kfold.score <- predict_gp(fit.kfold.score, x.test)
#'
#' # k-fold cross validation with the mse metric
#' kfold.mse <- gp_cv(y, x, k=4, metric="mse")
#' # fit
#' fit.kfold.mse <- mle_penalty(kfold.mse)
#' # prediction
#' pred.kfold.mse <- predict_gp(fit.kfold.mse, x.test)
#' }
#'
#'
#'

NULL
