% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Fit.R
\name{Fit}
\alias{Fit}
\title{The Fitting Function of \code{GPM} Package}
\usage{
Fit(X, Y, CorrType = "G", Eps = 10^(seq(-1, -12)), Nopt = 5,
  TraceIt = 0, MaxIter = 100, Seed = 1, LowerBound = NULL,
  UpperBound = NULL, Criterion = "LOOCV", StopFlag = 0, Progress = 0)
}
\arguments{
\item{X}{Matrix containing the training (aka design or input) data points. The rows and columns of \code{X} denote individual observation settings and input dimension, respectively.}

\item{Y}{Matrix containing the output (aka response) data points. The rows and columns of \code{Y} denote individual observation responses and output dimension, respectively.}

\item{CorrType}{The type of the correlation function of the GP model. Choices include \code{'G'} (default), \code{'PE'}, \code{'LBG'}, and \code{'LB'}. See the \code{references} for the details. For smooth (or analytic) functions, choose either}

\item{Eps}{A vector containing the smallest eigen value(s) that the correlation matrix is allowed to have. The elements of Eps must be in [0, 1] and sorted in a descending order.}

\item{Nopt}{The number of times the log-likelihood function is optimized when Eps[1] is used to constraint the smallest eigen value that the correlation matrix is allowed to have. Higher \code{Nopt} will increase fitting time as well as the chances of finding the global optimum.}

\item{TraceIt}{Non-negative integer. If positive, tracing information on the progress of the optimization is \strong{printed}. There are six levels of tracing (see \code{\link{optim}}) and higher values will produce more tracing information.}

\item{MaxIter}{Maximum number of iterations allowed for each optimization (see \code{\link{optim}}).}

\item{Seed}{An integer for the random number generator. Use this to make the results reproducible.}

\item{LowerBound, UpperBound}{To estimate the scale (aka roughness) parameters of the correlation function, the feasible range should be defined. \code{LowerBound} and \code{UpperBound} are vectors determining, resepectively, the lower and upper bounds and their length depends on the parametric form of the correlation function (see \code{reference 1} for the details).}

\item{Criterion}{Either \code{LOOCV} or \code{LogL}. Criterion used to determine the most optimum constraint (i.e. an element of \code{Eps}).}

\item{StopFlag}{\code{Flag} indicating whether the optimization must be stopped if the constraint on the correlation matrix is inactive for two consecuitvie elements of \code{Eps}.}

\item{Progress}{\code{Flag} indicating if the fitting process should be summarized. Set it to \code{!=0} to turn it on.}
}
\value{
Model A list containing the following components:
\itemize{
\item{\code{CovFunc}} {A list containing the type and estimated parameters of the correlation function.}
\item{\code{Data}} {A list storing the original (but scaled) data.}
\item{\code{Details}} {A list of some parameters (used in prediction) as well as some values reporting the total run-time (\code{cost}), leave-one-out cross-validation error (\code{LOOCV}), the added nugget (\code{Nug_opt}) for satisfying the constraint on the smallest eigen value of the correlation matrix.}
\item{\code{Opt_History}} {The optimization history.}
\item{\code{Setting}} {The default/provided settings for running the code.}
}
}
\description{
Fits a Gaussian process (GP) to a set of simulation data as described in \code{reference 1}. Both the inputs and outputs can be multi-dimensional. The outputs can be noisy.
}
\examples{
# 1D example: Fit a model (with default settings) and evaluate the performance
# by computing the root mean squared error (RMSE) in prediction.
library(lhs)
X <- 5*maximinLHS(10, 1)
Y <- 2*sin(2*X) + log(X+1)
M <- Fit(X, Y)
XF <- matrix(seq(0, 5, length.out = 100), 100, 1)
YF <- Predict(XF, M)
RMSE <- sqrt(mean((YF$YF - (2*sin(2*XF) + log(XF+1)))^2))

# 1D example: Fit a model, evaluate the performance, and plot the response
# along with 95\% prediction interval
X <- 10*maximinLHS(25, 1) - 5
Y <- X*cos(pi*X)
M <- Fit(X, Y)
XF <- matrix(seq(-5, 5, length.out = 100), 100, 1)
YF <- Predict(XF, M)
RMSE <- sqrt(mean((YF$YF - (XF*cos(pi*XF)))^2))
Draw(M, 1, res = 100)

\dontrun{
# 2D example: Fit a model, evaluate the performance, and plot the response
# surface along with 95\% prediction interval
X <- 2*maximinLHS(20, 2) - 1
Y <- X[, 1]^2 + X[, 2]^2
M <- Fit(X, Y, CorrType = "PE")
XF <- 2*maximinLHS(100, 2) - 1
YF <- Predict(XF, M)
RMSE <- sqrt(mean((YF$YF - (XF[, 1]^2 + XF[, 2]^2))^2))
library(lattice)
Draw(M, c(1, 1), res = 20, PI95=1)

# 2D example: Plot the previous model wrt X1 in the [-2, 2]
# interval with X2=1
Draw(M, c(1, 0), LB = -2, UB = 2, res = 20, PI95=1)

# 3D example: Compare the performance of Gaussian ("G") and lifted Browninan
# with Gamma=1 ("LBG")
X <- 2*maximinLHS(50, 3) - 1
Y <- cos(X[, 1]^2) + 2*sin(X[, 2]^2) + X[, 3]^2
M_G <- Fit(X, Y)
M_LBG <- Fit(X, Y, CorrType = "LBG")
XF <- 2*maximinLHS(1000, 3) - 1
YF_G <- Predict(XF, M_G)
YF_LBG <- Predict(XF, M_LBG)
RMSE_G <- sqrt(mean((YF_G$YF - (cos(XF[, 1]^2) + 2*sin(XF[, 2]^2) + XF[, 3]^2))^2))
RMSE_LBG <- sqrt(mean((YF_LBG$YF - (cos(XF[, 1]^2) + 2*sin(XF[, 2]^2) + XF[, 3]^2))^2))

# 3D example: Draw the response in 2D using the M_G model when X3=0
Draw(M_G, c(1, 1, 0), PI95 = 0, Values = 0, X1Label = 'Input 1', X2Label = 'Input 2')
}
}
\references{
\enumerate{
\item Bostanabad, R., Kearney, T., Tao, S. Y., Apley, D. W. & Chen, W. (2018) Leveraging the nugget parameter for efficient Gaussian process modeling. International Journal for Numerical Methods in Engineering, 114, 501-516.
\item M. Plumlee, D.W. Apley (2016). Lifted Brownian kriging models, Technometrics.
}
}
\seealso{
\code{\link[stats]{optim}} for the details on \code{L-BFGS-B} algorithm used in optimization.\cr
\code{\link[GPM]{Predict}} to use the fitted GP model for prediction.\cr
\code{\link[GPM]{Draw}} to plot the response via the fitted model.
}
