\name{DTRreg}
\alias{DTRreg}
\alias{print.DTRreg}
\alias{DTRreg.trunc}
\title{DTR Estimation and Inference via G-estimation and Dynamic WOLS}
\description{
Dynamic treatment regimen estimation and inference via G-estimation and dynamic WOLS.  Estimation of blip model parameters for multi-stage data.
}

\usage{
DTRreg(outcome, blip.mod, treat.mod, tf.mod, data = NULL,
        method = "gest", weight = "default", var.estim = "none",
        B = 200, M = 0, truncate = 0, verbose = "FALSE",
        interrupt = "FALSE", treat.range = NULL, missing = "default",
        interactive = FALSE, treat.mod.man = NULL, type = "DTR")
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{outcome}{The outcome variable.}
  \item{blip.mod}{A list of formula objects specifying covariates of a (linear) blip function for each stage in order. No dependent variable should be specified.}
  \item{treat.mod}{A list of formula objects specifying the treatment model for each stage in order. Treatment variable should be included as the dependent variable. If treatment is binary a logistic regression model will be used, otherwise a linear regression model will be used.}
  \item{tf.mod}{A list of formula objects specifying covariates of a (linear) treatment-free model for each stage in order. No dependent variable should be specified.}
  \item{data}{A data frame containing all necessary covariates contained in the above models.}
  \item{method}{The DTR method to be used, choose "dwols" for dynamic WOLS, "gest" for G-estimation, or "qlearn" for Q-learning.}
  \item{weight}{If using dynamic WOLS the option for the weights used. Default is the form |A - E[A|...]|, "iptw" gives inverse probability of treatment style weights.}
  \item{var.estim}{Covariance matrix estimation method, either "bootstrap" (for either dWOLS or G-estimation) or "sandwich" for recursive sandwich estimation in the G-estimation context.}
  \item{B}{Number of bootstrap samples.}
  \item{M}{Subsample size for m out of n bootstrap. If unspecified this is set to the sample size (i.e. n)}
  \item{truncate}{Bootstrap option. Truncate (a number between 0 and 0.5) will replace the lowest and highest specified proportion of parameter estimates with the relevant quantiles affording some robustness to extreme values when estimating covariance.}
  \item{verbose}{Bootstrap option. If TRUE then estimated time to completion will be printed approximately every 30 seconds.}
  \item{interrupt}{Bootstrap option. If TRUE then user will be given the option to abort if estimated time to completion exceeds 10 minutes.}
  \item{treat.range}{For continuous treatments. Specify the maximum/minimum value that treatments can be take. If unspecified then the minimum/maximum value of observed treatments is used. If you wish to have unrestricted treatments set this option to c(-Inf,+Inf).}
  \item{missing}{If set to "ipcw" and data are missing then inverse probability of censored weights is used with the probability of censoring estimated via logistic regression on the full covariate history up to that point.}
  \item{interactive}{If TRUE on-screen prompts will guide the user through the specification of blip, treatment and treatment-free models.}
  \item{treat.mod.man}{A list of vectors of known treatment weights can be specified to be used instead of those estimated by the routine.}
  \item{type}{If specified as something other than "DTR", DTRreg will take an `effect estimation' (as opposed to a DTR estimation) approach, treating the observed outcome as being equal to an outcome assuming no treatment is received at any stage, plus a blip component at each stage. The main difference is that each stage's pseudo-outcome is generated by subtracting a blip function, rather than adding a regret function as in the DTR framework. Note that most of the DTR-specific output will either be suppressed or irrelevant.}
}
\details{
DTRreg allows the estimation of optimal dynamic treatment regimens (DTRs, also known as adaptive treatment strategies) from multi-stage trials using G-estimation and dynamic weighted ordinary least squares (dWOLS).  Both methods focus on estimating the parameters of the blip: a model of the difference in expected outcome under the observed treatment and some reference treatment (usually a control) at a given stage, assuming identical histories and optimal treatment thereafter.  The reader is referred to Chakraborty and Moodie (2013) for a thorough introduction and review of DTR methods.  The dWOLS method may be used to obtain parameter estimates identical to those from Q-learning (by setting method = "qlearn").  This option is intended primarily for exploratory purposes; the authors note that there is a dedicated R package for Q-learning (qLearn), although it is limited to the 2-stage setting.

Both of these methods require the specification of three models for each stage of the analysis: a treatment model (conditional mean of the treatment variable), a treatment-free model (conditional mean of outcome assuming only reference treatments are used), and a blip model.  Only the blip model must be correctly specified (or over-specified), with consistent parameter estimates obtainable if at least one of the other two models is correctly specified.  Note that all of these must be specified as lists of formula objects, even if only one stage of treatment is considered.

Note that as is conventional, it is assumed a larger value of the outcome is preferred (which can be easily achieved via transformation of your data if necessary).

When treatment is binary, if confidence intervals are computed (via specification of var.estim other than 'none'), then DTRreg will calculate the proportion of subjects at each stage for whom optimal treatment is non-unique.  If this proportion exceeds 0.05 a non-regularity warning will be displayed, along with the proportion of subjects for whom this is the case.  Note that this warning is only displayed if a variance estimation option is selected.
}
\value{An object of class \code{DTR}, a list including elements
 \item{psi}{Blip parameter estimates for each stage of treatment.}
 \item{opt.treat}{Optimal treatment decisions for each subject at each stage of treatment.}
 \item{covmat}{Covariance matrix of blip parameter estimates.}
 \item{regret}{Estimates of the regret for each subject based on observed treatment and blip parameter estimates.}
 \item{beta}{Treatment-free model parameter estimates (note that these may not be consistent).}
 \item{opt.Y}{Predicted optimal outcome under recommended regimen.}
 \item{nonreg}{Non-regularity estimates.}
 The functions coef, predict and confint may be used with such model objects.  The first two have specific help files for their implementation, while confint is used in the same way as the standard confint command, with the exception of the parm option, which is not available.
}
\references{
Chakraborty, B., Moodie, E. E. M. (2013) \emph{Statistical Methods for Dynamic Treatment Regimes}. New York: Springer.

Robins, J. M. (2004) \emph{Optimal structural nested models for optimal sequential decisions}. In Proceedings of the Second Seattle Symposium on Biostatistics, D. Y. Lin and P. J. Heagerty (eds), 189--326. New York: Springer.

Wallace, M. P., Moodie, E. M. (2015) Doubly-Robust Dynamic Treatment Regimen Estimation Via Weighted Least Squares. \emph{Biometrics} \bold{71}(3), 636--644 (doi:10.1111/biom.12306.)
}
\author{Michael Wallace}

\examples{
##################
# example single run of a 2-stage g-estimation analysis
set.seed(1)
# expit function
expit <- function(x) {1 / (1 + exp(-x))}
# sample size
n <- 10000
# variables (X = patient information, A = treatment)
X1 <- rnorm(n)
A1 <- rbinom(n, 1, expit(X1))
X2 <- rnorm(n)
A2 <- rbinom(n, 1, expit(X2))
# blip functions
gamma1 <- A1 * (1 + X1)
gamma2 <- A2 * (1 + X2)
# observed outcome: treatment-free outcome plus blip functions
Y <- exp(X1) + exp(X2) + gamma1 + gamma2 + rnorm(n)
# models to be passed to DTRreg
# blip model
blip.mod <- list(~X1, ~X2)
# treatment model (correctly specified)
treat.mod <- list(A1~X1, A2~X2)
# treatment-free model (incorrectly specified)
tf.mod <- list(~X1, ~X2)

# perform G-estimation
mod1 <- DTRreg(Y, blip.mod, treat.mod, tf.mod, method = "gest")
mod1
##################
}

\keyword{dynamic treatment regimens,adaptive treatment strategies,personalized medicine,g-estimation,dynamic weighted ordinary least squares}
