% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/continuous_ss_sdf.R
\name{continuous_ss_sdf}
\alias{continuous_ss_sdf}
\title{SDF model selection with continuous spike-and-slab prior}
\usage{
continuous_ss_sdf(
  f,
  R,
  sim_length,
  psi0 = 1,
  r = 0.001,
  aw = 1,
  bw = 1,
  type = "OLS",
  intercept = TRUE
)
}
\arguments{
\item{f}{A matrix of factors with dimension \eqn{t \times k}, where \eqn{k} is the number of factors
and \eqn{t} is the number of periods;}

\item{R}{A matrix of test assets with dimension \eqn{t \times N}, where \eqn{t} is the number of periods
and \eqn{N} is the number of test assets;}

\item{sim_length}{The length of monte-carlo simulations;}

\item{psi0}{The hyper-parameter in the prior distribution of risk prices (see \bold{Details});}

\item{r}{The hyper-parameter related to the prior of risk prices (see \bold{Details});}

\item{aw}{The hyper-parameter related to the prior of \eqn{\gamma} (see \bold{Details});}

\item{bw}{The hyper-parameter related to the prior of \eqn{\gamma} (see \bold{Details});}

\item{type}{If \code{type = 'OLS'} (\code{type = 'GLS'}), the function returns Bayesian OLS (GLS) estimates of risk prices. The default is 'OLS'.}

\item{intercept}{If \code{intercept = TRUE} (\code{intercept = FALSE}), we include (exclude) the common intercept in the cross-sectional regression. The default is \code{intercept = TRUE}.}
}
\value{
The return of \code{continuous_ss_sdf} is a list of the following elements:
\itemize{
\item \code{gamma_path}: A \code{sim_length}\eqn{\times k} matrix of the posterior draws of \eqn{\gamma}. Each row represents
a draw. If \eqn{\gamma_j = 1} in one draw, factor \eqn{j} is included in the model in this draw and vice verse.
\item \code{lambda_path}: A \code{sim_length}\eqn{\times (k+1)} matrix of the risk prices \eqn{\lambda} if \code{intercept = TRUE}. Each row represents
a draw. Note that the first column is \eqn{\lambda_c} corresponding to the constant term. The next \eqn{k} columns (i.e., the 2-th -- \eqn{(k+1)}-th columns)
are the risk prices of the \eqn{k} factors. If \code{intercept = FALSE}, \code{lambda_path} is a \code{sim_length}\eqn{\times k} matrix of the risk prices,
without the estimates of \eqn{\lambda_c}.
\item \code{sdf_path}: A \code{sim_length}\eqn{\times t} matrix of posterior draws of SDFs. Each row represents a draw.
\item \code{bma_sdf}: BMA-SDF.
}
}
\description{
This function provides the SDF model selection procedure using the continuous spike-and-slab prior.
See Propositions 3 and 4 in \insertCite{bryzgalova2023bayesian;textual}{BayesianFactorZoo}.
}
\details{
To model the variable selection procedure, we introduce a vector of binary latent variables \eqn{\gamma^\top = (\gamma_0,\gamma_1,...,\gamma_K)},
where \eqn{\gamma_j \in \{0,1\} }. When \eqn{\gamma_j = 1}, factor \eqn{j} (with associated loadings \eqn{C_j}) should be included
in the model and vice verse.

The continuous spike-and-slab prior of risk prices \eqn{\lambda} is
\deqn{ \lambda_j | \gamma_j, \sigma^2 \sim N (0, r(\gamma_j) \psi_j \sigma^2  ) .}
When the factor \eqn{j} is included, we have \eqn{ r(\gamma_j = 1)=1 }.
When the factor is excluded from the model, \eqn{ r(\gamma_j = 0) =r \ll 1 }.
Hence, the Dirac "spike" is replaced by a Gaussian spike, which is extremely concentrated at zero
(the default value for \eqn{r} is 0.001).
If \code{intercept = TRUE}, we choose \eqn{ \psi_j =  \psi \tilde{\rho}_j^\top \tilde{\rho}_j  },
where \eqn{ \tilde{\rho}_j = \rho_j - (\frac{1}{N} \Sigma_{i=1}^{N} \rho_{j,i}  ) \times 1_N }
is the cross-sectionally demeaned vector of factor \eqn{j}'s correlations with asset returns.
Instead, if \code{intercept = FALSE}, we choose \eqn{ \psi_j =  \psi \rho_j^\top \rho_j  }.
In the codes, \eqn{\psi} is equal to the value of \code{psi0}.

The prior \eqn{\pi (\omega)} encoded the belief about the sparsity of the true model using the prior distribution
\eqn{\pi (\gamma_j = 1 | \omega_j) = \omega_j }. Following the literature on the variable selection, we set
\deqn{ \pi (\gamma_j = 1 | \omega_j) = \omega_j,  \ \ \omega_j \sim Beta(a_\omega, b_\omega) . }
Different hyperparameters \eqn{a_\omega} and \eqn{b_\omega} determine whether one a priori favors more parsimonious models or not.
We choose \eqn{a_\omega = 1} (\code{aw}) and \eqn{b_\omega=1} (\code{bw}) as the default values.

For each posterior draw of factors' risk prices \eqn{\lambda^{(j)}_f}, we can define the SDF as
\eqn{m^{(j)}_t = 1 - (f_t - \mu_f)^\top \lambda^{(j)}_f}.The Bayesian model averaging of the SDF (BMA-SDF)
over \eqn{J} draws is
\deqn{m^{bma}_t = \frac{1}{J}  \sum^J_{j=1} m^{(j)}_t.}
}
\examples{

## Load the example data
data("BFactor_zoo_example")
HML <- BFactor_zoo_example$HML
lambda_ols <- BFactor_zoo_example$lambda_ols
R2.ols.true <- BFactor_zoo_example$R2.ols.true
sim_f <- BFactor_zoo_example$sim_f
sim_R <- BFactor_zoo_example$sim_R
uf <- BFactor_zoo_example$uf

## sim_f: simulated strong factor
## uf: simulated useless factor

psi_hat <- psi_to_priorSR(sim_R, cbind(sim_f,uf), priorSR=0.1)
shrinkage <- continuous_ss_sdf(cbind(sim_f,uf), sim_R, 5000, psi0=psi_hat, r=0.001, aw=1, bw=1)
cat("Null hypothesis: lambda =", 0, "for each factor", "\n")
cat("Posterior probabilities of rejecting the above null hypotheses are:",
    colMeans(shrinkage$gamma_path), "\n")

## We also have the posterior draws of SDF: m(t) = 1 - lambda_g \%*\% (f(t) - mu_f)
sdf_path <- shrinkage$sdf_path

## We also provide the Bayesian model averaging of the SDF (BMA-SDF)
bma_sdf <- shrinkage$bma_sdf

}
\references{
\insertRef{bryzgalova2023bayesian}{BayesianFactorZoo}
}
