% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mixr.R
\name{mixr}
\alias{mixr}
\title{Transfer of net rainfall to ungauged catchments}
\usage{
mixr(
  obs,
  sim,
  mdist,
  distance = "rghosh",
  gres = 5,
  weightO = 0.8,
  weightC = 0.2,
  parallel = FALSE,
  cores = NULL,
  power = 1,
  ndonors = 5,
  donors = NULL,
  maxdist = 50000,
  flexible_donor = TRUE,
  cv = FALSE,
  save_donor = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{obs}{"transfR" object of the gauged catchments}

\item{sim}{"transfR" object of the ungauged catchments}

\item{mdist}{the distance matrix between gauged and ungauged catchments as computed by
the function \link{hdist}}

\item{distance}{the method to use for computing distance matrix if \code{mdist} is not provided.
Possible values are "ghosh", "rghosh", "points", "centroids", "combined" as available
in the function \link{hdist}}

\item{gres}{resolution of spatial discretisation (number of points by km²) for Ghosh
distance (see the function \link{hdist})}

\item{weightO}{weight given to the distance between outlets if \code{distance} is "combined"
(see the function \link{hdist})}

\item{weightC}{weight given to the distance between centroids if \code{distance} is "combined"
(see the function \link{hdist})}

\item{parallel}{logical indicating if the computation should be parallelised}

\item{cores}{the number of cores to use for parallel execution if \code{parallel} is TRUE.
If not specified, the number of cores is set to the value of \code{parallel::detectCores()}}

\item{power}{exponent applied in the inverse distance weighting strategy as defined by the function
\link{weightr}}

\item{ndonors}{maximum number of catchments to be used to simulate discharge of an ungauged
catchment as defined by the function \link{weightr}}

\item{donors}{vector of catchments id from which donors are selected. If empty, the \code{ndonors} closest
catchments are used}

\item{maxdist}{maximum distance between a gauged and an ungauged catchment to allow the net rainfall
to be transfered. This threshold is applied on the \code{mdist} distance matrix. If no units is provided,
\code{maxdist} is assumed to be in [m].}

\item{flexible_donor}{boolean indicating if the donor catchments can change during the simulation period
according to the availability of discharge observations. See \link{weightr} for more details}

\item{cv}{boolean indicating if cross validation evaluation should be done. If true, it will estimate
the net rainfall of every gauged catchments (\code{obs}) as if they were ungauged (leave-one-out evaluation)}

\item{save_donor}{boolean indicating if the net rainfall of each of the \code{ndonors} catchments
should be stored in the sim object for further analysis. If true, it is adding three new space-time attributes
in the \code{sim} object called "RnDonor", "Idonor" and "Wdonor" describing the net rainfall, the id and
the weight of the donor catchments respectively}

\item{verbose}{boolean indicating if information messages should be written to the console}
}
\value{
The \code{sim} object incremented by the new computed attributes.
}
\description{
Combine the net rainfall of gauged catchments to simulate the net rainfall
of an ungauged catchment.
}
\details{
This function is a wrapper function for \link{hdist} and \link{weightr} to directly estimate
the net rainfall on a set of ungauged catchments (\code{sim}) from a set of gauged catchments (\code{obs}).
It returns the simulated net rainfall as a new space-time attribute in the \code{sim} object called "RnSim".
The simulated net rainfall of a given ungauged catchment \eqn{i} is a weighted average of the net rainfalls
of \code{ndonors} gauged catchments \eqn{j}:
\deqn{R_n^i =\Sigma_{j=1}^{ndonors} R_n^j \cdot \lambda_j}
where \eqn{\lambda_j} are defined by an inverse distance weighting function (see \link{weightr}).
}
\examples{
\donttest{data(Oudon)
object <- as_transfr(st = Oudon$obs, hl = Oudon$hl)
object <- velocity(object)
object <- uh(object)
object <- lagtime(object)
object <- rapriori(object)
object <- inversion(object, parallel = TRUE, cores = 2)
mdist  <- hdist(x = object, y = object, method = "rghosh")
object <- mixr(obs = object, mdist = mdist, parallel = TRUE, cores=2,
cv = TRUE, flexible_donor = TRUE, save_donor = FALSE)
object <- convolution(object, save_donor = FALSE)
plot(object, i = 1, attribute = c("Qobs", "Qsim"))}
}
\seealso{
\link{hdist}, \link{weightr}
}
