% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inversion.R
\name{inversion}
\alias{inversion}
\alias{inversion.default}
\alias{inversion.units}
\alias{inversion.transfR}
\title{Estimate net rainfall by inversion}
\usage{
inversion(Qobs, ...)

\method{inversion}{default}(Qobs, uh, RnAp, deltat, ...)

\method{inversion}{units}(
  Qobs,
  uh,
  RnAp,
  deltat,
  Bd = 0.01,
  Dd = 1,
  Bp = 0.001,
  Tp = 20,
  Ad = 0.01,
  Ap = 0.9,
  warmup = 10,
  cooldown = 8,
  dosplit = TRUE,
  split = 30,
  fixedpar = TRUE,
  parallel = FALSE,
  cores = NULL,
  ...
)

\method{inversion}{transfR}(Qobs, verbose = TRUE, ...)
}
\arguments{
\item{Qobs}{discharge vector or object of class \code{transfR}. If no unit is provided,
\code{Qobs} is assumed to be in [mm/h]}

\item{...}{further arguments passed to or from other methods}

\item{uh}{unit hydrograph vector}

\item{RnAp}{net rainfall a priori. If no unit is provided, \code{RnAp} is assumed to be in [mm/h]}

\item{deltat}{time step of the time series. If no unit is provided, \code{deltat} is assumed to be in [min]}

\item{Bd}{parameter used to maintain a minimum value of standart deviation for low discharge values.
If no unit is provided, \code{Bd} is assumed to be in [mm/h]}

\item{Dd}{decorrelation time of discharge errors. If no unit is provided, \code{Dd} is assumed to be
in [h]}

\item{Bp}{parameter used to maintain a minimum value of standart deviation for low net rainfall values.
If no unit is provided, \code{Bp} is assumed to be in [mm/h]}

\item{Tp}{decorrelation time of net rainfall errors. If no unit is provided, \code{Tp} is assumed to
be in [h]}

\item{Ad}{parameter equivalent to the coefficient of variation of the discharge measurement error. If
no unit is provided, \code{Ad} is assumed to be dimensionless}

\item{Ap}{parameter equivalent to the coefficient of variation of the net rainfall error. If no unit
is provided, \code{Ap} is assumed to be dimensionless}

\item{warmup}{length of the warmup period. If no unit is provided, \code{warmup} is assumed to be in [days]}

\item{cooldown}{length of the period removed at the end of the simulation. If no unit is provided,
\code{cooldown} is assumed to be in [days]}

\item{dosplit}{boolean, if true the inversion is performed by
subperiods of length defined by \code{split}}

\item{split}{length the subperiods if dosplit is true. If no unit is provided, \code{split} is assumed to be
in [days]}

\item{fixedpar}{boolean, if false Ap and Ad are calibrated dynamically according to the coefficient of variation of
RnAp and Qobs respectively (see details)}

\item{parallel}{boolean, if true the splitting of the inversion
by subperiods is parallelised}

\item{cores}{the number of cores to use for parallel execution if \code{parallel} is TRUE. If not specified, the number of cores is set to the value of \code{parallel::detectCores()}}

\item{verbose}{boolean indicating if information messages should be written to the console}
}
\value{
An object of the same class of \code{Qobs}. If \code{Qobs} is a transfR object,
the same transfR object incremented by the new computed attributes.
}
\description{
Estimate net rainfall by inverse modelling, where the model is a convolution between net rainfall
and a unit hydrograph in order to simulate discharge.
}
\details{
In a convolution between the unit hydrograph (\code{uh}) and net rainfall that is simulating
streamflow at the outltet (\code{Qobs}), and where net rainfall is the only unknown variable, this function estimates
net rainfall by inversion \insertCite{Tarantola1982,Menke1989,Boudhraa2018}{transfR}. It requires an
a priori on this net rainfall (that could be estimated by the function \link{rapriori}), a description
of the errors on the discharge (\code{Ad}, \code{Bd}, \code{Dd}) and on the net rainfall (\code{Ap},
\code{Bp}, \code{Tp}) that are assumed to be Gaussian and unbiased. Default values of these parameters
are taken from \insertCite{deLavenne2016;textual}{transfR}. If \code{fixedpar} is deactivated, \code{Ap}
is estimated at 20% of the coefficient of variation of RnAp, and \code{Ad} estimated at 5% of the coefficient
of variation of Qobs.

It is recommanded to use \code{warmup} and \code{cooldown} periods in order to reduce the problem of oscillations
created by inversion.

If \code{object} is provided, results are stored as a new space-time attribute in the \code{object}
called "RnAp".
}
\examples{
\donttest{data(Oudon)
icatch <- 1 # Catchment index
itime <- 1:1000 # Using the first values for a quicker example
Qobs <- Oudon$obs[["Qobs"]][itime,icatch]
Qspec <- units::set_units(Qobs/st_area(st_geometry(Oudon$obs)[icatch]), "mm/h")
deltat <- units::set_units(1, "h")
uc <- velocity(hl = Oudon$hl[[icatch]])
uh <- uh(hl = Oudon$hl[[icatch]], uc = uc, deltat = units::set_units(1,"h"))$prob
RnAp <- rapriori(Qobs = Qspec, lagtime = lagtime(hl = Oudon$hl[[icatch]], uc = uc),
deltat = deltat)
RnInv <- inversion(Qobs = Qspec, RnAp = RnAp, uh = uh, deltat = deltat)}
}
\references{
\insertRef{Boudhraa2018}{transfR}

\insertRef{deLavenne2016}{transfR}

\insertRef{Menke1989}{transfR}

\insertRef{Tarantola1982}{transfR}
}
\seealso{
\link{rapriori}
}
