#' Get a show's episodes. All of them.
#'
#' \code{trakt.getEpisodeData} pulls detailed episode data.
#' Get details for a show's episodes, e.g. ratings, number of votes,
#' airdates, images, plot overviews…
#'
#' This is basically just an extension of \link[tRakt]{trakt.show.season}, which is used in
#' this function to collect all the episode data.
#' If you only want the episode data for a single season anyway, \code{trakt.show.season}
#' is recommended, yet this function makes some additions.
#' @param target The \code{slug} of the show requested
#' @param season_nums Vector of season numbers, e.g. \code{c(1, 2)}
#' @param extended Defaults to \code{full,images} to get season posters. Can be
#' \code{min}, \code{images}, \code{full}, \code{full,images}
#' @param dropunaired If \code{TRUE} (default), episodes which have not aired yet are dropped.
#' @return A \code{data.frame} containing episode details
#' @export
#' @import plyr
#' @import httr
#' @note This function is mainly for convenience.
#' @family show
#' @examples
#' \dontrun{
#' get_trakt_credentials() # Set required API data/headers
#' breakingbad.episodes <- trakt.getEpisodeData("breaking-bad", c(1, 2, 3, 4, 5))
#' }

trakt.getEpisodeData <- function(target, season_nums, extended = "full", dropunaired = TRUE){
  if (is.null(getOption("trakt.headers"))){
    stop("HTTP headers not set, see ?get_trakt_credentials")
  }
  # Bind variables later used to please R CMD CHECK
  episode <- NULL
  season  <- NULL
  rating  <- NULL

  episodes <- plyr::ldply(season_nums, function(s){
                                          trakt.show.season(target, s, extended)
                                       })

  # Arrange appropriately
  show.episodes       <- transform(episodes, epid = tRakt::pad(season, episode))
  show.episodes$epnum <- 1:(nrow(show.episodes))

  # Convert seasons to factors because ordering
  show.episodes$season <- factor(show.episodes$season,
                                 levels = as.character(1:max(show.episodes$season)),
                                 ordered = T)

  # Add z-scaled episode ratings, scale per season
  if (extended != "min"){
    show.episodes <- plyr::ddply(show.episodes, "season",
                                 transform, zrating.season = as.numeric(scale(rating)))

    # Drop episodes with a timestamp of 0, probably faulty data or unaired
    if (nrow(show.episodes[show.episodes$first_aired != 0, ]) > 0){
      show.episodes <- show.episodes[show.episodes$first_aired != 0, ]
    } else {
      warning("Data is probably faulty: Some first_aired values are 0")
    }

    if (dropunaired){
      show.episodes <- show.episodes[show.episodes$first_aired <= lubridate::now(tzone = "UTC"), ]
    }
  }

  # Append source
  show.episodes$src  <- "trakt.tv"

  return(show.episodes)
}
