% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/spJitter.R
\name{spJitterFinite}
\alias{spJitterFinite}
\title{Random perturbation of spatial points}
\usage{
spJitterFinite(points, candi, x.max, x.min, y.max, y.min, which.point)
}
\arguments{
\item{points}{Integer value, integer vector, data frame or matrix. If
\code{points} is an integer value, it defines the number of points that
should be randomly sampled from \code{candi} to form the starting system
configuration. If \code{points} is a vector of integer values, it contains
the row indexes of \code{candi} that correspond to the points that form the
starting system configuration. If \code{points} is a data frame or matrix,
it must have three columns in the following order: \code{[, "id"]} the
row indexes of \code{candi} that correspond to each point, \code{[, "x"]}
the projected x-coordinates, and \code{[, "y"]} the projected y-coordinates.
Note that in the later case, \code{points} must be a subset of \code{candi}.}

\item{candi}{Data frame or matrix with the candidate locations for the
perturbed points. \code{candi} must have two columns in the following
order: \code{[, "x"]} the projected x-coordinates, and \code{[, "y"]} the
projected y-coordinates.}

\item{x.max,x.min,y.max,y.min}{Numeric value. The minimum and maximum
quantity of random noise to be added to the projected x- and y-coordinates.
The minimum quantity should be equal to, at least, the minimum distance
between two neighbouring candidate locations. The units are the same as of
the projected x- and y-coordinates. If missing, they are estimated from
\code{candi}.}

\item{which.point}{Integer values defining which point should be perturbed.}
}
\value{
A matrix with the jittered projected coordinates of the points.
}
\description{
Perturb the coordinates of spatial points (\sQuote{jittering}).
}
\section{Jittering methods}{

There are two ways of jittering the coordinates. They differ on how the
set of candidate locations is defined. The first method uses an
\emph{infinite} set of candidate locations, that is, any point in the spatial
domain can be selected as the new location of a perturbed point. All that
this method needs is a polygon indicating the boundary of the spatial domain.
This method is not implemented in the \pkg{spsann} package (yet) because it
is computationally demanding: every time a point is jittered, it is necessary
to check if it is inside the spatial domain.

The second method consists of using a \emph{finite} set of candidate
locations for the perturbed points. A finite set of candidate locations is
created by discretizing the spatial domain, that is, creating a fine grid of
points that serve as candidate locations for the perturbed points. This is
the only method currently implemented in the \pkg{spsann} package because it
is one of the least computationally demanding.

Using a finite set of candidate locations has one important inconvenience.
When a point is selected to be jittered, it may be that the new location
already is occupied by another point. If this happens, another location is
iteratively sought for as many times as there are points in \code{points}.
Because the more points there are in \code{points}, the more likely it is
that the new location already is occupied by another point. If a solution is
not found, the point selected to be jittered point is kept in its original
location.

A more elegant method can be defined using a finite set of candidate
locations coupled with a form of \emph{two-stage random sampling} as
implemented in \code{\link[spcosa]{spsample}}. Because the candidate
locations are placed on a finite regular grid, they can be seen as being the
centre nodes of a finite set of grid cells (or pixels of a raster image). In
the first stage, one of the \dQuote{grid cells} is selected with replacement,
i.e. independently of already being occupied by another sample point. The new
location for the point chosen to be jittered is selected within that
\dQuote{grid cell} by simple random sampling. This method guarantees that
any location in the spatial domain can be a candidate location. It also
discards the need to check if the new location already is occupied by
another point. This method is not implemented (yet) in the \pkg{spsann}
package.
}

\section{Distance between two points}{

The distance between two points is computed as the Euclidean distance between
them. This computation assumes that the optimization is operating in the
two-dimensional Euclidean space, i.e. the coordinates of the sample points
and candidate locations should not be provided as latitude/longitude. Package
\pkg{spsann} has no mechanism to check if the coordinates are projected, and
the user is responsible for making sure that this requirement is attained.
}
\examples{
require(sp)
data(meuse.grid)
meuse.grid <- as.matrix(meuse.grid[, 1:2])
meuse.grid <- matrix(cbind(1:dim(meuse.grid)[1], meuse.grid), ncol = 3)
pts1 <- sample(c(1:dim(meuse.grid)[1]), 155)
pts2 <- meuse.grid[pts1, ]
pts3 <- spJitterFinite(points = pts2, candi = meuse.grid, x.min = 40,
                      x.max = 100, y.min = 40, y.max = 100, which.point = 10)
plot(meuse.grid[, 2:3], asp = 1, pch = 15, col = "gray")
points(pts2[, 2:3], col = "red", cex = 0.5)
points(pts3[, 2:3], pch = 19, col = "blue", cex = 0.5)

# Cluster of points
pts1 <- c(1:55)
pts2 <- meuse.grid[pts1, ]
pts3 <- spJitterFinite(points = pts2, candi = meuse.grid, x.min = 40,
                      x.max = 80, y.min = 40, y.max = 80, which.point = 1)
plot(meuse.grid[, 2:3], asp = 1, pch = 15, col = "gray")
points(pts2[, 2:3], col = "red", cex = 0.5)
points(pts3[, 2:3], pch = 19, col = "blue", cex = 0.5)
}
\author{
Alessandro Samuel-Rosa \email{alessandrosamuelrosa@gmail.com}
}
\references{
Edzer Pebesma, Jon Skoien with contributions from Olivier Baume, A. Chorti,
D.T. Hristopulos, S.J. Melles and G. Spiliopoulos (2013).
\emph{intamapInteractive: procedures for automated interpolation - methods
only to be used interactively, not included in \code{intamap} package.} R
package version 1.1-10.

van Groenigen, J.-W. \emph{Constrained optimization of spatial sampling:
a geostatistical approach.} Wageningen: Wageningen University, p. 148, 1999.
}
\seealso{
\code{ssaOptim}, \code{\link[sp]{zerodist}},
\code{\link[base]{jitter}}, \code{\link[geoR]{jitter2d}}.
}
\concept{
jitter perturb
}

