% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smoothContours.R
\name{getSmoothContour}
\alias{getSmoothContour}
\title{Smooth contour from anchors}
\usage{
getSmoothContour(
  anchors = data.frame(time = c(0, 1), value = c(0, 1)),
  len = NULL,
  thisIsPitch = FALSE,
  normalizeTime = TRUE,
  interpol = c("approx", "spline", "loess")[3],
  discontThres = 0.05,
  jumpThres = 0.01,
  loessSpan = NULL,
  valueFloor = NULL,
  valueCeiling = NULL,
  plot = FALSE,
  xlim = NULL,
  ylim = NULL,
  samplingRate = 16000,
  voiced = NULL,
  contourLabel = NULL,
  NA_to_zero = TRUE,
  ...
)
}
\arguments{
\item{anchors}{a numeric vector of values or a list/dataframe with one column
(value) or two columns (time and value). \code{achors$time} can be in ms
(with len=NULL) or in arbitrary units, eg 0 to 1 (with duration determined
by len, which must then be provided in ms). So anchors$time is assumed to
be in ms if len=NULL and relative if len is specified. \code{anchors$value}
can be on any scale.}

\item{len}{the required length of the output contour. If NULL, it will be
calculated based on the maximum time value (in ms) and \code{samplingRate}}

\item{thisIsPitch}{(boolean) is this a pitch contour? If true, log-transforms
before smoothing and plots in both Hz and musical notation}

\item{normalizeTime}{if TRUE, normalizes anchors$time values to range from 0 to 1}

\item{interpol}{deprecated}

\item{discontThres}{deprecated}

\item{jumpThres}{deprecated}

\item{loessSpan}{parameter that controlled the amount of smoothing when
interpolating pitch etc between anchors; passed on to
\code{\link[stats]{loess}}, so only has an effect if interpol = 'loess'}

\item{valueFloor, valueCeiling}{lowser/upper bounds for the contour}

\item{plot}{(boolean) produce a plot?}

\item{xlim, ylim}{plotting options}

\item{samplingRate}{sampling rate used to convert time values to points (Hz)}

\item{voiced, contourLabel}{graphical pars for plotting breathing contours
(see examples below)}

\item{NA_to_zero}{if TRUE, all NAs are replaced with zero}

\item{...}{other plotting options passed to \code{plot()}}
}
\value{
Returns a numeric vector of length \code{len}.
}
\description{
Returns a smooth contour based on an arbitrary number of anchors. Used by
\code{\link{soundgen}} for generating intonation contour, mouth opening, etc.
Although this function is mostly intended to be used internally by soundgen,
and it's slower than directly calling something like approx(), it may be a
convenient "lazy" option because it can both up- and downsample vectors to
the required size without aliasing. Note that pitch contours are treated as a
special case: values are log-transformed prior to smoothing, so that with 2
anchors we get a linear transition on a log scale (as if we were operating
with musical notes rather than frequencies in Hz). Pitch plots have two Y
axes: one showing Hz and the other showing musical notation.
}
\examples{
# downsampling (low-pass filter + decimation, see ?soundgen:::downsample for
# details)
getSmoothContour(rnorm(100), len = 5)

# upsampling
getSmoothContour(c(1, 3, 2), len = 5, interpol = 'approx')
getSmoothContour(c(1, 3, 2), len = 5, interpol = 'loess')

# long format: anchors are a dataframe
a = getSmoothContour(anchors = data.frame(
  time = c(50, 137, 300), value = c(0.03, 0.78, 0.5)),
  normalizeTime = FALSE,
  voiced = 200, valueFloor = 0, plot = TRUE, main = '',
  samplingRate = 16000) # breathing

# short format: anchors are a vector (equal time steps assumed)
a = getSmoothContour(anchors = c(350, 800, 600),
  len = 5500, thisIsPitch = TRUE, plot = TRUE,
  samplingRate = 3500) # pitch

# a single anchor gives constant value
a = getSmoothContour(anchors = 800,
  len = 500, thisIsPitch = TRUE, plot = TRUE, samplingRate = 500)

# two pitch anchors give loglinear F0 change
a = getSmoothContour(anchors = c(220, 440),
  len = 500, thisIsPitch = TRUE, plot = TRUE, samplingRate = 500)

## Two closely spaced anchors produce a pitch jump
# one loess for the entire contour
a1 = getSmoothContour(anchors = list(time = c(0, .15, .2, .7, 1),
    value = c(360, 116, 550, 700, 610)), len = 500, thisIsPitch = TRUE,
    plot = TRUE, samplingRate = 500)
# two segments with a linear transition
a2 = getSmoothContour(anchors = list(time = c(0, .15, .17, .7, 1),
    value = c(360, 116, 550, 700, 610)), len = 500, thisIsPitch = TRUE,
    plot = TRUE, samplingRate = 500)
# two segments with an abrupt jump
a3 = getSmoothContour(anchors = list(time = c(0, .15, .155, .7, 1),
    value = c(360, 116, 550, 700, 610)), len = 500, thisIsPitch = TRUE,
    plot = TRUE, samplingRate = 500)
# compare:
plot(a2)
plot(a3)  # NB: the segment before the jump is upsampled to compensate

# Control the amount of smoothing
getSmoothContour(c(1, 3, 9, 10, 9, 9, 2), len = 100, plot = TRUE,
  loessSpan = NULL)  # default amount of smoothing (depends on dur)
getSmoothContour(c(1, 3, 9, 10, 9, 9, 2), len = 100, plot = TRUE,
  loessSpan = .85)   # more smoothing than default
getSmoothContour(c(1, 3, 9, 10, 9, 9, 2), len = 100, plot = TRUE,
  loessSpan = .5)    # less smoothing
getSmoothContour(c(1, 3, 9, 10, 9, 9, 2), len = 100, plot = TRUE,
  interpol = 'approx')  # linear interpolation (no smoothing)
}
