\name{sim.popn}
\alias{sim.popn}

\title{ Simulate 2-D Population }

\description{
Simulate a Poisson process representing the locations of individual animals.
}

\usage{

sim.popn (D, core, buffer = 100, model2D = 'poisson',
    buffertype = 'rect', covariates = list(sex = c(M=0.5, F=0.5)),
    number.from = 1, Ndist = 'poisson', nsession = 1, details =
    NULL, seed = NULL)

}

\arguments{
  \item{D}{ density animals / hectare (10 000 m\^2) }
  \item{core}{ data frame of points defining the core area }
  \item{buffer}{ buffer radius about core area }
  \item{model2D}{ character string for 2-D distribution (`poisson', `cluster', `IHP') }
  \item{buffertype}{ character string for buffer type }
  \item{covariates}{ list of named covariates }
  \item{number.from}{ integer ID for animal }
  \item{Ndist}{ character string for distribution of number of individuals }
  \item{nsession}{ number of sessions to simulate }
  \item{details}{ optional list with additional parameters }
  \item{seed}{ value for setting .Random.seed - either NULL or an integer}
}
\details{

\code{core} must contain columns `x' and `y'; a \code{traps} object is
suitable. For \code{buffertype = 'rect'}, animals are simulated in the
rectangular area obtained by extending the bounding box of \code{core}
by \code{buffer} metres to top and bottom, left and right. This box has area \eqn{A}.

A notional random covariate `sex' is generated by default.

Each element of \code{covariates} defines a categorical (factor)
covariate with the given probabilities of membership in each class. No
mechanism is provided for generating continuous covariates, but these
may be added later (see Examples).

Ndist may be `poisson' or `fixed'. The number of individuals \eqn{N} has
expected value \eqn{DA}. If \eqn{DA} is non-integer then Ndist = 'fixed'
results in \eqn{N \in \{ \mathrm{trunc}(DA), \mathrm{trunc}(DA)+1 \} }{N
in \{ trunc(DA), trunc(DA)+1 \} }, with probabilities set to yield
\eqn{DA} individuals on average.

If \code{model2D = 'cluster'} then the simulated population approximates a Neyman-Scott
clustered Poisson distribution. Ancillary parameters are passed as
components of \code{details}: details$mu is the fixed number of
individuals per cluster and details$hsigma is the spatial scale
(\eqn{\sigma}{sigma}) of a 2-D kernel for location within each cluster.
The algorithm is
\enumerate{
\item Determine the number of clusters (parents) as a random Poisson variate
with \eqn{\lambda = DA/\mu}{ lambda = DA/mu }
\item Locate each parent by drawing uniform random x- and y-coordinates
\item Generate mu offspring for each parent and locate them by adding random
 normal error to each parent coordinate
\item Apply toroidal wrapping to ensure all offspring locations are inside the buffered area
}

Toroidal wrapping is a compromise. The result is more faithful to the
Neyman-Scott distribution if the buffer is large enough that only a
small proportion of the points are wrapped.

If \code{model2D = 'IHP'} then an inhomogeneous Poisson distribution is fitted.
\code{core} should be a habitat \link{mask} and \code{D} should be a
vector of length equal to the number of cells (rows) in \code{core}. The
number of individuals in each cell is Poisson-distributed with mean \eqn{DA} where
\eqn{A} is the cell area (an attribute of the mask).

The random number seed is managed as in \code{simulate.lm}.

}

\value{

An object of class `popn', a data frame with columns `x' and `y'. Rows
correspond to individuals. Individual covariates (optional) are stored
as a data frame attribute. The initial state of the R random number generator is
stored in the `seed' attribute.

}

\note{  Other \code{buffertype}s will be defined later. (e.g. convex hull, concave) }

\seealso{ \code{\link{popn}}, \code{\link[stats]{simulate}} }

\examples{

temppop <- sim.popn (D = 10, expand.grid(x = c(0,100), y =
    c(0,100)), buffer = 50)

## plot, distinguishing 'M' and 'F'
plot(temppop, pch = 1, cex= 1.5,
    col = c('green','red')[covariates(temppop)$sex])

## add a continuous covariate
## assumes covariates(temppop) is non-null
covariates(temppop)$size <- rnorm (nrow(temppop), mean = 15, sd = 3)
summary(covariates(temppop))

## Neyman-Scott cluster distribution
oldpar <- par(xpd = TRUE, mfrow=c(2,3))
for (h in c(5,15))
for (m in c(1,4,16)) {
    temppop <- sim.popn (D = 10, expand.grid(x = c(0,100),
        y = c(0,100)), model2D = 'cluster', buffer = 100,
        details = list(mu = m, hsigma = h))
    plot(temppop)
    text (50,230,paste(' mu =',m, 'hsigma =',h))
}
par(oldpar)

## Inhomogeneous Poisson distribution
xy <- secrdemo.0$mask$x + secrdemo.0$mask$y - 900
tempD <- xy^2 / 1000
plot(sim.popn(tempD, secrdemo.0$mask, model2D = 'IHP'))

}

\keyword{ datagen }

