#' Compute Fit Summaries for a Latent Structure Estimate
#'
#' This function computes various summaries of the fit of a clustering or feature allocation based on the expected
#' pairwise allocation matrix.
#'
#' @param estimate A clustering or feature allocation.  If \code{estimate} is a length \code{n}
#' vector, it is taken to be a clustering where items \code{i} and
#' \code{j} are in the same cluster if and only if \code{estimate[i] == estimate[j]}.  If \code{estimate}
#' is a binary matrix of \code{n} rows and an arbitrary number of columns, it is taken to be a feature allocation
#' where items \code{i} and \code{j} share \code{m} features if, for \code{k}
#' = 1, 2, ..., the expression \code{estimate[i,k] == estimate[j,k] == 1} is true exactly
#' \code{m} times.
#' @param expectedPairwiseAllocationMatrix A \code{n}-by-\code{n} symmetric matrix
#' whose \code{(i,j)} elements gives the estimated expected number of times that items
#' \code{i} and \code{j} are in the same subset (i.e., cluster or feature).
#'
#' @author David B. Dahl \email{dahl@stat.byu.edu}
#'
#' @examples
#' \donttest{
#' probabilities <- expectedPairwiseAllocationMatrix(iris.clusterings)
#' estimate <- salso(probabilities)
#' latentStructureFit(estimate, probabilities)
#'
#' expectedCounts <- expectedPairwiseAllocationMatrix(USArrests.featureAllocations)
#' estimate <- salso(expectedCounts,"featureAllocation")
#' latentStructureFit(estimate, expectedCounts)
#' 
#' \dontshow{
#' rscala::scalaDisconnect(sdols:::s)
#' }
#' }
#'
#' @seealso \code{\link{expectedPairwiseAllocationMatrix}}, \code{\link{salso}}
#'
#' @export
#' @import rscala

latentStructureFit <- function(estimate, expectedPairwiseAllocationMatrix) {
  if ( ! is.atomic(estimate) ) stop("'estimate' should be a vector of labels giving a clustering or a binary matrix giving a feature allocation.")
  doClustering <- ! is.matrix(estimate)
  epam <- as.expectedPairwiseAllocationMatrix(expectedPairwiseAllocationMatrix,doClustering)
  if ( doClustering ) {
    ref <- s$Clustering(as.integer(estimate))
    ss <- s$ClusteringSummary.sumOfSquares(ref,epam)
    sa <- s$ClusteringSummary.sumOfAbsolutes(ref,epam)
    binder <- s$ClusteringSummary.sumOfAbsolutes(ref,epam) / 2
    vi <- s$ClusteringSummary.lowerBoundVariationOfInformation(ref,epam)
    list("squaredError"=ss,"absoluteError"=sa,"binder"=binder,"lowerBoundVariationOfInformation"=vi)
  } else {
    ref <- scalaPush(estimate,"featureAllocation",s)
    ss <- s$FeatureAllocationSummary.sumOfSquares(ref,epam)
    sa <- s$FeatureAllocationSummary.sumOfAbsolutes(ref,epam)
    list("squaredError"=ss,"absoluteError"=sa)
  }
}

