% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cond_passage.R
\name{cond_passage}
\alias{cond_passage}
\alias{cond_passage,samc,missing,missing,location-method}
\alias{cond_passage,samc,missing,location,location-method}
\title{Conditional Mean First Passage Time}
\usage{
cond_passage(samc, init, origin, dest)

\S4method{cond_passage}{samc,missing,missing,location}(samc, dest)

\S4method{cond_passage}{samc,missing,location,location}(samc, origin, dest)
}
\arguments{
\item{samc}{A \code{\link{samc-class}} object created using the \code{\link{samc}} function.}

\item{init}{Placeholder/not currently implemented.}

\item{origin}{A positive integer or character name representing transient state
\eqn{\mathit{i}}. Corresponds to row \eqn{\mathit{i}} of matrix \eqn{\mathbf{P}}
in the \code{\link{samc-class}} object. When paired with the \code{dest}
parameter, multiple values may be provided as a vector.}

\item{dest}{A positive integer or character name representing transient state
\eqn{\mathit{j}}. Corresponds to column \eqn{\mathit{j}} of matrix \eqn{\mathbf{P}}
in the \code{\link{samc-class}} object. When paired with the \code{origin} parameter,
multiple values may be provided as a vector.}
}
\value{
See Details
}
\description{
Calculate the mean number of steps to first passage
}
\details{
\eqn{\tilde{t}=\tilde{B}_j^{-1}\tilde{F}\tilde{B}_j{\cdot}1}
\itemize{
  \item \strong{cond_passage(samc, dest)}

The result is a vector where each element corresponds to a cell in the landscape,
and can be mapped back to the landscape using the \code{\link{map}} function.
Element \emph{i} is the mean number of steps before absorption starting from
location \emph{i} conditional on absorption into \emph{j}

Note that mathematically, the formula actually does not return a value for when
\emph{i} is equal to \emph{j}. This leads to a situation where the resultant vector
is actually one element short and the index for some of the elements may be shifted.
The \strong{cond_passage()} function fills inserts a \code{0} value for vector indices
corresponding to \emph{i == j}. This corrects the final result so that vector indices
work as expected, and allows the vector to be properly used in the \code{\link{map}} function.

  \item \strong{cond_passage(samc, origin, dest)}

The result is a numeric value representing the mean number of steps before
absorption starting from a given origin conditional on absorption into \emph{j}.

As described above, mathematically the formula does not return a result for when
the \code{origin} and \code{dest} inputs are equal, so the function simply returns a \code{0}
in this case.
}

\strong{WARNING}: This function is not compatible when used with data where there
are states with total absorption present. When present, states representing
total absorption leads to unsolvable linear equations. The only exception to this
is when there is a single total absorption state that corresponds to input to
the \code{dest} parameter. In this case, the total absorption is effectively
ignored when the linear equations are solved.

\strong{WARNING}: This function will crash when used with data representing
a disconnected graph. This includes, for example, isolated pixels or islands
in raster data. This is a result of the transition matrix for disconnected
graphs leading to some equations being unsolvable. Different options
are being explored for how to best identify these situations in data and
handle them accordingly.
}
\section{Performance}{


Any relevant performance information about this function can be found in the
performance vignette: \href{../doc/performance.html}{\code{vignette("performance", package = "samc")}}
}

\examples{
# "Load" the data. In this case we are using data built into the package.
# In practice, users will likely load raster data using the raster() function
# from the raster package.
res_data <- samc::example_split_corridor$res
abs_data <- samc::example_split_corridor$abs
init_data <- samc::example_split_corridor$init


# Make sure our data meets the basic input requirements of the package using
# the check() function.
check(res_data, abs_data)
check(res_data, init_data)

# Setup the details for a random-walk model
rw_model <- list(fun = function(x) 1/mean(x), # Function for calculating transition probabilities
                 dir = 8, # Directions of the transitions. Either 4 or 8.
                 sym = TRUE) # Is the function symmetric?


# Create a `samc-class` object with the resistance and absorption data using
# the samc() function. We use the recipricol of the arithmetic mean for
# calculating the transition matrix. Note, the input data here are matrices,
# not RasterLayers.
samc_obj <- samc(res_data, abs_data, model = rw_model)


# Convert the initial state data to probabilities
init_prob_data <- init_data / sum(init_data, na.rm = TRUE)


# Calculate short- and long-term metrics using the analytical functions
short_mort <- mortality(samc_obj, init_prob_data, time = 50)
short_dist <- distribution(samc_obj, origin = 3, time = 50)
long_disp <- dispersal(samc_obj, init_prob_data)
visit <- visitation(samc_obj, dest = 4)
surv <- survival(samc_obj)


# Use the map() function to turn vector results into RasterLayer objects.
short_mort_map <- map(samc_obj, short_mort)
short_dist_map <- map(samc_obj, short_dist)
long_disp_map <- map(samc_obj, long_disp)
visit_map <- map(samc_obj, visit)
surv_map <- map(samc_obj, surv)
}
