% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rdbw2d.R
\name{rdbw2d}
\alias{rdbw2d}
\title{Bandwidth Selection for 2D Local Polynomial RD Design}
\usage{
rdbw2d(
  Y,
  X,
  t,
  b,
  p = 1,
  deriv = c(0, 0),
  tangvec = NULL,
  kernel = c("tri", "triangular", "epa", "epanechnikov", "uni", "uniform", "gau",
    "gaussian"),
  kernel_type = c("prod", "rad"),
  bwselect = c("mserd", "imserd", "msetwo", "imsetwo"),
  method = c("dpi", "rot"),
  vce = c("hc1", "hc0", "hc2", "hc3"),
  bwcheck = 50 + p + 1,
  masspoints = c("check", "adjust", "off"),
  C = NULL,
  scaleregul = 1,
  scalebiascrct = 1,
  stdvars = TRUE
)
}
\arguments{
\item{Y}{Dependent variable; a numeric vector of length \eqn{N}, where \eqn{N} is the sample size.}

\item{X}{Bivariate running variable (a.k.a score variable); a numeric matrix or data frame of dimension \eqn{N \times 2}, with each row \eqn{\mathbf{X}_i = (X_{1i}, X_{2i})}.}

\item{t}{Treatment indicator; a logical or binary vector indicating treatment assignment (\eqn{t_i = 1} if treated, \eqn{t_i = 0} otherwise).}

\item{b}{Evaluation points; a matrix or data frame specifying boundary points \eqn{\mathbf{b}_j = (b_{1j}, b_{2j})}, of dimension \eqn{J \times 2}.}

\item{p}{Polynomial order of local polynomial estimator.}

\item{deriv}{The order of the derivatives of the regression functions to be estimated; a numeric vector of length 2 specifying the number of derivatives in each coordinate (e.g., \eqn{c(1,2)} corresponds to \eqn{\partial_1 \partial_2^2}).}

\item{tangvec}{Tangent vectors; a matrix or data frame of dimension \eqn{J \times 2} specifying directional derivatives. Overrides \code{deriv} if provided.}

\item{kernel}{Kernel function to use. Options are \code{"unif"}, \code{"uniform"} (uniform), \code{"triag"}, \code{"triangular"} (triangular, default), and \code{"epan"}, \code{"epanechnikov"} (Epanechnikov).}

\item{kernel_type}{Kernel structure. Either \code{"prod"} for product kernels (default) or \code{"rad"} for radial kernels.}

\item{bwselect}{Bandwidth selection strategy. Options:
\itemize{
\item \code{"mserd"}. One common MSE-optimal bandwidth selector for the boundary RD treatment effect estimator for each evaluation point (default).
\item \code{"imserd"}. IMSE-optimal bandwidth selector for the boundary RD treatment effect estimator based on all evaluation points.
\item \code{"msetwo"}. Two different MSE-optimal bandwidth selectors (control and treatment) for the boundary RD treatment effect estimator for each evaluation point.
\item \code{"imsetwo"}. Two IMSE-optimal bandwidth selectors (control and treatment) for the boundary RD treatment effect estimator based on all evaluation points.
\item \code{"user provided"}. User-provided bandwidths. If \code{h} is not \code{NULL}, then \code{bwselect} is overwritten to \code{"user provided"}.
}}

\item{method}{Bandwidth selection method for bias estimator based on local polynomials. Either \code{"dpi"} (default) for data-driven plug-in MSE optimal bandwidth selector or \code{"rot"} for rule-of-thumb bandwidth selector.}

\item{vce}{Variance-covariance estimation method. Options are:
\itemize{
\item \code{"hc0"}: heteroskedasticity-robust plug-in residual variance estimator without small-sample adjustment.
\item \code{"hc1"}: heteroskedasticity-robust plug-in residual variance estimator with HC1 small-sample adjustment (default).
\item \code{"hc2"}: heteroskedasticity-robust plug-in residual variance estimator with HC2 adjustment.
\item \code{"hc3"}: heteroskedasticity-robust plug-in residual variance estimator with HC3 adjustment.
}}

\item{bwcheck}{If a positive integer is provided, the preliminary bandwidth used in the calculations is enlarged so that at least \code{bwcheck} observations are used. If \code{masspoints == "adjust"}, ensure at least \code{bwcheck} unique observations are used. The program stops with “not enough observations” if sample size \eqn{N} < \code{bwcheck}. Default is \code{50 + p + 1}.}

\item{masspoints}{Handling of mass points in the running variable. Options are:
\itemize{
\item \code{"check"}: detects presence of mass points and reports the number of unique observations (default).
\item \code{"adjust"}: adjusts preliminary bandwidths to ensure a minimum number of unique observations within each side of the cutoff.
\item \code{"off"}: ignores presence of mass points.
}}

\item{C}{Cluster ID variable used for cluster-robust variance estimation. Default is \code{C = NULL}.}

\item{scaleregul}{Scaling factor for the regularization term in bandwidth selection. Default is 3.}

\item{scalebiascrct}{Scaling factor used for bias correction based on higher order expansions. Default is 1.}

\item{stdvars}{Logical. If TRUE, the running variables \eqn{X_{1i}} and \eqn{X_{2i}} are standardized before computing the bandwidths. Default is \code{TRUE}. Standardization only affects automatic bandwidth selection if bandwidths are not manually provided via \code{h}.}
}
\value{
A list of class \code{"rdbw2d"} containing:
\describe{
\item{\code{bws}}{Data frame of estimated bandwidths for each evaluation point:
\describe{
\item{\code{b1}}{First coordinate of the evaluation point.}
\item{\code{b2}}{Second coordinate of the evaluation point.}
\item{\code{h01}}{Estimated bandwidth for \eqn{X_{1i}} in the control group (\eqn{\mathcal{A}_0}).}
\item{\code{h02}}{Estimated bandwidth for \eqn{X_{2i}} in the control group (\eqn{\mathcal{A}_0}).}
\item{\code{h11}}{Estimated bandwidth for \eqn{X_{1i}} in the treatment group (\eqn{\mathcal{A}_1}).}
\item{\code{h12}}{Estimated bandwidth for \eqn{X_{2i}} in the treatment group (\eqn{\mathcal{A}_1}).}
}
}
\item{\code{mseconsts}}{Data frame of intermediate quantities used in bandwidth calculation:
\describe{
\item{\code{Nh0}}{Effective sample size for the control group \eqn{\mathcal{A}_0}.}
\item{\code{Nh1}}{Effective sample size for the treatment group \eqn{\mathcal{A}_1}.}
\item{\code{bias.0}}{Bias constant estimate for the control group.}
\item{\code{bias.1}}{Bias constant estimate for the treatment group.}
\item{\code{var.0}}{Variance constant estimate for the control group.}
\item{\code{var.1}}{Variance constant estimate for the treatment group.}
\item{\code{reg.bias.0}}{Bias correction adjustment for the control group.}
\item{\code{reg.bias.1}}{Bias correction adjustment for the treatment group.}
\item{\code{reg.var.0}}{Variance of the bias estimate for the control group.}
\item{\code{reg.var.1}}{Variance of the bias estimate for the treatment group.}
}
}
\item{\code{opt}}{List containing:
\describe{
\item{\code{p}}{Polynomial order used for estimation.}
\item{\code{kernel}}{Kernel function used.}
\item{\code{kernel_type}}{Type of kernel (product or radial).}
\item{\code{stdvars}}{Logical indicating if standardization was applied.}
\item{\code{bwselect}}{Bandwidth selection strategy used.}
\item{\code{method}}{Bandwidth estimation method.}
\item{\code{vce}}{Variance estimation method.}
\item{\code{scaleregul}}{Scaling factor for regularization.}
\item{\code{scalebiascrct}}{Scaling factor for bias correction.}
\item{\code{N}}{Total sample size \eqn{N}.}
}
}
}
}
\description{
\code{rdbw2d} implements bandwidth selector for bivariate local polynomial boundary regression discontinuity (RD) point estimators with robust bias-corrected pointwise confidence intervals and
uniform confidence bands, developed in Cattaneo, Titiunik and Yu (2025a) with a companion software article Cattaneo, Titiunik and Yu (2025b). For robust bias-correction, see Calonico, Cattaneo, Titiunik (2014).

Companion commands are: \code{rd2d} for point estimation and inference procedures.

For other packages of RD designs, visit
\url{https://rdpackages.github.io/}
}
\examples{
# Simulated example
set.seed(123)
n <- 5000
X1 <- rnorm(n)
X2 <- rnorm(n)
t <- as.numeric(X1 > 0)
Y <- 3 + 2 * X1 + 1.5 * X2 + t + rnorm(n)
X <- cbind(X1, X2)
b <- matrix(c(0, 0, 0, 1), ncol = 2)

# MSE optimal bandwidth for rd2d
bws <- rdbw2d(Y, X, t, b)

# View the bandwidth selection results
print(bws)
summary(bws)
}
\references{
\itemize{
\item{\href{https://rdpackages.github.io/references/Cattaneo-Titiunik-Yu_2025_BoundaryRD.pdf}{Cattaneo, M. D., Titiunik, R., Yu, R. R. (2025a).}
Estimation and Inference in Boundary Discontinuity Designs}
\item{\href{https://rdpackages.github.io/references/Cattaneo-Titiunik-Yu_2025_rd2d.pdf}{Cattaneo, M. D., Titiunik, R., Yu, R. R. (2025b).}
rd2d: Causal Inference in Boundary Discontinuity Designs}
\item{\href{https://rdpackages.github.io/references/Calonico-Cattaneo-Titiunik_2014_ECMA.pdf}{Calonico, S., Cattaneo, M. D., Titiunik, R. (2014)}
Robust Nonparametric Confidence Intervals for Regression-Discontinuity Designs}
}
}
\seealso{
\code{\link{rd2d}}, \code{\link{print.rdbw2d}}, \code{\link{summary.rdbw2d}}
}
\author{
Matias D. Cattaneo, Princeton University. \email{cattaneo@princeton.edu} \cr
Rocío Titiunik, Princeton University. \email{titiunik@princeton.edu} \cr
Ruiqi Rae Yu, Princeton University. \email{rae.yu@princeton.edu}
}
