% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rd2d_dist.R
\name{rd2d.dist}
\alias{rd2d.dist}
\title{Local Polynomial RD Estimation on Distance-Based Running Variables}
\usage{
rd2d.dist(
  Y,
  D,
  h = NULL,
  b = NULL,
  p = 1,
  q = 2,
  kink = c("off", "on"),
  kernel = c("tri", "triangular", "epa", "epanechnikov", "uni", "uniform", "gau",
    "gaussian"),
  level = 95,
  cbands = TRUE,
  side = c("two", "left", "right"),
  repp = 1000,
  bwselect = c("mserd", "imserd", "msetwo", "imsetwo", "user provided"),
  vce = c("hc1", "hc0", "hc2", "hc3"),
  rbc = c("on", "off"),
  bwcheck = 50 + p + 1,
  masspoints = c("check", "adjust", "off"),
  C = NULL,
  scaleregul = 1,
  cqt = 0.5
)
}
\arguments{
\item{Y}{Dependent variable; a numeric vector of length \eqn{N}, where \eqn{N} is the sample size.}

\item{D}{Distance-based scores \eqn{\mathbf{D}_i=(\mathbf{D}_{i}(\mathbf{b}_1),\cdots,\mathbf{D}_{i}(\mathbf{b}_J))}; dimension is \eqn{N \times J} where \eqn{N} = sample size and \eqn{J} = number of cutoffs; non-negative values means data point in treatment group and negative values means data point in control group.}

\item{h}{Bandwidth(s); if \eqn{c=h} then same bandwidth is used for both groups; if a matrix of size \eqn{J \times 2} is provided, each row contains \eqn{(h_{\text{control}}, h_{\text{tr}})} for the evaluation point; if not specified, bandwidths are selected via \code{rdbw2d.dist()}.}

\item{b}{Optional evaluation points; a matrix or data frame specifying boundary points \eqn{\mathbf{b}_j = (b_{1j}, b_{2j})}, dimension \eqn{J \times 2}.}

\item{p}{Polynomial order for point estimation. Default is \code{p = 1}.}

\item{q}{Polynomial order for bias-corrected estimation. Must satisfy \eqn{q \geq p}. Default is \code{q = p + 1}.}

\item{kink}{Logical; whether to apply kink adjustment. Options: \code{"on"} (default) or \code{"off"}.}

\item{kernel}{Kernel function to use. Options are \code{"unif"}, \code{"uniform"} (uniform), \code{"triag"}, \code{"triangular"} (triangular, default), and \code{"epan"}, \code{"epanechnikov"} (Epanechnikov).}

\item{level}{Nominal confidence level for intervals/bands, between 0 and 100 (default is 95).}

\item{cbands}{Logical. If \code{TRUE}, also compute uniform confidence bands (default is \code{FALSE}).}

\item{side}{Type of confidence interval. Options: \code{"two"} (two-sided, default), \code{"left"} (left tail), or \code{"right"} (right tail).}

\item{repp}{Number of bootstrap repetitions used for critical value simulation. Default is \code{1000}.}

\item{bwselect}{Bandwidth selection strategy. Options:
\itemize{
\item \code{"mserd"}. One common MSE-optimal bandwidth selector for the boundary RD treatment effect estimator for each evaluation point (default).
\item \code{"imserd"}. IMSE-optimal bandwidth selector for the boundary RD treatment effect estimator based on all evaluation points.
\item \code{"msetwo"}. Two different MSE-optimal bandwidth selectors (control and treatment) for the boundary RD treatment effect estimator for each evaluation point.
\item \code{"imsetwo"}. Two IMSE-optimal bandwidth selectors (control and treatment) for the boundary RD treatment effect estimator based on all evaluation points.
\item \code{"user provided"}. User-provided bandwidths. If \code{h} is not \code{NULL}, then \code{bwselect} is overwritten to \code{"user provided"}.
}}

\item{vce}{Variance-covariance estimator for standard errors.
Options:
\describe{
\item{\code{"hc0"}}{Heteroskedasticity-robust variance estimator without small sample adjustment (White robust).}
\item{\code{"hc1"}}{Heteroskedasticity-robust variance estimator with degrees-of-freedom correction (default).}
\item{\code{"hc2"}}{Heteroskedasticity-robust variance estimator using leverage adjustments.}
\item{\code{"hc3"}}{More conservative heteroskedasticity-robust variance estimator (similar to jackknife correction).}
}}

\item{rbc}{Logical. Whether to apply robust bias correction. Options: \code{"on"} (default) or \code{"off"}. When \code{kink = off}, turn on \code{rbc} means setting \code{q} to \code{p + 1}.
When \code{kink = on}, turn on \code{rbc} means shrinking the bandwidth selector to be proportional to \eqn{N^{-1/3}}.}

\item{bwcheck}{If a positive integer is provided, the preliminary bandwidth used in the calculations is enlarged so that at least \code{bwcheck} observations are used. The program stops with “not enough observations” if sample size \eqn{N} < \code{bwcheck}. Default is \code{50 + p + 1}.}

\item{masspoints}{Strategy for handling mass points in the running variable.
Options:
\describe{
\item{\code{"check"}}{Check for repeated values and adjust inference if needed (default).}
\item{\code{"adjust"}}{Adjust bandwidths to guarantee a sufficient number of unique support points.}
\item{\code{"off"}}{Ignore mass points completely.}
}}

\item{C}{Cluster ID variable used for cluster-robust variance estimation. Default is \code{C = NULL}.}

\item{scaleregul}{Scaling factor for the regularization term in bandwidth selection. Default is \code{1}.}

\item{cqt}{Constant controlling subsample fraction for initial bias estimation. Default is \code{0.5}.}
}
\value{
An object of class \code{"rd2d.dist"}, a list containing:
\describe{
\item{\code{results}}{A data frame with point estimates, variances, p-values, confidence intervals, confidence bands, and bandwidths at each evaluation point.
\describe{
\item{\code{b1}}{First coordinate of the evaluation point.}
\item{\code{b2}}{Second coordinate of the evaluation point.}
\item{\code{Est.p}}{Point estimate \eqn{\widehat{\tau}_{\text{dist},p}(\mathbf{b})} with polynomial order \eqn{p}.}
\item{\code{Var.p}}{Variance of \eqn{\widehat{\tau}_{\text{dist},p}(\mathbf{b})}.}
\item{\code{Est.q}}{Bias-corrected estimate \eqn{\widehat{\tau}_{\text{dist},q}(\mathbf{b})} with polynomial order \eqn{q}.}
\item{\code{Var.q}}{Variance of \eqn{\widehat{\tau}_{\text{dist},q}(\mathbf{b})}.}
\item{\code{pvalue}}{Two-sided p-value based on \eqn{T_{\text{dist},q}(\mathbf{b})}.}
\item{\code{CI.lower}}{Lower bound of confidence interval.}
\item{\code{CI.upper}}{Upper bound of confidence interval.}
\item{\code{CB.lower}}{Lower bound of uniform confidence band (if \code{cbands=TRUE}).}
\item{\code{CB.upper}}{Upper bound of uniform confidence band (if \code{cbands=TRUE}).}
\item{\code{h0}}{Bandwidth used for control group (\eqn{D_i(\mathbf{b}) < 0}).}
\item{\code{h1}}{Bandwidth used for treatment group (\eqn{D_i(\mathbf{b}) \geq 0}).}
\item{\code{Nh0}}{Effective sample size for control group.}
\item{\code{Nh1}}{Effective sample size for treatment group.}
}
}
\item{\code{results.A0}}{Same structure as \code{results} but for control group outcomes.}
\item{\code{results.A1}}{Same structure as \code{results} but for treatment group outcomes.}
\item{\code{tau.hat}}{Vector of point estimates \eqn{\widehat{\tau}_p(\mathbf{b})}.}
\item{\code{se.hat}}{Standard errors corresponding to \eqn{\widehat{\tau}_p(\mathbf{b})}.}
\item{\code{cb}}{Confidence intervals and uniform bands.}
\item{\code{cov.q}}{Covariance matrix for bias-corrected estimates \eqn{\widehat{\tau}_{\text{dist},q}(\mathbf{b})} for all point evaluations \eqn{\mathbf{b}}.}
\item{\code{opt}}{List of options used in the function call.}
}
}
\description{
\code{rd2d.dist} implements distance-based local polynomial boundary regression discontinuity (RD) point estimators with robust bias-corrected pointwise confidence intervals and
uniform confidence bands, developed in Cattaneo, Titiunik and Yu (2025a) with a companion software article Cattaneo, Titiunik and Yu (2025b). For robust bias-correction, see Calonico, Cattaneo, Titiunik (2014).

Companion commands are: \code{rdbw2d.dist} for data-driven bandwidth selection.

For other packages of RD designs, visit
\url{https://rdpackages.github.io/}
}
\details{
MSE bandwidth selection for geometrical RD design
}
\examples{
set.seed(123)
n <- 5000

# Generate running variables x1 and x2
x1 <- rnorm(n)
x2 <- rnorm(n)

# Define treatment assignment: treated if x1 >= 0
d <- as.numeric(x1 >= 0)

# Generate outcome variable y with some treatment effect
y <- 3 + 2 * x1 + 1.5 * x2 + 1.5 * d + rnorm(n, sd = 0.5)

# Define evaluation points (e.g., at the origin and another point)
eval <- data.frame(x.1 = c(0, 0), x.2 = c(0, 1))

# Compute Euclidean distances to evaluation points
dist.a <- sqrt((x1 - eval$x.1[1])^2 + (x2 - eval$x.2[1])^2)
dist.b <- sqrt((x1 - eval$x.1[2])^2 + (x2 - eval$x.2[2])^2)

# Combine distances into a matrix
D <- as.data.frame(cbind(dist.a, dist.b))

# Assign positive distances for treatment group, negative for control
d_expanded <- matrix(rep(2 * d - 1, times = ncol(D)), nrow = nrow(D), ncol = ncol(D))
D <- D * d_expanded

# Run the rd2d.dist function
result <- rd2d.dist(y, D, b = eval)

# View the estimation results
print(result)
summary(result)
}
\references{
\itemize{
\item{\href{https://rdpackages.github.io/references/Cattaneo-Titiunik-Yu_2025_BoundaryRD.pdf}{Cattaneo, M. D., Titiunik, R., Yu, R. R. (2025a).}
Estimation and Inference in Boundary Discontinuity Designs}
\item{\href{https://rdpackages.github.io/references/Cattaneo-Titiunik-Yu_2025_rd2d.pdf}{Cattaneo, M. D., Titiunik, R., Yu, R. R. (2025b).}
rd2d: Causal Inference in Boundary Discontinuity Designs}
\item{\href{https://rdpackages.github.io/references/Calonico-Cattaneo-Titiunik_2014_ECMA.pdf}{Calonico, S., Cattaneo, M. D., Titiunik, R. (2014)}
Robust Nonparametric Confidence Intervals for Regression-Discontinuity Designs}
}
}
\seealso{
\code{\link{rdbw2d.dist}}, \code{\link{rd2d}}, \code{\link{print.rd2d.dist}}, \code{\link{summary.rd2d.dist}}
}
\author{
Matias D. Cattaneo, Princeton University. \email{cattaneo@princeton.edu} \cr
Rocío Titiunik, Princeton University. \email{titiunik@princeton.edu} \cr
Ruiqi Rae Yu, Princeton University. \email{rae.yu@princeton.edu}
}
