% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/loops.R
\name{pedigreeLoops}
\alias{pedigreeLoops}
\alias{breakLoops}
\alias{tieLoops}
\alias{findLoopBreakers}
\alias{findLoopBreakers2}
\title{Pedigree loops}
\usage{
pedigreeLoops(x)

breakLoops(x, loop_breakers = NULL, verbose = TRUE)

tieLoops(x)

findLoopBreakers(x)

findLoopBreakers2(x)
}
\arguments{
\item{x}{a \code{\link{linkdat}} object.}

\item{loop_breakers}{either NULL (resulting in automatic selection of loop
breakers) or a numeric containing IDs of individuals to be used as loop
breakers.}

\item{verbose}{a logical: Verbose output or not?}
}
\value{
For \code{breakLoops}, a \code{linkdat} object in which the
indicated loop breakers are duplicated. The returned object will also have a
non-null \code{loop_breakers} entry, namely a matrix with the IDs of the
original loop breakers in the first column and the duplicates in the second.

For \code{tieLoops}, a \code{linkdat} object in which any duplicated
individuals (as given in the \code{x$loop_breakers} entry) are merged. For
any linkdat object \code{x}, the call \code{tieLoops(breakLoops(x))} should
return \code{x}.

For \code{pedigreeLoops}, a list containing all inbreeding loops (not
marriage loops) found in the pedigree. Each loop is represented as a list
with elements 'top', a 'bottom' individual, 'pathA' (individuals forming a
path from top to bottom) and 'pathB' (creating a different path from top to
bottom, with no individuals in common with pathA). Note that the number of
loops reported here counts all closed paths in the pedigree and will in
general be larger than the genus of the underlying graph.

For \code{findLoopBreakers} and \code{findLoopBreakers2}, a numeric vector
of individual ID's.
}
\description{
Functions for identifying, breaking and restoring loops in pedigrees.
}
\details{
Most of paramlink's handling of pedigree loops is done under the hood -
using the functions described here - without need for explicit action from
end users. When a linkdat object \code{x} is created, an internal routine
detects if the pedigree contains loops, in which case \code{x$hasLoops} is
set to TRUE. In analyses of \code{x} where loops must be broken (e.g. lod
score computation or marker simulation), this is done automatically by
calling \code{breakLoops}.

In some cases with complex inbreeding, it can be instructive to plot the
pedigree after breaking the loops. Duplicated individuals are plotted with
appropriate labels (see examples).

The function \code{findLoopBreakers} identifies a set of individuals
breaking all inbreeding loops, but not marriage loops. These require more
machinery for efficient detection, and paramlink does this is a seperate
function, \code{findLoopBreakers2}, utilizing methods from the \code{igraph}
package. Since this is rarely needed for most users, \code{igraph} is not
imported when loading paramlink, only when \code{findLoopBreakers2} is
called.

In practice, \code{breakLoops} first calls \code{findLoopBreakers} and
breaks at the returned individuals. If the resulting linkdat object still
has loops, \code{findLoopBreakers2} is called to break any marriage loops.
}
\examples{

x = cousinsPed(1, child=TRUE)

# Make the child affected, and homozygous for rare allele.
x = swapAff(x, 9)
x = setMarkers(x, marker(x, 9, c(2,2), alleles=1:2, afreq=c(0.99, 0.01)))

# Compute the LOD score under a recessive model. Loops are automatically broken in lod().
x = setModel(x, 2)
LOD1 = lod(x, theta=0.1)
stopifnot(round(LOD1, 2) == 0.88)

# Or we can break the loop manually before computing the LOD:
loopfree = breakLoops(x, loop_breaker=8)
plot(loopfree)
LOD2 = lod(loopfree, theta=0.1)
stopifnot(all.equal(x, tieLoops(loopfree)))
stopifnot(all.equal(LOD1, LOD2))

# Pedigree with marriage loop: Double first cousins
if(require('igraph')) {
    y = doubleCousins(1, 1, child=TRUE)
    findLoopBreakers(y) # --> 9
    findLoopBreakers2(y) # --> 9 and 4
    breakLoops(y) # uses both 9 and 4
}

}
\author{
Magnus Dehli Vigeland
}
