% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dpoisLocal_normal.R
\name{dpoisLocal_normal}
\alias{dpoisLocal_normal}
\alias{rpoisLocal_normal}
\title{Local evaluation of a Poisson SCR detection process}
\usage{
dpoisLocal_normal(
  x,
  detNums = -999,
  detIndices,
  lambda = -999,
  lambdaTraps,
  sigma,
  s,
  trapCoords,
  localTrapsIndices,
  localTrapsNum,
  resizeFactor = 1,
  habitatGrid,
  indicator,
  lengthYCombined = 0,
  log = 0
)

rpoisLocal_normal(
  n = 1,
  detNums = -999,
  detIndices,
  lambda = -999,
  lambdaTraps,
  sigma,
  s,
  trapCoords,
  localTrapsIndices,
  localTrapsNum,
  resizeFactor = 1,
  habitatGrid,
  indicator,
  lengthYCombined = 0
)
}
\arguments{
\item{x}{Vector of individual detection frequencies. This argument can be provided in two formats: (i) with the \emph{y} object as returned by the \code{\link{getSparseY}} function; (ii) with the \emph{yCombined} object as returned by \code{\link{getSparseY}}. 
Note that when the random generation functionality is used (\code{rpoisLocal_normal}), only the \emph{yCombined} format can be used. 
The \emph{yCombined} object combines \emph{detNums}, \emph{x}, and \emph{detIndices} (in that order).  When such consolidated representation of the detection data \emph{x} is used, \emph{detIndices} and \emph{detNums} arguments shouldn’t be specified.}

\item{detNums}{Number of detections recorded in \emph{x}, as returned by the \emph{detNums} object from the \code{\link{getSparseY}} function. This argument should not be specified when the \emph{yCombined} object (returned by \code{\link{getSparseY}}) is provided as \emph{x}, and when detection data are simulated.}

\item{detIndices}{Vector of indices of traps where the detections in x were recorded, as returned by the \emph{detIndices} object from the \code{\link{getSparseY}} function. This argument should not be specified when \emph{x} is provided as the \emph{yCombined} object (returned by \code{\link{getSparseY}}) and when detection data are simulated.}

\item{lambda}{Baseline detection rate used in the half-normal detection function.}

\item{lambdaTraps}{Vector of baseline detection rate for each trap used in the half-normal detection function. When \emph{lambdaTraps} is used, \emph{lambda} should not be provided.}

\item{sigma}{Scale parameter of the half-normal detection function.}

\item{s}{Individual activity center x- and y-coordinates.}

\item{trapCoords}{Matrix of x- and y-coordinates of all traps.}

\item{localTrapsIndices}{Matrix of indices of local traps around each habitat grid cell, as returned by the \code{\link{getLocalObjects}} function.}

\item{localTrapsNum}{Vector of numbers of local traps around all habitat grid cells, as returned by the \code{\link{getLocalObjects}} function.}

\item{resizeFactor}{Aggregation factor used in the \code{\link{getLocalObjects}} function to reduce the number of habitat grid cells to retrieve local traps for.}

\item{habitatGrid}{Matrix of habitat grid cells indices, as returned by the \code{\link{getLocalObjects}} function.}

\item{indicator}{Logical argument specifying whether the individual is available for detection.}

\item{lengthYCombined}{The length of the  x argument when the (\emph{yCombined}) format of the detection data is provided (as returned by the \emph{lengthYCombined} object from \code{\link{getSparseY}}).}

\item{log}{Logical argument, specifying whether to return the log-probability of the distribution.}

\item{n}{Integer specifying the number of realizations to generate.  Only n = 1 is supported.}
}
\value{
The log-likelihood value associated with the vector of detections, given the location of the activity center (s),
 and the half-normal detection function : \eqn{p = lambda * exp(-d^2 / \sigma^2)}.
}
\description{
The \code{dpoisLocal_normal} distribution is a NIMBLE custom distribution which can be used to model and simulate
Poisson observations (\emph{x}) of a single individual over a set of detectors defined by their 
coordinates (\emph{trapCoords}). The distribution assumes that an individual’s detection probability at any detector
follows a half-normal function of the distance between the individual's activity center (\emph{s}) and the detector location.
}
\details{
The \code{dpoisLocal_normal} distribution incorporates three features to increase computation efficiency (see Turek et al., 2021 <doi.org/10.1002/ecs2.3385>  for more details):
\enumerate{
\item A local evaluation of the detection probability calculation (see Milleret et al., 2019 <doi:10.1002/ece3.4751> for more details)
\item A sparse matrix representation (\emph{x}, \emph{detIndices} and \emph{detNums}) of the observation data to reduce the size of objects to be processed.
\item An indicator (\emph{indicator}) to shortcut calculations for individuals unavailable for detection.
}

The \code{dpoisLocal_normal} distribution requires x- and y- detector coordinates (\emph{trapCoords}) to be scaled to the habitat grid (\emph{habitatGrid}) using the (\code{\link{scaleCoordsToHabitatGrid}} function.)

When the aim is to simulate detection data: 
\enumerate{
\item \emph{x} should be provided using the \emph{yCombined} object as returned by \code{\link{getSparseY}}, 
\item arguments \emph{detIndices} and \emph{detNums} should not be provided, 
\item argument \emph{lengthYCombined} should be provided using the \emph{lengthYCombined} object as returned by  \code{\link{getSparseY}}.
}
}
\examples{
# I. DATA SET UP 
coordsHabitatGridCenter <- matrix(c(0.5, 3.5,
                                    1.5, 3.5,
                                    2.5, 3.5,
                                    3.5, 3.5,
                                    0.5, 2.5,
                                    1.5, 2.5,
                                    2.5, 2.5,
                                    3.5, 2.5,
                                    0.5, 1.5,
                                    1.5, 1.5,
                                    2.5, 1.5,
                                    3.5, 1.5,
                                    0.5, 0.5,
                                    1.5, 0.5,
                                    2.5, 0.5,
                                    3.5, 0.5), ncol=2,byrow = TRUE)
colnames(coordsHabitatGridCenter) <- c("x","y")
# CREATE OBSERVATION WINDOWS
trapCoords <- matrix(c(1.5, 1.5, 2.5, 1.5, 1.5, 2.5, 2.5, 2.5), nrow = 4, byrow = TRUE)
colnames(trapCoords) <- c("x","y")
# PLOT CHECK
plot(coordsHabitatGridCenter[,"y"]~coordsHabitatGridCenter[,"x"],pch=16) 
points(trapCoords[,"y"]~trapCoords[,"x"],col="red",pch=16) 

# PARAMETERS
lambda <- 0.2
sigma <- 2
indicator <- 1 
# WE CONSIDER 2 INDIVIDUALS
y <- matrix(c(0, 1, 1, 0,
              0, 1, 0, 1),ncol=4,nrow=2)
s <- matrix(c(0.5, 1,
              1.6, 2.3),ncol=2,nrow=2)

# RESCALE COORDINATES 
ScaledtrapCoords <- scaleCoordsToHabitatGrid(coordsData =  trapCoords,
                                             coordsHabitatGridCenter = coordsHabitatGridCenter)
ScaledtrapCoords<- ScaledtrapCoords$coordsDataScaled
habitatMask <- matrix(1, nrow = 4, ncol=4, byrow = TRUE)


# CREATE LOCAL OBJECTS 
TrapLocal <- getLocalObjects(habitatMask = habitatMask,
                                   coords = ScaledtrapCoords,
                                   dmax=2.5,
                                   resizeFactor = 1,
                                   plot.check = TRUE
)

# GET SPARSE MATRIX 
SparseY <- getSparseY(y)

# II. USING THE DENSITY FUNCTION 
 # WE TAKE THE FIRST INDIVIDUAL
i=1
  # OPTION 1: USING THE RANDOM GENERATION FUNCTIONNALITY 
dpoisLocal_normal(x=SparseY$y[i,,1],
                   detNums=SparseY$detNums[i],
                   detIndices=SparseY$detIndices[i,,1],
                   lambda = lambda,
                   sigma= sigma, 
                   s=s[i,1:2],
                   trapCoords=ScaledtrapCoords,
                   localTrapsIndices=TrapLocal$localIndices,
                   localTrapsNum=TrapLocal$numLocalIndices,
                   resizeFactor=TrapLocal$resizeFactor,
                   habitatGrid=TrapLocal$habitatGrid,
                   indicator=indicator)
                                                                
  # OPTION 2: USING RANDOM GENERATION FUNCTIONNALITY 
  # WE DO NOT PROVIDE THE detNums AND detIndices ARGUMENTS
dpoisLocal_normal(x=SparseY$yCombined[i,,1],
                   lambda = lambda,
                   sigma= sigma, 
                   s=s[i,1:2],
                   trapCoords=ScaledtrapCoords,
                   localTrapsIndices=TrapLocal$localIndices,
                   localTrapsNum=TrapLocal$numLocalIndices,
                   resizeFactor=TrapLocal$resizeFactor,
                   habitatGrid=TrapLocal$habitatGrid,
                   indicator=indicator,
                   lengthYCombined = SparseY$lengthYCombined)

# III. USING THE RANDOM GENERATION FUNCTION 
rpoisLocal_normal(n=1,
                   lambda = lambda,
                   sigma= sigma, 
                   s=s[i,1:2],
                   trapCoords=ScaledtrapCoords,
                   localTrapsIndices=TrapLocal$localIndices,
                   localTrapsNum=TrapLocal$numLocalIndices,
                   resizeFactor=TrapLocal$resizeFactor,
                   habitatGrid=TrapLocal$habitatGrid,
                   indicator=indicator,
                   lengthYCombined = SparseY$lengthYCombined)

}
\author{
Cyril Milleret, Soumen Dey
}
