% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IdealGas.R
\name{thermodynamics}
\alias{thermodynamics}
\alias{IdealGas}
\alias{Adiabat}
\alias{VirtualTemperature}
\alias{MixingRatio}
\alias{ClausiusClapeyron}
\alias{DewPoint}
\title{Thermodynamics}
\usage{
IdealGas(p, t, rho, R = 287.058)

Adiabat(p, t, theta, p0 = 1e+05, kappa = 2/7)

VirtualTemperature(p, t, e, tv, epsilon = 0.622)

MixingRatio(p, es, w, epsilon = 0.622)

ClausiusClapeyron(t, es)

DewPoint(p, ws, td, epsilon = 0.622)
}
\arguments{
\item{p}{pressure}

\item{t}{temperature}

\item{rho}{density}

\item{R}{gas constant for air}

\item{theta}{potential temperature}

\item{p0}{reference pressure}

\item{kappa}{ratio of dry air constant and specific heat capacity at constant pressure}

\item{e}{vapour partial pressure}

\item{tv}{virtual temperature}

\item{epsilon}{ratio of dry air constant and vapour constant}

\item{es}{saturation vapour partial pressure}

\item{w}{mixing ratio}

\item{ws}{saturation mixing ratio}

\item{td}{dewpoint}
}
\value{
Each function returns the value of the missing state variable.
}
\description{
Functions related to common atmospheric thermodynamic relationships.
}
\details{
\code{IdealGas} computes pressure, temperature or density of air according to the
ideal gas law \eqn{P=\rho R T}.

\code{Adiabat} computes pressure, temperature or potential temperature according to
the adiabatic relationship \eqn{\theta = T (P0/P)^\kappa}.

\code{VirtualTemperature} computes pressure, temperature, vapour partial pressure or
virtual temperature according to the virtual temperature definition
\eqn{T(1 - e/P(1 - \epsilon))^{-1}}.

\code{MixingRatio} computes pressure, vapour partial temperature, or mixing ratio
according to \eqn{w = \epsilon e/(P - e)}.

\code{ClausiusClapeyron} computes saturation pressure or temperature according
to the August-Roche-Magnus formula \eqn{es = a exp{bT/(T + c)}} with temperature
in Kelvin and saturation pressure in Pa.

\code{DewPoint} computes pressure, saturation mixing ration or dew point
from the relationship \eqn{ws =  \epsilon es(Td)/(p - es(Td))}. Note that the
computation of dew point is approximated.

Is important to take note of the units in which each variable is provided.
With the default values, pressure should be passed in Pascals, temperature and
potential temperature in Kelvins, and density in \eqn{kg/m^3}.
\code{ClausiusClayperon} and \code{DewPoint} require and return values in those units.

The defaults value of the \code{R} and \code{kappa} parameters are correct for dry air,
for the case of moist air, use the virtual temperature instead of the actual
temperature.
}
\examples{
IdealGas(1013*100, 20 + 273.15)
IdealGas(1013*100, rho = 1.15) - 273.15

(theta <- Adiabat(70000, 20 + 273.15))
Adiabat(70000, theta = theta) - 273.15

# Relative humidity from T and Td
t <- 25 + 273.15
td <- 20 + 273.15
p <- 1000000
(rh <- ClausiusClapeyron(td)/ClausiusClapeyron(t))

# Mixing ratio
ws <- MixingRatio(p, ClausiusClapeyron(t))
w <- ws*rh
DewPoint(p, w) - 273.15    # Recover Td

}
\references{
http://www.atmo.arizona.edu/students/courselinks/fall11/atmo551a/ATMO_451a_551a_files/WaterVapor.pdf
}
\seealso{
Other meteorology functions: \code{\link{Derivate}},
  \code{\link{EOF}}, \code{\link{GeostrophicWind}},
  \code{\link{WaveFlux}}, \code{\link{waves}}
}
\concept{meteorology functions}
