% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.R
\name{find_extremum}
\alias{find_extremum}
\title{Find Extremum of Fitted Lagrangian Multiplier Smoothing Spline}
\usage{
find_extremum(
  object,
  vars = NULL,
  quick_heuristic = TRUE,
  initial = NULL,
  B_predict = NULL,
  minimize = FALSE,
  stochastic = FALSE,
  stochastic_draw = function(mu, sigma, ...) {
     N <- length(mu)
     rnorm(N, mu,
    sigma)
 },
  sigmasq_predict = object$sigmasq_tilde,
  custom_objective_function = NULL,
  custom_objective_derivative = NULL,
  ...
)
}
\arguments{
\item{object}{A fitted lgspline model object containing partition information and fitted values}

\item{vars}{Vector; A vector of numeric indices (or character variable names) of predictors to optimize for. If NULL (by default), all predictors will be optimized.}

\item{quick_heuristic}{Logical; whether to search only the top-performing partition. When TRUE (default),
optimizes within the best partition. When FALSE, initiates searches from all partition local maxima.}

\item{initial}{Numeric vector; Optional initial values for optimization. Useful for fixing binary
predictors or providing starting points. Default NULL}

\item{B_predict}{Matrix; Optional custom coefficient list for prediction. Useful for posterior
draws in Bayesian optimization. Default NULL}

\item{minimize}{Logical; whether to find minimum instead of maximum. Default FALSE}

\item{stochastic}{Logical; whether to add noise for stochastic optimization. Enables better
exploration of the function space. Default FALSE}

\item{stochastic_draw}{Function; Generates random noise/modifies predictions for stochastic optimization, analogous to
posterior_predictive_draw. Takes three arguments:
\itemize{
  \item mu: Vector of predicted values
  \item sigma: Vector of standard deviations (square-root of sigmasq_tilde)
  \item ...: Additional arguments to pass through
}
Default \code{rnorm(length(mu), mu, sigma)}}

\item{sigmasq_predict}{Numeric; Variance parameter for stochastic optimization. Controls
the magnitude of random perturbations. Defaults to object$sigmasq_tilde}

\item{custom_objective_function}{Function; Optional custom objective function for optimization.
Takes arguments:
\itemize{
  \item mu: Vector of predicted response values
  \item sigma: Vector of standard deviations
  \item y_best: Numeric; Best observed response value
  \item ...: Additional arguments passed through
}
Default NULL}

\item{custom_objective_derivative}{Function; Optional gradient function for custom optimization
objective. Takes arguments:
\itemize{
  \item mu: Vector of predicted response values
  \item sigma: Vector of standard deviations
  \item y_best: Numeric; Best observed response value
  \item d_mu: Gradient of fitted function (for chain-rule computations)
  \item ...: Additional arguments passed through
}
Default NULL}

\item{...}{Additional arguments passed to internal optimization routines.}
}
\value{
A list containing the following components:
\describe{
  \item{t}{Numeric vector of input values at the extremum.}
  \item{y}{Numeric value of the objective function at the extremum.}
}
}
\description{
Finds global extrema of a fitted lgspline model using deterministic or stochastic
optimization strategies. Supports custom objective functions for advanced applications like
Bayesian optimization acquisition functions.
}
\details{
This method finds extrema (maxima or minima) of the fitted function or composite functions
of the fit. The optimization process can be customized through several approaches:
\itemize{
  \item Partition-based search: Either focuses on the top-performing partition (quick_heuristic = TRUE)
        or searches across all partition local maxima
  \item Stochastic optimization: Adds random noise during optimization for better exploration
  \item Custom objectives: Supports user-defined objective functions and gradients for
        specialized optimization tasks like Bayesian optimization
}
}
\examples{

## Basic usage with simulated data
set.seed(1234)
t <- runif(1000, -10, 10)
y <- 2*sin(t) + -0.06*t^2 + rnorm(length(t))
model_fit <- lgspline(t, y)
plot(model_fit)

## Find global maximum and minimum
max_point <- find_extremum(model_fit)
min_point <- find_extremum(model_fit, minimize = TRUE)
abline(v = max_point$t, col = 'blue')  # Add maximum point
abline(v = min_point$t, col = 'red')   # Add minimum point

## Advanced usage: custom objective functions
# expected improvement acquisition function
ei_custom_objective_function = function(mu, sigma, y_best, ...) {
  d <- y_best - mu
  d * pnorm(d/sigma) + sigma * dnorm(d/sigma)
}
# derivative of ei
ei_custom_objective_derivative = function(mu, sigma, y_best, d_mu, ...) {
  d <- y_best - mu
  z <- d/sigma
  d_z <- -d_mu/sigma
  pnorm(z)*d_mu - d*dnorm(z)*d_z + sigma*z*dnorm(z)*d_z
}

## Single iteration of Bayesian optimization
post_draw <- generate_posterior(model_fit)
acq <- find_extremum(model_fit,
                     stochastic = TRUE,  # Enable stochastic exploration
                     B_predict = post_draw$post_draw_coefficients,
                     sigmasq_predict = post_draw$post_draw_sigmasq,
                     custom_objective_function = ei_custom_objective_function,
                     custom_objective_derivative = ei_custom_objective_derivative)
abline(v = acq$t, col = 'green')  # Add acquisition point


}
\seealso{
\code{\link{lgspline}} for fitting the model,
\code{\link{generate_posterior}} for generating posterior draws
}
