% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/api_define_extract.R
\name{define_extract_micro}
\alias{define_extract_micro}
\alias{define_extract_usa}
\alias{define_extract_cps}
\alias{define_extract_ipumsi}
\title{Define an extract request for an IPUMS microdata collection}
\usage{
define_extract_micro(
  collection,
  description,
  samples,
  variables = NULL,
  time_use_variables = NULL,
  sample_members = NULL,
  data_format = "fixed_width",
  data_structure = "rectangular",
  rectangular_on = NULL,
  case_select_who = "individuals",
  data_quality_flags = NULL
)
}
\arguments{
\item{collection}{Code for the IPUMS collection represented by this
extract request. See \code{\link[=ipums_data_collections]{ipums_data_collections()}} for supported microdata
collection codes.}

\item{description}{Description of the extract.}

\item{samples}{Vector of samples to include in the extract
request. Use \code{\link[=get_sample_info]{get_sample_info()}} to identify
sample IDs for a given collection.}

\item{variables}{Vector of variable names or a list of detailed
variable specifications to include in the extract
request. Use \code{\link[=var_spec]{var_spec()}} to create a \code{var_spec} object containing a
detailed variable specification. See examples.}

\item{time_use_variables}{Vector of names of IPUMS-defined time use variables
or a list of specifications for user-defined time use variables
to include in the extract request. Use \code{\link[=tu_var_spec]{tu_var_spec()}} to create a
\code{tu_var_spec} object containing a time use variable specification. See
examples.

Time use variables are only available for IPUMS Time Use collections
(\code{"atus"}, \code{"ahtus"}, and \code{"mtus"}).}

\item{sample_members}{Indication of whether to include additional sample
members in the extract request. If provided, must be one of
\code{"include_non_respondents"}, \code{"include_household_members"}, or both.

Sample member selection is only available for the IPUMS ATUS collection
(\code{"atus"}).}

\item{data_format}{Format for the output extract data file. Either
\code{"fixed_width"} or \code{"csv"}.

Note that while \code{"stata"}, \code{"spss"}, and \code{"sas9"} are also accepted, these
file formats are not supported by ipumsr data-reading functions.

Defaults to \code{"fixed_width"}.}

\item{data_structure}{Data structure for the output extract data.
\itemize{
\item \code{"rectangular"} provides data in which every row has the same record type
(determined by \code{"rectangular_on"}), with variables from other record types
written onto associated records of the chosen type (e.g. household
variables written onto person records).
\item \code{"hierarchical"} provides data that include rows of differing record
types, with records ordered according to their hierarchical structure (e.g.
each person record is followed by the activity records for that person).
\item \code{"household_only"} provides household records only. This data structure
is only available for the IPUMS USA collection (\code{"usa"}).
}

Defaults to \code{"rectangular"}.}

\item{rectangular_on}{If \code{data_structure} is \code{"rectangular"},
records on which to rectangularize. One of \code{"P"} (person), \code{"A"}
(activity), \code{"I"} (injury) or \code{"R"} (round).

Defaults to \code{"P"} if \code{data_structure} is \code{"rectangular"} and \code{NULL}
otherwise.}

\item{case_select_who}{Indication of how to interpret any case selections
included for variables in the extract definition.
\itemize{
\item \code{"individuals"} includes records for all individuals who match the
specified case selections.
\item \code{"households"} includes records for all members of each household that
contains an individual who matches the specified case selections.
}

Defaults to \code{"individuals"}. Use \code{\link[=var_spec]{var_spec()}} to add case selections for
specific variables.}

\item{data_quality_flags}{Set to \code{TRUE} to include data quality
flags for all applicable variables in the extract definition. This will
override the \code{data_quality_flags} specification for individual variables
in the definition.

Use \code{\link[=var_spec]{var_spec()}} to add data quality flags for specific variables.}
}
\value{
An object of class \code{\link[=ipums_extract-class]{micro_extract}} containing
the extract definition.
}
\description{
Define the parameters of an IPUMS microdata extract request to be submitted
via the IPUMS API.

The IPUMS API currently supports the following microdata collections:
\itemize{
\item IPUMS USA
\item IPUMS CPS
\item IPUMS International
\item IPUMS Time Use (ATUS, AHTUS, MTUS)
\item IPUMS Health Surveys (NHIS, MEPS)
}

Note that not all extract request parameters and options apply to all
collections. For a summary of supported features by collection, see the
\href{https://developer.ipums.org/docs/v2/apiprogram/apis/microdata/}{IPUMS API documentation}.

Learn more about the IPUMS API in \code{vignette("ipums-api")} and
microdata extract definitions in \code{vignette("ipums-api-micro")}.
}
\examples{
usa_extract <- define_extract_micro(
  collection = "usa",
  description = "2013-2014 ACS Data",
  samples = c("us2013a", "us2014a"),
  variables = c("SEX", "AGE", "YEAR")
)

usa_extract

# Use `var_spec()` to created detailed variable specifications:
usa_extract <- define_extract_micro(
  collection = "usa",
  description = "Example USA extract definition",
  samples = c("us2013a", "us2014a"),
  variables = var_spec(
    "SEX",
    case_selections = "2",
    attached_characteristics = c("mother", "father")
  )
)

# For multiple variables, provide a list of `var_spec` objects and/or
# variable names.
cps_extract <- define_extract_micro(
  collection = "cps",
  description = "Example CPS extract definition",
  samples = c("cps2020_02s", "cps2020_03s"),
  variables = list(
    var_spec("AGE", data_quality_flags = TRUE),
    var_spec("SEX", case_selections = "2"),
    "RACE"
  )
)

cps_extract

# To recycle specifications to many variables, it may be useful to
# create variables prior to defining the extract:
var_names <- c("AGE", "SEX")

my_vars <- purrr::map(
  var_names,
  ~ var_spec(.x, attached_characteristics = "mother")
)

ipumsi_extract <- define_extract_micro(
  collection = "ipumsi",
  description = "Extract definition with predefined variables",
  samples = c("br2010a", "cl2017a"),
  variables = my_vars
)

# Extract specifications can be indexed by name
names(ipumsi_extract$samples)

names(ipumsi_extract$variables)

ipumsi_extract$variables$AGE

# IPUMS Time Use collections allow selection of IPUMS-defined and
# user-defined time use variables:
define_extract_micro(
  collection = "atus",
  description = "ATUS extract with time use variables",
  samples = "at2007",
  time_use_variables = list(
    "ACT_PCARE",
    tu_var_spec(
      "MYTIMEUSEVAR",
      owner = "example@example.com"
    )
  )
)

\dontrun{
# Use the extract definition to submit an extract request to the API
submit_extract(usa_extract)
}
}
\seealso{
\code{\link[=submit_extract]{submit_extract()}} to submit an extract request for processing.

\code{\link[=save_extract_as_json]{save_extract_as_json()}} and \code{\link[=define_extract_from_json]{define_extract_from_json()}} to share an
extract definition.
}
