#' Generate spectrograms for seismic stations at different time periods
#' 
#' The function generates a set of spectrograms (PSDs) for all seismic
#' stations provided, for daily, weekly, monthly and total time periods. 
#' It depends on seismic files being organised in a coherent structure as, 
#' for example, generated by \code{aux_Organisecubefiles}.
#' 
#' The function calls a series of other functions, partly with modified 
#' default values, which can be changed by the ...-argument. By default,
#' the seismic files are imported as eseis objects using 
#' \code{aux_getevent(..., eseis = TRUE)}. The signals are deconvolved with
#' \code{signal_deconvolve()} using the default options, i.e., 
#' \code{sensor = "TC120s"} and \code{logger = "Cube3extBOB"}.
#' Then, the signals are bandpass filtered with \code{signal_filter}, using 
#' \code{f = c(1, 90)}. The PSDs are calculated with \code{signal_spectrogram} 
#' using \code{Welch = TRUE}, \code{window = 90} and \code{window_sub = 30}. 
#' 
#' This and all other aux-functions are primarily written for internal use 
#' amongst the GFZ Geomorphology Section group members and their usual data 
#' handling scheme. Thus, they may be of limited use when adopted for other 
#' scopes. However, many of these functions are internally consistent in 
#' usage.
#' 
#' @param station \code{Character} value, seismic station ID, which must
#' correspond to the ID in the file name of the data directory structure 
#' (cf. \code{aux_organisecubefiles()}).
#' 
#' @param component \code{Character} value, seismic component, which must
#' correspond to the component name in the file name of the data directory  
#' structure (cf. \code{aux_organisecubefiles()}). Default is 
#' \code{"BHZ"} (vertical component of a sac file).
#' 
#' @param period \code{POSIXct} vector of length two, time period to be 
#' processed. 
#' 
#' @param output \code{Character} vector, output PSD types. One or more out 
#' of \code{"daily"}, \code{"weekly"}, \code{"monthly"}, \code{"total"}. 
#' Default is \code{c("daily", "weekly", "monthly", "total")}.
#' 
#' @param input_dir \code{Character} value, path to directory where the 
#' seismic files are stored. 
#' 
#' @param output_dir \code{Character} value, path to directory where PSD
#' image files are saved to.
#' 
#' @param aggregate \code{Numeric} vector of length two, aggregation factors 
#' for the processed PSD matrics. First entry denotes time aggregation, 
#' second entry frequency aggregation. Default is \code{c(1, 5)}.
#' 
#' @param n_dates \code{Numeric} value, final number of spectra per output 
#' PSD. Default is \code{2000}.
#' 
#' @param jpg_dim \code{Numeric} vector of length four, JPEG image properties 
#' in the form \code{c(width, height, resolution, quality)}. Default is 
#' \code{c(4444, 2500, 300, 90)}.
#' 
#' @param verbose \code{Logical} value, optional screen output of processing 
#' progress. Default is \code{FALSE}.
#' 
#' @param \dots Additional arguments passed to different functions. See 
#' details section for default values.
#' 
#' @return A set of JPEG images wirtten to disk
#' 
#' @author Michael Dietze
#' 
#' @keywords eseis
#' 
#' @examples
#' 
#' \dontrun{
#' 
#' ## PSD generation with minimum input arguments
#' aux_psdsummary(station = c("STA01", "STA02"), 
#'                period = as.POSIXct(x = c("2017-04-01",
#'                                          "2017-04-03"), 
#'                                    tz = "UTC"), 
#'                input_dir = "~/data/seismic/sac/")
#' 
#' ## PSD generation with some more arguments
#' aux_psdsummary(station = c("STA01", "STA02"), 
#'                component = "BHZ",
#'                period = as.POSIXct(x = c("2017-04-01",
#'                                          "2017-04-03"), 
#'                                    tz = "UTC"), 
#'                output = c("daily", "monthly"),
#'                input_dir = "~/data/seismic/sac/",
#'                aggregate = c(2, 10), 
#'                n_dates = 1000,
#'                jpg_dim = c(1600, 900, 300, 50), 
#'                verbose = TRUE)
#' }
#' 
#' @export aux_psdsummary
aux_psdsummary <- function(
  
  station,
  component = "BHZ",
  period,
  output = c("daily", "weekly", "monthly", "total"),
  input_dir,
  output_dir,
  aggregate = c(1, 5),
  n_dates = 2000,
  jpg_dim = c(4444, 2500, 300, 90),
  verbose = FALSE,
  ...
) {
  
  ## check/set additional arguments -------------------------------------------
  extraArgs <- list(...)
  
  ## set sensor type
  if("sensor" %in% names(extraArgs)) {
    
    sensor <- extraArgs$sensor
  } else {
    
    sensor <- "TC120s"
  }
  
  ## set logger type
  if("logger" %in% names(extraArgs)) {
    
    logger <- extraArgs$logger
  } else {
    
    logger <- "Cube3extBOB"
  }
  
  ## set NA replace option
  if("na.replace" %in% names(extraArgs)) {
    
    na.replace <- extraArgs$na.replace
  } else {
    
    na.replace <- FALSE
  }
  
  ## set filter frequencies
  if("f" %in% names(extraArgs)) {
    
    f <- extraArgs$f
  } else {
    
    f <- c(1, 90)
  }
  
  ## set filter type
  if("type" %in% names(extraArgs)) {
    
    type <- extraArgs$type
  } else {
    
    type <- c("BP")
  }
  
  ## set Welch option for PSD
  if("Welch" %in% names(extraArgs)) {
    
    Welch <- extraArgs$Welch
  } else {
    
    Welch <- TRUE
  }
  
  ## set window for PSD
  if("window" %in% names(extraArgs)) {
    
    window <- extraArgs$window
  } else {
    
    window <- 90
  }
  
  ## set sub window for PSD
  if("window_sub" %in% names(extraArgs)) {
    
    window_sub <- extraArgs$window_sub
  } else {
    
    window_sub <- 30
  }

  ## set z-limits
  if("zlim" %in% names(extraArgs)) {
    
    zlim_psd <- extraArgs$zlim
  } else {
    
    zlim_psd <- c(-180, -100)
  }
  
  ## set legend
  if("legend" %in% names(extraArgs)) {
    
    legend <- extraArgs$legend
  } else {
    
    legend <- TRUE
  }
  
  ## remove keywords from plot arguments
  keywords <- c("zlim", "legend", "f", "sensor", "logger", 
                "type", "Welch", "window", "window_sub", "na.replace")
  
  extraArgs <- extraArgs[!names(extraArgs)%in%keywords]
  
  ## create directory structure -----------------------------------------------

  ## check/set output directory
  if(missing(output_dir) == TRUE) {
    
    output_dir <- file.path(tempdir(), "output")
    print(paste("Output will be written to", output_dir))
  }
  
  if(dir.exists(output_dir) == FALSE) {

    warning("Output directory did not exist. Created by function")
    dir.create(output_dir)
  }
  
  if("total" %in% output == TRUE) {
    
    if(dir.exists(paste(output_dir, "/total", sep = "")) == FALSE) {
      
      dir.create(path = paste(output_dir, "/total", sep = ""))
    }
  }
  
  if("monthly" %in% output == TRUE) {
    
    if(dir.exists(paste(output_dir, "/monthly", sep = "")) == FALSE) {
      
      dir.create(path = paste(output_dir, "/monthly", sep = ""))
    }
  }
  
  if("weekly" %in% output == TRUE) {
    
    if(dir.exists(paste(output_dir, "/weekly", sep = "")) == FALSE) {
      
      dir.create(path = paste(output_dir, "/weekly", sep = ""))
    }
  }
  
  if("daily" %in% output == TRUE) {
    
    if(dir.exists(paste(output_dir, "/daily", sep = "")) == FALSE) {
      
      dir.create(path = paste(output_dir, "/daily", sep = ""))
    }
  }
  
  ## GENERATE PROCESSING PERIOD -----------------------------------------------
  
  days_show <- seq(from = period[1], 
                   to = period[2], 
                   by = 24 * 60 * 60)
  
  JD_show <- format(x = days_show, 
                    format = "%j")
  
  ## PROCESSING ---------------------------------------------------------------
  
  ## loop through all stations
  for(i in 1:length(station)) {
    
    ## create output object
    P_list <- vector(mode = "list", 
                     length = length(JD_show))
    
    ## loop through all days
    for(j in 1:length(JD_show)) {
      
      ## PROCESSING | DATA IMPORT AND PROCESSING ------------------------------
      
      ## set start time, accounting for boundary condition
      if(j == 1) {
        
        day_start <- days_show[j]
      } else {
        
        day_start <- days_show[j] - 600
      }

      ## read input files
      s <- try(eseis::aux_getevent(start = day_start, 
                                   duration = 24 * 60 * 60 + 600, 
                                   station = station[i], 
                                   component = component, 
                                   dir = input_dir,
                                   eseis = TRUE), 
               silent = TRUE)

      ## deconvolve data
      s <- try(eseis::signal_deconvolve(data = s, 
                                 sensor = sensor, 
                                 logger = logger,
                                 na.replace = na.replace),
               silent = TRUE)
      
      ## detrend data
      s <- try(eseis::signal_detrend(data = s),
               silent = TRUE)
      
      ## filter data
      s <- try(eseis::signal_filter(data = s, 
                             f = f,
                             type = type),
               silent = TRUE)
      
      ## taper data
      s <- try(eseis::signal_taper(data = s,
                            n = 1 / s$meta$dt * 600 / 2),
               silent = TRUE)
      
      ## PROCESSING | PSD CALCULATION AND ERROR HANDLING ----------------------
      
      ## create PSD
      P <- try(eseis::signal_spectrogram(data = s, 
                                  window = window, 
                                  Welch = Welch, 
                                  window_sub = window_sub),
               silent = TRUE)
      
      ## check/replace PSD if unsuccessfully calculated
      if(class(P) == "try-error") {
        
        dt <- try(s$meta$dt, 
                  silent = TRUE)
        
        if(class(dt) == "try-error" | is.null(dt) == TRUE) {
          
          dt <- 1 / 200
        }
        
        t_null <- seq(from = days_show[j], 
                      to = days_show[j] + 24 * 60 * 60, 
                      by = 3600)
        
        f_null <- seq(from = 0, 
                      to = 0.5 / dt, 
                      length.out = 0.5 / dt * 
                        ifelse(Welch == TRUE, window_sub, window))
        
        P_null <- matrix(data = rep(-500, length(t_null) * length(f_null)),
                         ncol = length(t_null), 
                         nrow = length(f_null))
        
        P_plot <- list(S = P_null,
                       t = t_null,
                       f = f_null)
        
      } else {
        
        P_plot <- P$PSD
      }
      
      ## clip PSD
      P_clip <- P_plot
      
      P_clip$S <- P_clip$S[,P_plot$t >= days_show[j] & 
                             P_plot$t <= days_show[j] + 24 * 60 * 60]
      
      P_clip$t <- P_clip$t[P_plot$t >= days_show[j] & 
                             P_plot$t <= days_show[j] + 24 * 60 * 60]
      
      ## aggregate PSD
      P_agg <- P_clip
      
      t_agg <- seq(from = 1, 
                   by = aggregate[1], 
                   to = length(P_agg$t))
      
      f_agg <- seq(from = 1, 
                   by = aggregate[2], 
                   to = length(P_agg$f))
      
      P_agg$S <- P_agg$S[f_agg, t_agg]
      P_agg$t <- P_agg$t[t_agg]
      P_agg$f <- P_agg$f[f_agg]
      
      ## assign PSD to output list
      P_list[[j]] <- P_agg
      
      ## print progress information
      if(verbose == TRUE) {
        
        print(paste("PSD for station ", 
                    station[i], 
                    " and JD ", 
                    JD_show[j], 
                    " done (",
                    Sys.time(), 
                    ").", 
                    sep = ""))        
      }
    }
    
    ## merge list elements to vectors and matrix
    S_total <- do.call(cbind, lapply(X = P_list, FUN = function(P_list) {
      
      P_list$S
    }))
    
    t_total <- do.call(c, lapply(X = P_list, FUN = function(P_list) {
      
      P_list$t
    }))
    
    f_total <- P_list[[1]]$f
    
    ## convert data to PSD object
    P_total <- list(S = S_total,
                    t = t_total,
                    f = f_total)
    
    ## PLOTTING | TOTAL PERIOD ------------------------------------------------
    if("total" %in% output == TRUE) {
      
      ## decrease time resolution
      if(length(P_total$t) > n_dates) {
        
        rescale_index <- seq(from = 1, 
                             to = length(P_total$t), 
                             by = round(x = length(P_total$t) / n_dates, 
                                        digits = 0))
      } else {
        
        rescale_index <- seq(from = 1,
                             to = length(P_total$t))
      }
      
      P_total_res <- P_total
      P_total_res$S <- P_total$S[,rescale_index]
      P_total_res$t <- P_total$t[rescale_index]
      
      ## prepare output device
      jpeg(filename = paste(output_dir,
                            "/total/", 
                            station[i], 
                            "_PSD_total.jpg", 
                            sep = ""), 
           width = jpg_dim[1], 
           height = jpg_dim[2], 
           res = jpg_dim[3],
           quality = jpg_dim[4])
      
      ## generate plot
      do.call(what = eseis::plot_spectrogram, 
              args = c(list(data = P_total_res,
                            main = paste(station[i], 
                                         " | total period (f = ", 
                                         f[1],
                                         "-",
                                         f[2],
                                         ")",
                                         sep = ""), 
                            zlim = zlim_psd,
                            legend = legend), 
                       extraArgs))

      ## close plot device
      dev.off()
    }
    
    ## PLOTTING | MONTHLY PLOTS -------------------------------------------------
    if("monthly" %in% output == TRUE) {
      
      ## identify months
      months_plot <- as.numeric(x = format(x = P_total$t, 
                                           format = "%m"))
      
      months_unique <- unique(x = months_plot)
      
      ## loop through all months
      for(k in 1:length(months_unique)) {
        
        ## crop data set
        P_monthly <- P_total
        P_monthly$S <- P_monthly$S[,months_plot == months_unique[k]]
        P_monthly$t <- P_monthly$t[months_plot == months_unique[k]]
        
        ## decrease time resolution
        if(length(P_monthly$t) > n_dates) {
          
          rescale_index <- seq(from = 1, 
                               to = length(P_monthly$t), 
                               by = round(x = length(P_monthly$t) / n_dates, 
                                          digits = 0))
        } else {
          
          rescale_index <- seq(from = 1,
                               to = length(P_monthly$t))
        }
        
        P_monthly_res <- P_monthly
        P_monthly_res$S <- P_monthly$S[,rescale_index]
        P_monthly_res$t <- P_monthly$t[rescale_index]
        
        ## generate month number
        n_month <- months_unique[k]
        
        if(nchar(n_month) == 1) {
          n_month <- paste("0", 
                           n_month, 
                           sep = "")
        }
        
        ## prepare output device
        jpeg(filename = paste(output_dir,
                              "/monthly/", 
                              station[i], 
                              "_PSD_month_",
                              n_month, ".jpg", 
                              sep = ""), 
             width = jpg_dim[1], 
             height = jpg_dim[2], 
             res = jpg_dim[3],
             quality = jpg_dim[4])
        
        ## generate plot
        do.call(what = eseis::plot_spectrogram, 
                args = c(list(data = P_monthly_res,
                              main = paste(station[i], 
                                           " | month ",
                                           months_unique[k],
                                           " (f = ", 
                                           f[1],
                                           "-",
                                           f[2],
                                           ")",
                                           sep = ""), 
                              zlim = zlim_psd,
                              legend = legend), 
                         extraArgs))
        
        ## close plot device
        dev.off()
      }
    }
    
    ## PLOTTING | WEEKLY PLOTS --------------------------------------------------
    if("weekly" %in% output == TRUE) {
      
      ## identify months
      weeks_plot <- as.numeric(x = format(x = P_total$t, 
                                          format = "%W"))
      
      weeks_unique <- unique(x = weeks_plot)
      
      ## loop through all weeks
      for(k in 1:length(weeks_unique)) {
        
        ## crop data set
        P_weekly <- P_total
        P_weekly$S <- P_weekly$S[,weeks_plot == weeks_unique[k]]
        P_weekly$t <- P_weekly$t[weeks_plot == weeks_unique[k]]
        
        ## decrease time resolution
        if(length(P_weekly$t) > n_dates) {
          
          rescale_index <- seq(from = 1, 
                               to = length(P_weekly$t), 
                               by = round(x = length(P_weekly$t) / n_dates, 
                                          digits = 0))
        } else {
          
          rescale_index <- seq(from = 1,
                               to = length(P_weekly$t))
        }
        
        P_weekly_res <- P_weekly
        P_weekly_res$S <- P_weekly$S[,rescale_index]
        P_weekly_res$t <- P_weekly$t[rescale_index]
        
        ## generate week number
        n_week <- weeks_unique[k]
        
        if(nchar(n_week) == 1) {
          n_week <- paste("0", 
                          n_week, 
                          sep = "")
        }
        
        ## prepare output device
        jpeg(filename = paste(output_dir,
                              "/weekly/", 
                              station[i], 
                              "_PSD_week_",
                              n_week, ".jpg", 
                              sep = ""), 
             width = jpg_dim[1], 
             height = jpg_dim[2], 
             res = jpg_dim[3],
             quality = jpg_dim[4])
        
        ## generate plot
        do.call(what = eseis::plot_spectrogram, 
                args = c(list(data = P_weekly_res,
                              main = paste(station[i], 
                                           " | week ",
                                           weeks_unique[k],
                                           " (f = ", 
                                           f[1],
                                           "-",
                                           f[2],
                                           ")",
                                           sep = ""), 
                              zlim = zlim_psd,
                              legend = legend), 
                         extraArgs))
        
        ## close plot device
        dev.off()
      }
    }
    
    ## PLOTTING | DAILY PLOTS ---------------------------------------------------
    if("daily" %in% output == TRUE) {
      
      ## identify months
      days_plot <- as.numeric(x = format(x = P_total$t, 
                                         format = "%j"))
      
      days_unique <- unique(x = days_plot)
      
      ## loop through all days
      for(k in 1:length(days_unique)) {
        
        ## crop data set
        P_daily <- P_total
        P_daily$S <- P_daily$S[,days_plot == days_unique[k]]
        P_daily$t <- P_daily$t[days_plot == days_unique[k]]
        
        ## decrease time resolution
        if(length(P_daily$t) > n_dates) {
          
          rescale_index <- seq(from = 1, 
                               to = length(P_daily$t), 
                               by = round(x = length(P_daily$t) / n_dates, 
                                          digits = 0))
        } else {
          
          rescale_index <- seq(from = 1,
                               to = length(P_daily$t))
        }
        
        P_daily_res <- P_daily
        P_daily_res$S <- P_daily$S[,rescale_index]
        P_daily_res$t <- P_daily$t[rescale_index]
        
        ## generate month number
        n_day <- days_unique[k]
        
        if(nchar(n_day) == 1) {
          n_week <- paste("0", 
                          n_week, 
                          sep = "")
        }
        
        ## prepare output device
        jpeg(filename = paste(output_dir,
                              "/daily/", 
                              station[i], 
                              "_PSD_JD_",
                              n_day, ".jpg", 
                              sep = ""), 
             width = jpg_dim[1], 
             height = jpg_dim[2], 
             res = jpg_dim[3],
             quality = jpg_dim[4])
        
        ## generate plot
        try(do.call(what = eseis::plot_spectrogram, 
                args = c(list(data = P_daily_res,
                              main = paste(station[i], 
                                           " | JD ",
                                           days_unique[k],
                                           " (f = ", 
                                           f[1],
                                           "-",
                                           f[2],
                                           ")",
                                           sep = ""),
                              zlim = zlim_psd,
                              legend = legend), 
                         extraArgs)), 
            silent = TRUE)
        
        ## close plot device
        dev.off()
      }
    }
  }
}





