% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vcov.cgaim.R, R/vcov.boot.cgaim.R
\name{vcov.cgaim}
\alias{vcov.cgaim}
\alias{vcov.boot.cgaim}
\title{Calculate Variance-Covariance Matrix for a Fitted CGAIM Object}
\usage{
\method{vcov}{cgaim}(object, parm = c("alpha", "beta"), type = c("normal",
  "bootstrap"), B = 100, complete = TRUE, ...)

\method{vcov}{boot.cgaim}(object, parm = c("alpha", "beta"),
  complete = TRUE, ...)
}
\arguments{
\item{object}{A \code{cgaim} or \code{boot.cgaim} object.}

\item{parm}{The model components for which to get confidence intervals. 
Either \code{"alpha"} (the default) for index weights or \code{"beta"} for scaling coefficients.}

\item{type}{The type of confidence intervals. Either \code{"normal"} (the default)
or \code{"bootstrap"}. See details.}

\item{B}{The number of samples to be simulated.}

\item{complete}{Indicates whether the full variance-covariance matrix should be returned when some of the parameters could not be estimated. If so, the matrix is padded with \code{NA}s.}

\item{...}{Additional parameters to be passed to \code{\link{boot.cgaim}} for bootstrap replications.}
}
\value{
A variance-covariance matrix object.
}
\description{
Returns the variance covariance matrix of the main parameters of a fitted \code{cgaim} object. These parameters correspond to the index weights \code{alpha} and the scaling coefficients \code{beta}.
}
\details{
Two types of computation are currently implemented in the function.
When \code{type = "normal"}, variance-covariance matrices are computed assuming 
components are normally distributed. Beta coefficients are treated as
regular linear regression coefficients and alpha
coefficients are assumed to follow a Truncated Multivariate Normal distribution. 
The latter is obtained by simulating from TMVN (see \code{\link[TruncatedNormal]{tmvnorm}}) 
and computing the empirical variance covariance matrix from these simulations. The parameter \code{B} controls the number of simulations from the TMVN (and is not used when \code{parm = "beta"}).

When \code{type = "bootstrap"}, the variance-covariance matrix is computed on Bootstrap replications. In this case \code{\link{boot.cgaim}} is called internally and \code{B} corresponds to the number of replications. Alternatively, the user can directly call \code{\link{boot.cgaim}} and feed the result into \code{vcov.boot.cgaim} (see examples).
}
\examples{
# A simple CGAIM
n <- 200
x1 <- rnorm(n)
x2 <- x1 + rnorm(n)
z <- x1 + x2
y <- z + rnorm(n)
df1 <- data.frame(y, x1, x2) 
ans <- cgaim(y ~ g(x1, x2, acons = list(monotone = 1)), data = df1)

# (Truncated) Normal variance-covariance matrix
set.seed(1)
vcov(ans, B = 1000)
set.seed(1)
vcov(ans, parm = "alpha", B = 1000) # Same result
vcov(ans, parm = "beta", B = 1000)

# Confidence intervals by bootstrap (more computationally intensive, B should be increased)
set.seed(2)
vcov(ans, type = "boot", B = 10)

# Alternatively, bootstrap samples can be performed beforehand
set.seed(2) 
boot1 <- boot.cgaim(ans, B = 10)
vcov(boot1)

}
\references{
Masselot, P. and others, 2022. Constrained groupwise additive index models.
    Biostatistics.
    
  Pya, N., Wood, S.N., 2015. Shape constrained additive models. 
   Stat. Comput. 25, 543–559. 
   
  Wood, S.N., 2017. Generalized Additive Models: An Introduction with R, 
    2nd ed, Texts in Statistical Science. Chapman and Hall/CRC.
}
\seealso{
\code{\link{boot.cgaim}} for bootstrapping and \code{\link{confint.cgaim}} for confidence intervals.
}
