% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xEnricherGenesAdv.r
\name{xEnricherGenesAdv}
\alias{xEnricherGenesAdv}
\title{Function to conduct enrichment analysis given a list of gene sets and a list of ontologies}
\usage{
xEnricherGenesAdv(list_vec, background = NULL, check.symbol.identity =
F,
ontologies = NA, size.range = c(10, 2000), min.overlap = 3,
which.distance = NULL, test = c("hypergeo", "fisher", "binomial"),
background.annotatable.only = NULL, p.tail = c("one-tail",
"two-tails"),
p.adjust.method = c("BH", "BY", "bonferroni", "holm", "hochberg",
"hommel"),
ontology.algorithm = c("none", "pc", "elim", "lea"), elim.pvalue =
0.01,
lea.depth = 2, path.mode = c("all_paths", "shortest_paths",
"all_shortest_paths"), true.path.rule = F, verbose = T, silent = FALSE,
plot = TRUE, fdr.cutoff = 0.05, displayBy = c("zscore", "fdr",
"pvalue",
"fc", "or"), RData.location = "http://galahad.well.ox.ac.uk/bigdata")
}
\arguments{
\item{list_vec}{an input vector containing gene symbols. Alternatively
it can be a list of vectors, representing multiple groups of genes}

\item{background}{a background vector containing gene symbols as the
test background. If NULL, by default all annotatable are used as
background}

\item{check.symbol.identity}{logical to indicate whether to match the
input data/background via Synonyms for those unmatchable by official
gene symbols. By default, it sets to false}

\item{ontologies}{the ontologies supported currently. Pre-built
ontology and annotation data are detailed in
\code{\link{xDefineOntology}}.}

\item{size.range}{the minimum and maximum size of members of each term
in consideration. By default, it sets to a minimum of 10 but no more
than 2000}

\item{min.overlap}{the minimum number of overlaps. Only those terms
with members that overlap with input data at least min.overlap (3 by
default) will be processed}

\item{which.distance}{which terms with the distance away from the
ontology root (if any) is used to restrict terms in consideration. By
default, it sets to 'NULL' to consider all distances}

\item{test}{the test statistic used. It can be "fisher" for using
fisher's exact test, "hypergeo" for using hypergeometric test, or
"binomial" for using binomial test. Fisher's exact test is to test the
independence between gene group (genes belonging to a group or not) and
gene annotation (genes annotated by a term or not), and thus compare
sampling to the left part of background (after sampling without
replacement). Hypergeometric test is to sample at random (without
replacement) from the background containing annotated and non-annotated
genes, and thus compare sampling to background. Unlike hypergeometric
test, binomial test is to sample at random (with replacement) from the
background with the constant probability. In terms of the ease of
finding the significance, they are in order: hypergeometric test >
fisher's exact test > binomial test. In other words, in terms of the
calculated p-value, hypergeometric test < fisher's exact test <
binomial test}

\item{background.annotatable.only}{logical to indicate whether the
background is further restricted to the annotatable. By default, it is
NULL: if ontology.algorithm is not 'none', it is always TRUE;
otherwise, it depends on the background (if not provided, it will be
TRUE; otherwise FALSE). Surely, it can be explicitly stated}

\item{p.tail}{the tail used to calculate p-values. It can be either
"two-tails" for the significance based on two-tails (ie both over- and
under-overrepresentation) or "one-tail" (by default) for the
significance based on one tail (ie only over-representation)}

\item{p.adjust.method}{the method used to adjust p-values. It can be
one of "BH", "BY", "bonferroni", "holm", "hochberg" and "hommel". The
first two methods "BH" (widely used) and "BY" control the false
discovery rate (FDR: the expected proportion of false discoveries
amongst the rejected hypotheses); the last four methods "bonferroni",
"holm", "hochberg" and "hommel" are designed to give strong control of
the family-wise error rate (FWER). Notes: FDR is a less stringent
condition than FWER}

\item{ontology.algorithm}{the algorithm used to account for the
hierarchy of the ontology. It can be one of "none", "pc", "elim" and
"lea". For details, please see 'Note' below}

\item{elim.pvalue}{the parameter only used when "ontology.algorithm" is
"elim". It is used to control how to declare a signficantly enriched
term (and subsequently all genes in this term are eliminated from all
its ancestors)}

\item{lea.depth}{the parameter only used when "ontology.algorithm" is
"lea". It is used to control how many maximum depth is used to consider
the children of a term (and subsequently all genes in these children
term are eliminated from the use for the recalculation of the
signifance at this term)}

\item{path.mode}{the mode of paths induced by vertices/nodes with input
annotation data. It can be "all_paths" for all possible paths to the
root, "shortest_paths" for only one path to the root (for each node in
query), "all_shortest_paths" for all shortest paths to the root (i.e.
for each node, find all shortest paths with the equal lengths)}

\item{true.path.rule}{logical to indicate whether the true-path rule
should be applied to propagate annotations. By default, it sets to
false}

\item{verbose}{logical to indicate whether the messages will be
displayed in the screen. By default, it sets to false for no display}

\item{silent}{logical to indicate whether the messages will be silent
completely. By default, it sets to false. If true, verbose will be
forced to be false}

\item{plot}{logical to indicate whether heatmap plot is drawn}

\item{fdr.cutoff}{fdr cutoff used to declare the significant terms. By
default, it is set to 0.05. This option only works when setting plot
(see above) is TRUE}

\item{displayBy}{which statistics will be used for drawing heatmap. It
can be "fc" for enrichment fold change, "fdr" for adjusted p value (or
FDR), "pvalue" for p value, "zscore" for enrichment z-score (by
default), "or" for odds ratio. This option only works when setting plot
(see above) is TRUE}

\item{RData.location}{the characters to tell the location of built-in
RData files. See \code{\link{xRDataLoader}} for details}
}
\value{
an object of class "ls_eTerm", a list with following components:
\itemize{
\item{\code{df}: a data frame of n x 12, where the 12 columns are
"group" (the input group names), "ontology" (input ontologies), "id"
(term ID), "name" (term name), "nAnno" (number in members annotated by
a term), "nOverlap" (number in overlaps), "fc" (enrichment fold
changes), "zscore" (enrichment z-score), "pvalue" (nominal p value),
"adjp" (adjusted p value (FDR)), "or" (odds ratio), "CIl" (lower bound
confidence interval for the odds ratio), "CIu" (upper bound confidence
interval for the odds ratio), "distance" (term distance or other
information), "members" (members (represented as Gene Symbols) in
overlaps)}
\item{\code{mat}: NULL if the plot is not drawn; otherwise, a matrix of
term names X groups with numeric values for the signficant enrichment,
NA for the insignificant ones}
\item{\code{gp}: NULL if the plot is not drawn; otherwise, a 'ggplot'
object}
}
}
\description{
\code{xEnricherGenesAdv} is supposed to conduct enrichment analysis
given a list of gene sets and a list of ontologies. It is an advanced
version of \code{xEnricherGenes}, returning an object of the class
'ls_eTerm'.
}
\note{
none
}
\examples{
\dontrun{
# Load the library
library(XGR)
RData.location <- "http://galahad.well.ox.ac.uk/bigdata/"

# Gene-based enrichment analysis using ontologies (REACTOME and GOMF)
# a) provide the input Genes of interest (eg 100 randomly chosen human genes)
## load human genes
org.Hs.eg <- xRDataLoader(RData='org.Hs.eg',
RData.location=RData.location)
set.seed(825)
data <- as.character(sample(org.Hs.eg$gene_info$Symbol, 100))
data

# optionally, provide the test background (if not provided, all human genes)
#background <- as.character(org.Hs.eg$gene_info$Symbol)

# b) perform enrichment analysis
ls_eTerm <- xEnricherGenesAdv(data, ontologies=c("REACTOME","GOMF"),
RData.location=RData.location)
ls_eTerm
## forest plot of enrichment results
gp <- xEnrichForest(ls_eTerm, top_num=10)
## heatmap plot of enrichment results
gp <- xEnrichHeatmap(ls_eTerm, fdr.cutoff=0.1, displayBy="or")
}
}
\seealso{
\code{\link{xRDataLoader}}, \code{\link{xEnricherGenes}},
\code{\link{xEnrichViewer}}, \code{\link{xHeatmap}}
}
