% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get-nmfi.R
\name{get_nmfi}
\alias{get_nmfi}
\title{Calculate Normalised MFI (nMFI) Values for a Plate}
\usage{
get_nmfi(
  plate,
  reference_dilution = 1/400,
  data_type = "Median",
  sample_type_filter = "ALL",
  verbose = TRUE
)
}
\arguments{
\item{plate}{(\code{Plate()}) a plate object for which to calculate the nMFI values}

\item{reference_dilution}{(\verb{numeric(1) or character(1)}) the dilution value of the standard curve sample
to use as a reference for normalisation. The default is \code{1/400}.
It should refer to a dilution of a standard curve sample in the given plate object.
This parameter could be either a numeric value or a string.
In case it is a character string, it should have format \verb{1/d+}, where \verb{d+} is any positive integer.}

\item{data_type}{(\code{character(1)}) type of data for the computation. Median is the default}

\item{sample_type_filter}{(\code{character()}) The types of samples to normalise.
(e.g., \code{"TEST"}, \code{"STANDARD CURVE"}). It can also be a vector of sample types.
In that case, dataframe with multiple sample types will be returned.
The default value is \code{"ALL"}, which corresponds to returning all the samples.}

\item{verbose}{(\code{logical(1)}) print additional information. The default is \code{TRUE}}
}
\value{
nmfi (\code{data.frame}) a data frame with normalised MFI values for each analyte in the plate and all test samples.
}
\description{
Calculates normalised MFI (nMFI) values for each analyte in a Luminex plate.
The nMFI values are computed as the ratio of each test sample's MFI to the
MFI of a standard curve sample at a specified reference dilution.
}
\details{
Normalised MFI (nMFI) is a simple, model-free metric used to compare test
sample responses relative to a fixed concentration from the standard curve.
It is particularly useful when model fitting (e.g., for RAU calculation)
is unreliable or not possible, such as when test sample intensities fall
outside the standard curve range.

The function locates standard samples with the specified dilution and divides
each test sample’s MFI by the corresponding standard MFI value for each analyte.
}
\section{When Should nMFI Be Used?}{

While RAU values are generally preferred for antibody quantification,
they require successful model fitting of the standard curve.
This may not be feasible when:
\itemize{
\item The test samples produce MFI values outside the range of the standard curve.
\item The standard curve is poorly shaped or missing critical points.
}

In such cases, nMFI serves as a useful alternative, allowing for
plate-to-plate comparison without the need for extrapolation.
}

\examples{

# read the plate
plate_file <- system.file("extdata", "CovidOISExPONTENT.csv", package = "SerolyzeR")
layout_file <- system.file("extdata", "CovidOISExPONTENT_layout.csv", package = "SerolyzeR")

plate <- read_luminex_data(plate_file, layout_file)

# artificially bump up the MFI values of the test samples (the Median data type is default one)
plate$data[["Median"]][plate$sample_types == "TEST", ] <-
  plate$data[["Median"]][plate$sample_types == "TEST", ] * 10

# calculate the nMFI values
nmfi <- get_nmfi(plate, reference_dilution = 1 / 400)

# we don't do any extrapolation and the values should be comparable across plates
head(nmfi)
# different params
nmfi <- get_nmfi(plate, reference_dilution = "1/50")
nmfi <- get_nmfi(plate, reference_dilution = "1/50", sample_type_filter = c("TEST", "BLANK"))

}
\references{
L. Y. Chan, E. K. Yim, and A. B. Choo, Normalized median fluorescence: An alternative flow cytometry analysis method for tracking human embryonic stem cell states during differentiation,  http://dx.doi.org/10.1089/ten.tec.2012.0150
}
