#' @title Write/Overwrite Raven rvh file
#'
#' @description
#' Given an HRU and SubBasin dataframe, writes to the specified .rvh file. In the case of rvn_rvh_overwrite,
#' just the :SubBasins-:EndSubBasins and :HRUs-:EndHRUs blocks are re-written, retaining all other content.
#'
#' @details
#' \code{rvn_rvh_write} is typically used to create a 'clean' .rvh file.
#'
#' \code{\link{rvn_rvh_overwrite}} is usually used after reading an original .rvh file and
#'   processing the HRU and SubBasin tables, using (e.g., \code{\link{rvn_rvh_cleanhrus}}). This may also be used
#'   to preserve commands in the rvh file such as reservoir information, comments outside of the
#'   subbasin and HRU blocks, RedirectToFile commands, etc.
#'
#' Note that if using the rvn_rvh_overwrite function and the filename and basefile arguments are the same, the
#' original file will be overwritten while preserving lines outside of the subbasin and HRU blocks.
#'
#' If using \code{rvn_rvh_write}, the SBtable or HRUtable parameters may be omitted and provided as \code{NULL}.
#' In these cases, those sections will not be written in the rvh file. This may be useful in cases where one wishes to
#' separate the SubBasins and HRUs in different files.
#'
#' @param filename filepath of rvh file to write to (with .rvh extension)
#' @param SBtable Valid subbasin dataframe with required columns "SBID", "Name", "Downstream_ID", "Profile",
#'   "ReachLength", and "Gauged". Can have additional columns.
#' @param HRUtable Valid HRU dataframe with required columns 'ID', 'Area', 'Elevation', 'Latitude', 'Longitude',
#'    'SBID', 'LandUse', 'Vegetation', 'SoilProfile', 'Aquifer', 'Terrain', 'Slope', and 'Aspect'. Can have additional columns.
#' @param description (optional) Description added after file header
#' @param author (optional) Name of author, to be printed in file header.
#' @param basefile original rvh file to overwrite (only used with rvn_rvh_overwrite)
#'
#' @return \item{TRUE}{returns \code{TRUE} if function runs properly}
#' @author James R. Craig, University of Waterloo
#' @author Leland Scantlebury
#'
#' @seealso
#' \code{\link{rvn_rvh_read}} for the function used to read in the HRU and SubBasin data
#' \code{\link{rvn_rvh_cleanhrus}} for the function used to process HRU dataframe\cr
#' For generating blank SubBasin and HRU tables, see: \code{\link{rvn_rvh_blankSBdf}} and \code{\link{rvn_rvh_blankHRUdf}}
#'
#' @examples
#' ## Example: write a blank rvh file
#' ## create some blank tables
#' sbs_data <- rvn_rvh_blankSBdf(nSubBasins = 1)
#' hru_data <- rvn_rvh_blankHRUdf(nHRUs = 3)
#'
#' # write to rvh file
#' rvn_rvh_write(file.path(tempdir(), "Blank.rvh"),
#'               SBtable = sbs_data,
#'               HRUtable = hru_data,
#'               description = "Example output - Blank Watershed Discretization File",
#'               author = "Raven Development Team")
#'
#' # Example: Read in an rvh, clean its contents and write back to new file
#' nith <- system.file("extdata","Nith.rvh",package = "RavenR")
#' rvh <- rvn_rvh_read(nith)
#'
#' # remove HRUs covering less than 5% of the total area
#' rvh$HRUtable <- rvn_rvh_cleanhrus(rvh$HRUtable, rvh$SBtable, area_tol = 0.05)
#'
#'
#' # write the Subbasin and HRU tables to new file using rvn_rvh_write:
#' rvn_rvh_write(filename=file.path(tempdir(), "Nith_cleaned_write.rvh"),
#'              SBtable = rvh$SBtable,
#'              HRUtable = rvh$HRUtable)
#'
#' # write to new file, while preserving all unedited information using rvn_rvh_overwrite:
#' rvn_rvh_overwrite(filename=file.path(tempdir(), "Nith_cleaned_overwrite.rvh"),
#'                   basefile=nith,
#'                   SBtable = rvh$SBtable,
#'                   HRUtable = rvh$HRUtable)
#'
#' @export rvn_rvh_write
rvn_rvh_write <- function(filename, SBtable=NULL, HRUtable=NULL,
                          description="Generated by RavenR rvn_rvh_write",
                          author=NULL) {

  if(missing(filename)) {
    stop("No filename provided!")
  }

  rvn_write_Raven_newfile(filename = filename,
              description,
              filetype = '.rvh',
              author = author)
  #-- SubBasins
  if (!is.null(SBtable)) {
    rvn_write_Raven_label('SubBasins',filename)
    rvn_write_Raven_table(attributes = c('Attributes','NAME','DOWNSTREAM_ID','PROFILE','REACH_LENGTH','GAUGED'),
                          units = c('Units','none','none','none','km','none'),
                          df = SBtable[c('SBID','Name','Downstream_ID','Profile','ReachLength','Gauged')],
                          filename = filename)
    rvn_write_Raven_label('EndSubBasins\n',filename)
  } else {
    warning(sprintf("No SBtable provided, SubBasins block not written to file %s",filename))
  }

  #-- HRUs
  if (!is.null(HRUtable)) {
    rvn_write_Raven_label("HRUs",filename)
    rvn_write_Raven_table(attributes = c('Attributes','AREA', 'ELEVATION','LATITUDE','LONGITUDE','BASIN_ID','LAND_USE_CLASS',
                                         'VEG_CLASS','SOIL_PROFILE','AQUIFER_PROFILE','TERRAIN_CLASS','SLOPE','ASPECT'),
                          units = c('Units','km2','masl','deg','deg','none','none','none','none','none','none','deg','degN'),
                          df = HRUtable[c('ID','Area','Elevation','Latitude','Longitude','SBID','LandUse',
                                          'Vegetation','SoilProfile','Aquifer','Terrain','Slope','Aspect')],
                          filename = filename)
    rvn_write_Raven_label("EndHRUs\n", filename)
  } else {
    warning(sprintf("No HRUtable provided, HRUs block not written to file %s",filename))
  }

  return(TRUE)
}
