/*
 * This file is part of RFrEAK. For licensing and copyright information
 * please see the file COPYING in the root directory of this
 * distribution or contact <robin.nunkesser@udo.edu>.
 * 
 * This file is a modification of the original file distributed with
 * FrEAK (http://sourceforge.net/projects/freak427/).
 * Last modification: 06/28/2007
 */

package freak.gui.scheduleeditor;

import freak.core.fitness.*;
import freak.core.modulesupport.*;
import freak.gui.*;
import java.util.*;
import javax.swing.table.*;

/**
 * @author  Oliver
 */
public class PanelFitnessFunction extends ScheduleEditorPanel {

	/**
	 * @uml.property  name="fitnessFunctions"
	 * @uml.associationEnd  multiplicity="(0 -1)"
	 */
	private Module[] fitnessFunctions;

	/** Creates new JPanel */
	public PanelFitnessFunction(ScheduleEditor scheduleEditor) {
		super(scheduleEditor);
		initComponents();
		// -- initialize the transformer table
		DefaultTableModel tm = new DefaultTableModel(null, new Object[] { "Transformer", "Configuration" });
		tableFitnessTransformer.setModel(tm);
	}

	public static String getDescription() {
		return "Select Fitness Function";
	}

	public String getHelpURL() {
		return "node4.html#SECTION00442000000000000000";
	}

	protected void initForStartedSchedule() {
		listFitnessFunctions.setEnabled(false);
		buConfigFitnessFunction.setVisible(false);
		buEditTransformers.setVisible(false);
	}

	public boolean mayAdvance() {
		return schedule.getFitnessFunction() != null;
	}

	public void enter() {
		super.enter();
		fitnessFunctions = moduleCollector.getFitnessFunctions(schedule.getPhenotypeSearchSpace());
		FitnessFunction f = schedule.getRealFitnessFunction();
		((ModuleList)listFitnessFunctions).setModules(fitnessFunctions, f);
		parseFitnessTransformers();						
	}

	private void parseFitnessFunction() {
		Module m = schedule.getRealFitnessFunction();

		if (m != null) {
			Util.displayText(taDescrFitness, m.getDescription());
			if (m instanceof Configurable)
				Util.displayText(tfOptionsFitnessFunction, ((Configurable)m).getConfiguration().getDescription());
			else
				Util.displayText(tfOptionsFitnessFunction, "<not configurable>");
			buConfigFitnessFunction.setEnabled(m instanceof Configurable);
		
			buEditTransformers.setEnabled(m != null);
		
			scheduleEditor.updateComponents();
		} else {
			// the fitness function has been removed from the schedule,
			// probably due to invalid fitness transformers
			listFitnessFunctions.clearSelection();
		}
	}

	private void parseFitnessTransformers() {
		panelTransformers.setVisible(schedule.getFitnessFunction() instanceof FitnessTransformer);
		DefaultTableModel tm = (DefaultTableModel)tableFitnessTransformer.getModel();
		tm.setRowCount(0);
		List fitnessTransformers = Util.getFitnessTransformersFrom(schedule);
		for (Iterator i = fitnessTransformers.iterator(); i.hasNext();) {
			FitnessTransformer ft = (FitnessTransformer)i.next();
			String confstr;
			if (ft instanceof Configurable) {
				Configuration conf = ((Configurable)ft).getConfiguration();
				confstr = conf.getDescription();
			} else
				confstr = "<not configurable>";
			tm.addRow(new Object[] { ft.getName(), confstr });
		}
	}

	private void setRealFitnessFunction(FitnessFunction function) {
		FitnessFunction f = schedule.getFitnessFunction();
		if (f instanceof FitnessTransformer) {
			// -- take care of fitness transformers
			FitnessTransformer ft = (FitnessTransformer)f;
			while (ft != null) {
				if (ft.getFitnessFunction() instanceof FitnessTransformer) {
					ft = (FitnessTransformer)ft.getFitnessFunction();
				} else {
					ft.setFitnessFunction(function);
					ft = null;
				}
			}
		} else {
			schedule.setFitnessFunction(function);
		}
	}

	private void chainFitnessTransformers(List fitnessTransformers) {
		// -- chain up fitness transformers
		FitnessFunction f = schedule.getRealFitnessFunction();
		for (Iterator i = fitnessTransformers.iterator(); i.hasNext();) {
			FitnessTransformer ft = (FitnessTransformer)i.next();
			ft.setFitnessFunction(f);
			f = ft;
		}
		schedule.setFitnessFunction(f);
		scheduleDependencyChecker.processNewFitnessFunction();
	}

	/** This method is called from within the constructor to
	 * initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is
	 * always regenerated by the Form Editor.
	 */
	private void initComponents() { //GEN-BEGIN:initComponents
		java.awt.GridBagConstraints gridBagConstraints;

		jPanel3 = new javax.swing.JPanel();
		jScrollPane7 = new javax.swing.JScrollPane();
		listFitnessFunctions = new ModuleList();
		jScrollPane8 = new javax.swing.JScrollPane();
		taDescrFitness = new javax.swing.JTextArea();
		jLabel20 = new javax.swing.JLabel();
		jLabel21 = new javax.swing.JLabel();
		jLabel6 = new javax.swing.JLabel();
		tfOptionsFitnessFunction = new javax.swing.JTextField();
		buConfigFitnessFunction = JButtonFactory.newButton();
		buEditTransformers = JButtonFactory.newButton();
		panelTransformers = new javax.swing.JPanel();
		jScrollPane11 = new javax.swing.JScrollPane();
		tableFitnessTransformer = new javax.swing.JTable();

		setLayout(new java.awt.GridBagLayout());

		setBorder(new javax.swing.border.TitledBorder("Select Fitness Function"));
		jPanel3.setLayout(new java.awt.GridBagLayout());

		jScrollPane7.setMinimumSize(new java.awt.Dimension(220, 0));
		jScrollPane7.setPreferredSize(new java.awt.Dimension(220, 0));
		listFitnessFunctions.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
		listFitnessFunctions.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
			public void valueChanged(javax.swing.event.ListSelectionEvent evt) {
				listFitnessFunctionsValueChanged(evt);
			}
		});

		jScrollPane7.setViewportView(listFitnessFunctions);

		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.gridy = 1;
		gridBagConstraints.gridheight = 3;
		gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
		gridBagConstraints.weighty = 0.1;
		gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 5);
		jPanel3.add(jScrollPane7, gridBagConstraints);

		taDescrFitness.setEditable(false);
		taDescrFitness.setLineWrap(true);
		taDescrFitness.setText("<no fitness function selected>");
		taDescrFitness.setWrapStyleWord(true);
		jScrollPane8.setViewportView(taDescrFitness);

		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 1;
		gridBagConstraints.gridy = 1;
		gridBagConstraints.gridwidth = 2;
		gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
		gridBagConstraints.weightx = 0.1;
		gridBagConstraints.weighty = 0.1;
		gridBagConstraints.insets = new java.awt.Insets(0, 0, 5, 0);
		jPanel3.add(jScrollPane8, gridBagConstraints);

		jLabel20.setText("Fitness Functions:");
		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.gridy = 0;
		gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
		jPanel3.add(jLabel20, gridBagConstraints);

		jLabel21.setText("Description:");
		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 1;
		gridBagConstraints.gridy = 0;
		gridBagConstraints.gridwidth = 2;
		gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
		jPanel3.add(jLabel21, gridBagConstraints);

		jLabel6.setText("Options:");
		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 1;
		gridBagConstraints.gridy = 2;
		gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
		gridBagConstraints.insets = new java.awt.Insets(0, 0, 5, 0);
		jPanel3.add(jLabel6, gridBagConstraints);

		tfOptionsFitnessFunction.setEditable(false);
		tfOptionsFitnessFunction.setText("<no fitness function selected>");
		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 2;
		gridBagConstraints.gridy = 2;
		gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
		gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
		gridBagConstraints.weightx = 0.1;
		gridBagConstraints.insets = new java.awt.Insets(0, 5, 5, 0);
		jPanel3.add(tfOptionsFitnessFunction, gridBagConstraints);

		buConfigFitnessFunction.setIcon(new javax.swing.ImageIcon(getClass().getResource("/toolbarButtonGraphics/general/Preferences16.gif")));
		buConfigFitnessFunction.setMnemonic('o');
		buConfigFitnessFunction.setText("Configure...");
		buConfigFitnessFunction.setEnabled(false);
		buConfigFitnessFunction.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent evt) {
				buConfigFitnessFunctionActionPerformed(evt);
			}
		});

		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 1;
		gridBagConstraints.gridy = 3;
		gridBagConstraints.gridwidth = 2;
		gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
		jPanel3.add(buConfigFitnessFunction, gridBagConstraints);

		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.gridy = 0;
		gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
		gridBagConstraints.weightx = 0.1;
		gridBagConstraints.weighty = 0.2;
		gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
		add(jPanel3, gridBagConstraints);

		buEditTransformers.setText("Edit Fitness Transformers");
		buEditTransformers.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent evt) {
				buEditTransformersActionPerformed(evt);
			}
		});

		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.gridy = 2;
		gridBagConstraints.insets = new java.awt.Insets(0, 0, 5, 0);
		add(buEditTransformers, gridBagConstraints);

		panelTransformers.setLayout(new java.awt.GridBagLayout());

		panelTransformers.setBorder(new javax.swing.border.TitledBorder("Fitness Transformers"));
		tableFitnessTransformer.setEnabled(false);
		jScrollPane11.setViewportView(tableFitnessTransformer);

		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
		gridBagConstraints.weightx = 0.1;
		gridBagConstraints.weighty = 0.1;
		panelTransformers.add(jScrollPane11, gridBagConstraints);

		gridBagConstraints = new java.awt.GridBagConstraints();
		gridBagConstraints.gridx = 0;
		gridBagConstraints.gridy = 1;
		gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
		gridBagConstraints.weighty = 0.1;
		gridBagConstraints.insets = new java.awt.Insets(0, 5, 5, 5);
		add(panelTransformers, gridBagConstraints);

	} //GEN-END:initComponents

	private void listFitnessFunctionsValueChanged(javax.swing.event.ListSelectionEvent evt) { //GEN-FIRST:event_listFitnessFunctionsValueChanged
		if (evt.getValueIsAdjusting())
			return;
		FitnessFunction oldObj = schedule.getRealFitnessFunction();
		FitnessFunction newObj = (FitnessFunction)listFitnessFunctions.getSelectedValue();
		if (newObj != oldObj) {
			if (oldObj != null) {
				eventController.replaceGlobalEventSource(oldObj, newObj);
				eventController.removeModule(oldObj);
			}
			setRealFitnessFunction(newObj);
			scheduleDependencyChecker.processNewFitnessFunction();
			newObj.createEvents();
		}
		parseFitnessFunction();
		parseFitnessTransformers();
	} //GEN-LAST:event_listFitnessFunctionsValueChanged

	private void buConfigFitnessFunctionActionPerformed(java.awt.event.ActionEvent evt) { //GEN-FIRST:event_buConfigFitnessFunctionActionPerformed
		String s = Util.configureModule(scheduleEditor, schedule.getRealFitnessFunction(), schedule);
		if (s != null) {
			Util.displayText(tfOptionsFitnessFunction, s);
			scheduleDependencyChecker.processNewModuleConfiguration(schedule.getFitnessFunction());
		}
	} //GEN-LAST:event_buConfigFitnessFunctionActionPerformed

	private void buEditTransformersActionPerformed(java.awt.event.ActionEvent evt) { //GEN-FIRST:event_buEditTransformersActionPerformed
		// -- show the Transformer Dialog
		FitnessTransformerDialog ftdlg = new FitnessTransformerDialog(scheduleEditor, schedule);
		ftdlg.setVisible(true);
		// -- fetch the new Transformer List
		List fts = ftdlg.getFitnessTransformers();
		// -- chain up the transformers and put them into the schedule
		chainFitnessTransformers(fts);
		parseFitnessTransformers();
		scheduleDependencyChecker.processNewFitnessFunction();
		scheduleEditor.updateComponents();
	} //GEN-LAST:event_buEditTransformersActionPerformed

	// Variables declaration - do not modify//GEN-BEGIN:variables
	private javax.swing.JButton buConfigFitnessFunction;
	private javax.swing.JButton buEditTransformers;
	private javax.swing.JLabel jLabel20;
	private javax.swing.JLabel jLabel21;
	private javax.swing.JLabel jLabel6;
	private javax.swing.JPanel jPanel3;
	private javax.swing.JScrollPane jScrollPane11;
	private javax.swing.JScrollPane jScrollPane7;
	private javax.swing.JScrollPane jScrollPane8;
	private javax.swing.JList listFitnessFunctions;
	private javax.swing.JPanel panelTransformers;
	private javax.swing.JTextArea taDescrFitness;
	private javax.swing.JTable tableFitnessTransformer;
	private javax.swing.JTextField tfOptionsFitnessFunction;
	// End of variables declaration//GEN-END:variables

}
