\name{nlFit}
\alias{nlFit}
\alias{print.nlFit}
\alias{plot.nlFit}
\alias{coef.nlFit}
\alias{vcov.nlFit}

\title{Fit the Normal Laplace Distribution to Data}
\description{
  Fits a normal Laplace distribution to data.
  Displays the histogram, log-histogram (both with fitted densities),
  Q-Q plot and P-P plot for the fit which has the maximum likelihood.
}

\usage{
  nlFit(x, freq = NULL, breaks = "FD", paramStart = NULL,
        startMethod = "Nelder-Mead",
        startValues = c("MoM", "US"),
        method = c("Nelder-Mead", "BFGS", "L-BFGS-B",
                   "nlm", "nlminb"),
        hessian = FALSE,
        plots = FALSE, printOut = FALSE,
        controlBFGS = list(maxit = 200),
        controlLBFGSB = list(maxit = 200),
        controlNLMINB = list(),
        controlNM = list(maxit = 1000),
        maxitNLM = 1500, ...)
  \method{print}{nlFit}(x, digits = max(3, getOption("digits") - 3), ...)
  \method{plot}{nlFit}(x, which = 1:4,
    plotTitles = paste(c("Histogram of ","Log-Histogram of ",
                         "Q-Q Plot of ","P-P Plot of "), x$obsName,
                       sep = ""),
    ask = prod(par("mfcol")) < length(which) & dev.interactive(), ...)
  \method{coef}{nlFit}(object, ...)
  \method{vcov}{nlFit}(object, ...)
}

\arguments{
  \item{x}{Data vector for \code{nlFit}.}
  \item{freq}{A vector of weights with length equal to \code{length(x)}.}
  \item{breaks}{Breaks for plotted histogram, defaults to those generated by
    \code{hist(x, right = FALSE, plot = FALSE)}.}
  \item{paramStart}{A user specified starting parameter vector taking the form
    \code{c(mu, sigma, alpha, beta)}.}
  \item{startMethod}{Method used by \code{nlFitStart} in calls to
    \code{\link{optim}}.}
  \item{startValues}{Code giving the method of determining starting values
    for finding the maximum likelihood estimate of the parameters.}
  \item{method}{Different optimisation methods to consider.
    See \bold{Details}.}
  \item{hessian}{Logical. If \code{TRUE} the value of the Hessian
    is returned.}
  \item{plots}{Logical. If \code{FALSE} the printing of the histogram,
    log-histogram, Q-Q plot and P-P plot are suppressed.}
  \item{printOut}{Logical. If \code{FALSE} the printing of the results of
    fitting will be suppressed.}
  \item{controlBFGS}{A list of control parameters for \code{optim} when
    using the \code{"BFGS"} method of optimisation.}
  \item{controlLBFGSB}{A list of control parameters for \code{optim} when
    using the \code{"L-BFGS-B"} method of optimisation.}
  \item{controlNLMINB}{A list of control parameters for \code{optim} when
    using the \code{"nlminb"} method of optimisation.}
  \item{controlNM}{A list of control parameters for \code{optim} when
    using the \code{"Nelder-Mead"} method of optimisation.}
  \item{maxitNLM}{A positive integer specifying the maximum number of
    iterations that are to be undertaken when using the \code{"nlm"} method
    of optimisation.}
  \item{object}{Object of class \code{"nlFit"} for \code{print.nlFit},
    \code{plot.nlFit}, \code{coef.nlFit} and \code{vcov.nlFit}.}
  \item{digits}{Desired number of digits to be shown when the object
    is printed.}
  \item{which}{If a subset of the plots if required, specify a subset of
    the numbers \code{1:4}.}
  \item{plotTitles}{Titles to appear as the main title above the plots.}
  \item{ask}{Logical. If \code{TRUE}, the user is \emph{ask}ed before
    each plot, see \code{\link{par}(ask = .)}.}
  \item{\dots}{Passes arguments to \code{par}, \code{hist}, \code{logHist},
    \code{qqnl} and \code{ppnl}. For the \code{print}, \code{coef} and
    \code{vcov} methods this parameter has no effect.}
}

\details{
    \code{startMethod} must be \code{"Nelder-Mead"}.

    \code{startValues} can only be \code{"MoM"} when using the Method of
    Moments for estimation, or \code{"US"} for user-supplied parameters. For
    details regarding the use of \code{paramStart}, \code{startMethod} and
    \code{startValues}, see \code{\link{nlFitStart}}.

    Three optimisation methods are available for use:
    \itemize{
      \item{\code{"BFGS"}}{Uses the quasi-Newton method \code{"BFGS"} as
        documented in \code{\link{optim}}.}
      \item{\code{"L-BFGS-B"}}{Uses the constrained method \code{"L-BFGS-B"} as
        documented in \code{\link{optim}}.}
      \item{\code{"Nelder-Mead"}}{Uses an implementation of the Nelder and
        Mead method as documented in \code{\link{optim}}.}
      \item{\code{"nlm"}}{Uses the \code{\link{nlm}} function in R.}
      \item{\code{"nlminb"}}{Uses the \code{\link{nlminb}} function in R,
        with constrained parameters.}
    }

    For details on how to pass control information for optimisation using
    \code{\link{optim}} and \code{\link{nlm}}, see \code{\link{optim}} and
    \code{\link{nlm}}.

    When \code{method = "nlm"} or \code{method = "nlm"} is used, warnings
    may be produced. However, these do not appear to be problematic.
}

\value{
  A list with components:
    \item{param}{A vector giving the maximum likelihood estimate of
      parameters, as
      \code{c(mu, sigma, alpha, beta)}.}
    \item{maxLik}{The value of maximised log-likelihood.}
    \item{hessian}{If \code{hessian} was set to \code{TRUE}, the value of
      the Hessian. Not present otherwise.}
    \item{method}{Optimisation method used.}
    \item{conv}{Convergence code. See the relevant documentation (either
      \code{\link{optim}} or \code{\link{nlm}}) for details on convergence.}
    \item{iter}{Number of iterations made by the optimisation routine.}
    \item{obs}{The data used to fit the normal Laplace distribution.}
    \item{obsName}{A character vector with the actual \code{x} argument name.}
    \item{paramStart}{Starting value of parameters returned by call to
      \code{\link{nlFitStart}}.}
    \item{svName}{Descriptive name for the method of finding start values.}
    \item{startValues}{Acronym for the method of finding start values.}
    \item{breaks}{The cell boundaries found by a call to
      \code{\link{hist}}.}
    \item{midpoints}{The cell midpoints found by a call to
      \code{\link{hist}}.}
    \item{empDens}{The estimated density found by a call to
      \code{\link{hist}}.}
}

\author{David Scott \email{d.scott@auckland.ac.nz},
  Simon Potter}

\seealso{
  \code{\link{optim}}, \code{\link{nlm}}, \code{\link{par}},
  \code{\link{hist}}, \code{\link{logHist}}, \code{\link{qqnl}},
  \code{\link{ppnl}}, \code{\link{dnl}} and
  \code{\link{nlFitStart}}.
}

\examples{
param <- c(0, 2, 1, 1)
dataVector <- rnl(1000, param = param)

## Let's see how well nlFit works
nlFit(dataVector)
nlFit(dataVector, plots = TRUE)
fit <- nlFit(dataVector)
par(mfrow = c(1, 2))
plot(fit, which = c(1, 3))  # See only histogram and Q-Q plot
}

\keyword{distribution}
