### Class Definitions ----

#' Electrocardiogram data class for 12-lead ECG studies
#'
#' @description This class serves as a specialized extension of the `egm` class
#'   specifically for standard 12-lead electrocardiogram data. It inherits all
#'   functionality from `egm` while providing additional validation and methods
#'   specific to 12-lead ECG analysis.
#'
#' @details The `ecg` object contains the same three components as `egm`:
#'   * signal data in multiple channels (specifically 12 standard ECG leads)
#'   * header information
#'   * annotation labels at specified time points
#'
#'   The primary difference is that this class enforces validation to ensure
#'   the data represents a standard 12-lead ECG with appropriate lead names.
#'
#' @returns An object of class `ecg` (which inherits from `egm`) containing
#'   signal, header, and annotation components.
#'
#' @param x An `ecg` object to be used with support functions
#'
#' @param signal A `signal_table` object generated by the [signal_table()]
#'   function containing standard ECG leads
#'
#' @param header A `header_table` object generated by the [header_table()]
#'   function
#'
#' @param annotation A `annotation_table` object generated by the
#'   [annotation_table()] function
#'
#' @param ... Additional arguments to be passed to the function
#'
#' @name ecg
#' @export
ecg <- function(signal = signal_table(),
								header = header_table(),
								annotation = annotation_table(),
								...) {

	# Create a new ecg object
	new_ecg(signal,
					header = header,
					annotation = annotation)
}

#' @keywords internal
new_ecg <- function(signal = signal_table(),
										header = header_table(),
										annotation = annotation_table(),
										...) {

	# Validate that this is appropriate 12-lead ECG data
	validate_ecg_data(signal, header)

	# Create an egm object first, then add the ecg class
	structure(
		list(
			signal = signal,
			header = header,
			annotation = annotation
		),
		class = c('ecg', 'egm', 'list')
	)
}

#' Validate that signal data represents a standard 12-lead ECG
#' @param signal A signal_table object
#' @param header A header_table object
#' @return TRUE if valid, throws error otherwise
#' @keywords internal
validate_ecg_data <- function(signal, header) {
	if (nrow(signal) == 0) return(TRUE)  # Empty signal is valid initially

	# Get lead names (exclude sample column)
	lead_names <- names(signal)[-1]

	# Standard ECG lead names (case-insensitive matching)
	std_leads <- c("I", "II", "III", "AVR", "AVL", "AVF", "V1", "V2", "V3", "V4", "V5", "V6")

	# Check if we have 12 leads
	if (length(lead_names) != 12) {
		warning("ECG should contain 12 leads, found ", length(lead_names),
						". Converting to ecg class anyway.")
		return(TRUE)
	}

	# Check lead names (allowing for some variation in naming)
	normalized_leads <- toupper(gsub("[_\\s-]", "", lead_names))
	std_leads_norm <- toupper(gsub("[_\\s-]", "", std_leads))

	if (!all(normalized_leads %in% std_leads_norm)) {
		non_std <- lead_names[!normalized_leads %in% std_leads_norm]
		warning("Non-standard ECG lead names detected: ",
						paste(non_std, collapse=", "),
						". Converting to ecg class anyway.")
	}

	TRUE
}

#' @export
format.ecg <- function(x, ...) {
	# Call the parent format method
	NextMethod()

	# Add ECG-specific information
	cat('Type: Standard 12-lead ECG\n')

	# Return invisibly for chaining
	invisible(x)
}

#' @export
print.ecg <- function(x, ...) {
	format(x)
}

#' @export
#' @rdname ecg
is_ecg <- function(x) {
	inherits(x, "ecg")
}

#' Convert an egm object to an ecg object
#'
#' @description Converts a general `egm` object to a specialized `ecg` object
#'   for 12-lead ECG analysis.
#'
#' @param x An object of class `egm`
#' @param ... Additional arguments
#' @return An object of class `ecg`
#' @export
as_ecg <- function(x, ...) {
	if (!is_egm(x)) {
		stop("Input must be of class 'egm'")
	}

	ecg(signal = x$signal,
			header = x$header,
			annotation = x$annotation)
}
